<?php

/**
 * The Admin Toold Controller Class
 *
 * @author Yaidier Perez
 * */

namespace Jptgb\Controllers;

use Jptgb\Resources\Utils;
use Jptgb\Controllers\CacheController;
use Jptgb\Controllers\CriticalCssController;
use Jptgb\Includes\ColumnPanel;


class AdminToolsController extends BaseController {
    public static function init() {
        add_action( 'admin_bar_menu',                   [self::class, 'clean_cache_adminbar'], 100 );
        add_action( 'admin_post_clear_current_page',    [self::class, 'handle_clear_current_page_cache'] );
        add_action( 'admin_post_clear_current_page_and_critical',  [self::class, 'handle_clear_current_page_critical_css'] );
        add_action( 'admin_post_clear_all_pages',       [self::class, 'handle_clear_all_pages_cache'] );

        if(self::$settings['jptgb_setting_show_columntool_on_posts']){
            add_action('admin_init', [self::class, 'add_optimizations_columns']);
        }
    }

    public static function add_optimizations_columns() {
        /**
         * Retrieve all public post types including built-in and custom ones.
         */
        $post_types = get_post_types(['public' => true], 'names');
    
        foreach ($post_types as $post_type) {
            /**
             * Hook into the manage_{$post_type}_posts_columns filter for each post type to add the column.
             */
            add_filter("manage_{$post_type}_posts_columns", [self::class, 'add_column_header__callback']);

            /**
             * Hook into the manage_{$post_type}_posts_custom_column action to display the column content.
             */
            add_action("manage_{$post_type}_posts_custom_column", [self::class, 'add_column_content__callback'], 10, 2);
        }
    }

    /**
     * Adds a new column header to all post types.
     *
     * @param array $columns The existing columns.
     * @return array Modified columns with the new custom column.
     */
    public static function add_column_header__callback($columns) {
        $columns['jptgb-optimizations_column'] = __('WebSpeed', 'webspeed');
        return $columns;
    }

    /**
     * Displays content for the custom column for all post types.
     *
     * @param string $column_name The name of the current column.
     * @param int $post_id The ID of the current post.
     */
    public static function add_column_content__callback($column_name, $post_id) {
        if ('jptgb-optimizations_column' === $column_name) {
            ColumnPanel::render( $post_id );
        }
    }

    private static function get_cache_status( $post_id, $cache_data ) {
        /**
         * Check if the if we reached the max number of pages already.
         */
        $is_url_limit = BaseController::get_setting( 'jptgb_setting_is_url_limit_reached' );

        if( 32009 ){
            $stop = 1;
        }

        if( $is_url_limit ){
            /**
             * Get the requested url.
             */
            $requested_url = Utils::get_permalink( $post_id );
    
            /**
             * Get the allowed urls.
             */
            $allowed_urls = BaseController::get_setting( 'jptgb_setting_allowed_urls' );

            /**
             * Extract all URLs into a flat array.
             */
            // $urls = array_column( $allowed_urls, 'url' );

            /**
             * Search if current url is allowed.
             */
            if( in_array( $requested_url, $allowed_urls ) ) {
                return $cache_data['status'];
            } else {
                return 'blocked';
            }
        } else {
            return $cache_data['status'];
        }
    }

    public static function clean_cache_adminbar( \WP_Admin_Bar $admin_bar ) {
        /**
         * Return if current user is not administrator.
         */
        if ( !current_user_can( 'manage_options' ) ) {
            return;
        }

        /**
         * Get requested url.
         */
        $requested_url  = Utils::get_current_url();
        $encoded_url    = base64_encode( $requested_url );

        /**
         * Get the curent post id.
         */
        $post_id = get_the_ID();
    
        /**
         * Add a parent item.
         */
        $admin_bar->add_menu( array(
            'id'    => 'jptgb_adminbar',
            'title' => 'WebSpeed',
            'href'  => '#',
        ));
    
        /**
         * Add a submenu item for clearing current page cache.
         */
        $admin_bar->add_menu( array(
            'parent' => 'jptgb_adminbar',
            'id'     => 'jptgb_clear_current_page_cache',
            'title'  => 'Clear Current Page Cache',
            'href'   => wp_nonce_url( admin_url( 'admin-post.php?action=clear_current_page&requested_url='. $encoded_url .'&id=' . $post_id ), 'clear_current_page_nonce' ),
        ));
    
        /**
         * Add a submenu item for clearing all page cache.
         */
        $admin_bar->add_menu( array(
            'parent' => 'jptgb_adminbar',
            'id'     => 'jptgb_clear_current_page_critical_css',
            'title'  => 'Clear Current Page Cache and Critical Css',
            'href'   => wp_nonce_url( admin_url( 'admin-post.php?action=clear_current_page_and_critical&requested_url='. $encoded_url .'&id=' . $post_id ), 'clear_current_page_and_critical_nonce' ),
        ));

        /**
         * Add a submenu item for clearing all page cache.
         */
        // $admin_bar->add_menu( array(
        //     'parent' => 'jptgb_adminbar',
        //     'id'     => 'jptgb_clear_all_pages_cache',
        //     'title'  => 'Clear All Pages Cache',
        //     'href'   => wp_nonce_url( admin_url( 'admin-post.php?action=clear_all_pages&requested_url='. $encoded_url .'&id=' . $post_id ), 'clear_all_pages_nonce' ),
        // ));
    }

    /**
     * Handle clearing the current page cache.
     */
    public static function handle_clear_current_page_cache() {
        check_admin_referer( 'clear_current_page_nonce' );

        /**
         * Get the post id.
         */
        $post_id = isset( $_GET['id'] ) ? (int) $_GET['id'] : 0;

        /**
         * Flush the url cache.
         */
        CacheController::regenerate_cache_for_post_id( $post_id );

        /**
         * Prepare front-end message.
         */
        set_transient('jptgb_cache_cleared_message', 'Current Page Cache Cleared', 10);

        wp_redirect(wp_get_referer());
        exit;
    }

    /**
     * Handle clearing all pages cache.
     */
    public static function handle_clear_current_page_critical_css() {
        check_admin_referer( 'clear_current_page_and_critical_nonce' );

        /**
         * Get the post id.
         */
        $post_id = $_GET['id'];

        /**
         * Get the requested url.
         */
        $requested_url  = Utils::get_permalink( $post_id );
        $url_hash       = Utils::create_url_safe_hash( $requested_url );
        $cc_data        = CriticalCssController::get_critical_data( $url_hash );

        /**
         * Command to regenerate the cache of css.
         */
        $cc_data['flush_desktop']   = true;
        $cc_data['flush_mobile']    = true;

        /**
         * Update the critical css data.
         */
        CriticalCssController::update_critical_data( $url_hash, $cc_data );

        /**
         * Flush the url cache.
         */
        CacheController::regenerate_cache_for_post_id( $post_id );

        set_transient( 'jptgb_cache_cleared_message', 'Current Page Critical Css Cleared', 10 );

        wp_redirect( wp_get_referer() );
        exit;
    }

    /**
     * Handle clearing the current page cache.
     */
    public static function handle_clear_all_pages_cache() {
        check_admin_referer( 'clear_all_pages_nonce' );

        /**
         * Flush all cache files
         */
        CacheController::flush_all_cache_files();

        /**
         * Prepare front-end message.
         */
        set_transient('jptgb_cache_cleared_message', 'All Pages Cache Flushed', 10);

        wp_redirect(wp_get_referer());
        exit;
    }
}