<?php

/**
 * The Ajax Calls Controller Class
 *
 * @author Yaidier Perez
 * */

namespace Jptgb\Controllers;

use Jptgb\Controllers\CacheController;
use Jptgb\Controllers\CriticalCssController;
use Jptgb\Controllers\ImagesController;
use Jptgb\Controllers\LazyDomLoadController;
use Jptgb\Resources\Utils;
use Jptgb\Controllers\MainController;
use WP_Error;

class AjaxController extends BaseController {
    public static function init(){
        $ajax_hanlders = [
            'delete_user',
            'request_new_api',
            'flush_all_cache',
            'flush_all_images',
            'flush_critical_css',
            'generate_cache',
            'sync_with_api_ajax',
            'delete_plugins_data',
            'test_api_endpoint',
            'trigger_cron_job',
            'get_ldl_data',
            'tools_column_generate_cache',
            'tools_column_rebuild_cache',
            'tools_column_delete_cache',
            'get_tasks_statuses',
        ];

        

        foreach ( $ajax_hanlders as $ajax_hanlder ) {
            add_action( 'wp_ajax_nopriv_' . $ajax_hanlder, [self::class, $ajax_hanlder] );
            add_action( 'wp_ajax_' . $ajax_hanlder, [self::class, $ajax_hanlder] );
        }
    }

    public static function sync_with_api_ajax() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );

        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_sync_api_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        $data = [
            'callback' => 'refreshPageWithMessage',
        ];

        $response = BaseController::sync_with_api();

        if( is_wp_error( $response ) ){
            self::send_json( $response->get_error_message() ?: 'There was an error during the request.', 'error', $data );
            return;
        }

        /**
         * Respond the backend.
         */
        self::send_json( 'Api Sync Successfully', 'notice-success is-dismissible', $data );
    }

    public static function sync_api_data_to_plugin( $response_code, $response ) {       
        /** 
         * Unauthorized responses. 
         */
        self::handle_unauthorized_reponse( $response_code );

        /**
         * Sync wp data with the data from the api.
         */
        $body = wp_remote_retrieve_body( $response );

        if( !$body ){
            return;
        }

        /**
         * Decode the response.
         */
        $body = json_decode( $body, true );
        $data = $body['subscription_data'] ?? [];

        if( empty( $data ) ){
            return;
        }

        /**
         * Update plan details
         */
        if( !empty( $data['plan_name'] ) ){
            BaseController::update_setting( 'jptgb_setting_api_plan', $data['plan_name'] );
        }

        if( !empty( $data['expires_on'] ) ){
            BaseController::update_setting( 'jptgb_setting_expiration_date', $data['expires_on'] );
        }

        if( isset( $data['is_active'] ) ){
            $is_active = $data['is_active'] == 1 ? true : false;
            BaseController::update_setting( 'jptgb_setting_subscription_status', $is_active );
        }

        if( isset( $data['number_of_urls'] ) ){
            BaseController::update_setting( 'number_of_urls', $data['number_of_urls'] );
        }

        if( isset( $data['registered_urls'] ) ){
            BaseController::update_setting( 'registered_urls', $data['registered_urls'] );
        }
    }

    public static function handle_unauthorized_reponse( $response_code ) {
        /**
         * If Api key is not correct, then change
         * the api key status icon.
         */
        if( 401 === $response_code  ){
            BaseController::update_setting( 'is_api_key_valid', 'error' );
        } else {
            BaseController::update_setting( 'is_api_key_valid', 'success' );
        }

        if( 402 === $response_code ){
            BaseController::update_setting( 'jptgb_setting_api_plan_status', 'expired' );
        } else {
            BaseController::update_setting( 'jptgb_setting_api_plan_status', '' );
        }
    }

    public static function send_json_updated_tasks( $tasks ) {
        $updated_tasks = [];
        foreach( $tasks as $object_id => $task_data ){
            $cache_data = self::get_cache_status( $object_id );

            if( !$cache_data ){
                $updated_tasks[ $object_id ] = $task_data;   
                
                continue;
            }

            if( ( $cache_data['status'] ?? false ) === 'working-on' && !empty( $cache_data['percentage'] ) ) {
                $percentage = (int) $cache_data['percentage'];

                if( $percentage < 20 ){
                    $random_number = wp_rand(1, 3);
                    $cache_data['percentage'] = $percentage + $random_number;
                    self::update_cache_status( $object_id, $cache_data['status'], $cache_data['message'], $percentage + $random_number );
                }
            }

            $updated_tasks[ $object_id ] = $cache_data;
        }

        $tasks_overview     = get_option( 'jptgb_tasks_overview', [] );
        $total_tasks        = count( $tasks_overview['total_tasks']    ?? [] );
        $ready_tasks        = count( $tasks_overview['ready_tasks']    ?? [] );
        $percentage         = $tasks_overview['percentage'] ?? 0;

        

        /*
        * Prepare the response body.
        */
        $data = [
            'callback'          => 'checkStatusRecursevely',
            'tasks'             => $updated_tasks,
            'totalTasks'        => $total_tasks,
            'readyTasks'        => $ready_tasks,
            'percentage'        => $percentage,
        ];

        

        /**
         * Respond the backend.
         */
        self::send_json( '', 'success', $data );
    }

    public static function get_tasks_statuses() {
         /**
         * Get the nonce.
         */
        $tasks = wp_unslash( $_POST['tasks'] ?? [] );
        $tasks = json_decode( $tasks, true );

        self::send_json_updated_tasks( $tasks );
    }

    public static function tools_column_generate_cache() {
         /**
         * Get the nonce.
         */
        $nonce          = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        $object_id      = absint( wp_unslash( $_POST['object_id'] ?? 0 ) );
        $object_type    = sanitize_key( wp_unslash( $_POST['object_type'] ?? '' ) );

        if( !self::get_setting( 'cache_activate' ) ){
            self::send_json_message( 'Cache is turned off' );
        }

        /**
         * Return if there is no either object id or object type.
         */
        if( !$object_id || !$object_type ){
            self::send_json( 'Missing either object id or object type', 'error' );
        }

        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_tools_column_generate_cache_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        $response = self::tools_generate_cache( $object_id, $object_type, 10 );

        if( is_wp_error( $response ) ){
            // Get the first error code
            $error_code = $response->get_error_code();

            // Get the error message for the first code
            $error_message = $response->get_error_message();

            // Get the data associated with the error code
            $error_data = $response->get_error_data( $error_code );

            switch( $error_code ) {
                case 'set_blocked_status':
                    self::send_json( $error_message, 'error', $error_data, true );
                    break;
                
                default:
                    # code...
                    break;
            }
        }
    }

    

    public static function max_urls_requests_reached( $response_body, $object_id ) {
        /**
         * Get the error message.
         */
        $error_message = $response_body['message'] ?? '';

        /**
         * Update cache status of current request.
         */
        self::update_cache_status( $object_id, 'no-cache', '', 0 );

        /**
         * Let our base controller know that we reached the max number of 
         * allowd urls.
         */
        BaseController::update_setting( 'jptgb_setting_is_url_limit_reached', true );

        /*
        * Prepare the response body.
        */
        $data = [
            'callback'      => 'setBlockedStatus',
            'percentage'    => 0,
            'message'       => 'This post is not serving any cache file',
            'status'        => 'blocked',
        ];

        /**
         * Respond the backend.
         */
        self::send_json( $error_message, 'error', $data );
    }

    public static function tools_column_delete_cache() {
        /**
        * Get the nonce.
        */
        $nonce          = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        $object_id      = absint( wp_unslash( $_POST['object_id'] ?? 0 ) );
        $object_type    = sanitize_key( wp_unslash( $_POST['object_type'] ?? '' ) );
        /**
        * Return if there is no either object id or object type.
        */
        if( !$object_id || !$object_type ){
            self::send_json( 'Missing either object id or object type', 'error' );
        }

        /**
        * Check the wp nonce.
        */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_tools_column_generate_cache_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
        * Get the requested url.
        */
        $requested_url = Utils::get_permalink( $object_id );
        $url_hash = Utils::create_url_safe_hash( $requested_url );

        /**
        * Return if we can't get the requested url.
        */
        if( !$requested_url ){
            self::send_json( "We can't get the url for the object id: {$object_id}", 'error' );
        }

        /**
         * Delete the cache files.
         */
        CacheController::delete_cache_by_post_id( $object_id );

        /**
         * Signal the API that we have deleted the cache file.
         */
        $response = CacheController::request_api_to_delete_cache( $requested_url, $url_hash );

        if( false !== $response ) {
            $response_code = wp_remote_retrieve_response_code( $response );
            AjaxController::sync_api_data_to_plugin( $response_code, $response );
        }

        /**
         * Flush hosting's cache.
         */
        MainController::flush_hostings_cache( $object_id );

        /**
        * Update the cache status.
        */
        self::update_cache_status( $object_id, 'no-cache', 'This post is not serving a cache file', 0 );

        /*
        * Prepare the response body.
        */
        $data = [
            'callback'      => 'setNoCacheStatus',
            'percentage'    => 0,
            'message'       => 'This post is not serving any cache file',
            'status'        => 'no-cache',
        ];

        /**
        * Respond the backend.
        */
        self::send_json( 'Cache deleted successfully', 'success', $data );
   }

    /**
     * Check if a post is public by its post ID.
     *
     * @param int $post_id The ID of the post to check.
     * @return bool True if the post is public, false otherwise.
     */
    public static function is_post_public( int $post_id ) {
        // Get the post by ID
        $post = get_post( $post_id );

        // Check if the post exists and its status is 'publish'

        if( !$post ){
            return 'not public';
        }

        if( $post->post_status !== 'publish' ) {
            $post_status = $post->post_status;

            return $post_status;
        }

        return true;
    }

    public static function send_json_message( $message, $status = 'error' ) {        
        /*
        * Prepare the response body.
        */
        $data = [
            'callback'      => 'setBlockedStatus',
            'percentage'    => 0,
            'message'       => $message,
            'status'        => $status,
        ];

        /**
         * Respond the backend.
         */
        self::send_json( $message, $status, $data, true );
    }

    public static function tools_generate_cache_for_consumer( $object_id, $object_type, $priority = 0 ) {
        self::update_cache_status( $object_id, 'working-on', 'Cache flushed', 5 ); 

        /**
         * Check if post is public.
         */
        $is_post_public = self::is_post_public( $object_id );

        if( true !== $is_post_public ){
            $message = 'This post status is ' . $is_post_public;
            $status = 'error';

            self::update_cache_status( $object_id, 'error', 'This post is not public' );
            
            /*
            * Prepare the response body.
            */
            $data = [
                'callback'      => 'setBlockedStatus',
                'percentage'    => 0,
                'message'       => $message,
                'status'        => $status,
            ];

            /**
             * Respond the backend.
             */
            return new WP_Error( 'set_blocked_status', $message, $data );
        }

        /**
         * Get the requested url.
         */
        $requested_url = Utils::get_permalink( $object_id );
        BaseController::$logs->register( 'Generate cache called. Requested url: ' . $requested_url, 'info' );

        /**
         * Return if we can't get the requested url.
         */
        if( !$requested_url ){
            return new WP_Error( 'no_url', "We can't get the url for the object id: {$object_id}" );
        }

        /**
         * Delete the cache files but keep the consumer.
         */
        CacheController::delete_cache_by_post_id( $object_id, false );
        
        /**
         * Request start the cache build.
         */
        $response = CacheController::start_the_build_cache_request( $object_id, $requested_url, $priority );

        /**
         * Flush hosting's cache.
         */
        MainController::flush_hostings_cache( $object_id );

        /**
         * Return if there has been an error during the api request.
         */
        if( false === $response ){
            Utils::add_a_task_ready_to_tasks_overview( $object_id );

            return new WP_Error( 'general_error', "Something went wrong" );
        }

        /**
         * Return if there has been an error during the api request.
         */
        if( is_wp_error( $response ) ){
            $message    = $response->get_error_message() ?: 'Something went wrong.';
            $status     = $response->get_error_data()['status'] ?: 'error';

            self::update_cache_status( $object_id, 'error', $message, 0 );


            Utils::add_a_task_ready_to_tasks_overview( $object_id );

            /*
            * Prepare the response body.
            */
            $data = [
                'callback'      => 'setBlockedStatus',
                'percentage'    => 0,
                'message'       => $message,
                'status'        => $status,
            ];

            /**
             * Respond the backend.
             */
            return new WP_Error( 'set_blocked_status', $message, $data );
        }

        /**
         * Get the response code.
         */
        $response_code = wp_remote_retrieve_response_code( $response );

        /**
         * Sync api data to our plugin.
         */
        self::sync_api_data_to_plugin( $response_code, $response );

        /**
         * Handle exceeding quote.
         */
        self::handle_exceeding_quote( $response_code, $response );

        /**
         * 
         */
        if( 200 !== $response_code ) {
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );

            /**
             * Get the response message.
             */
            $message = $data['message'] ?? $response['response']['message'] ?? 'Something went wrong';

            /**
             * Update the object cache status.
             */
            self::update_cache_status( $object_id, 'error', $message, 0 );

            /*
            * Prepare the response body.
            */
            $data = [
                'callback'      => 'setBlockedStatus',
                'percentage'    => 0,
                'message'       => $message,
                'status'        => 'error',
            ];

            Utils::add_a_task_ready_to_tasks_overview( $object_id );

            /**
             * Respond the backend.
             */
            return new WP_Error( 'set_blocked_status', $message, $data );
        }

        /**
         * Update the cache status.
         */
        self::update_cache_status( $object_id, 'working-on', 'Waiting in Queue', 10 );

        return true;
    }

    public static function tools_generate_cache( $object_id, $object_type, $priority = 0 ) {
        self::update_cache_status( $object_id, 'working-on', 'Cache flushed', 5 ); 

        /**
         * Check if post is public.
         */
        $is_post_public = self::is_post_public( $object_id );

        if( true !== $is_post_public ){
            $message = 'This post status is ' . $is_post_public;
            $status = 'error';

            self::update_cache_status( $object_id, 'error', 'This post is not public' );
            
            /*
            * Prepare the response body.
            */
            $data = [
                'callback'      => 'setBlockedStatus',
                'percentage'    => 0,
                'message'       => $message,
                'status'        => $status,
            ];

            /**
             * Respond the backend.
             */
            return new WP_Error( 'set_blocked_status', $message, $data );
        }

        /**
         * Get the requested url.
         */
        $requested_url = Utils::get_permalink( $object_id );
        BaseController::$logs->register( 'Generate cache called. Requested url: ' . $requested_url, 'info' );

        /**
         * Return if we can't get the requested url.
         */
        if( !$requested_url ){
            return new WP_Error( 'no_url', "We can't get the url for the object id: {$object_id}" );
        }

        /**
         * Delete the cache files.
         */
        CacheController::delete_cache_by_post_id( $object_id );
        
        /**
         * Request start the cache build.
         */
        $response = CacheController::start_the_build_cache_request( $object_id, $requested_url, $priority );

        /**
         * Flush hosting's cache.
         */
        MainController::flush_hostings_cache( $object_id );

        /**
         * Return if there has been an error during the api request.
         */
        if( false === $response ){
            Utils::add_a_task_ready_to_tasks_overview( $object_id );

            return new WP_Error( 'general_error', "Something went wrong" );
        }

        /**
         * Return if there has been an error during the api request.
         */
        if( is_wp_error( $response ) ){
            $message    = $response->get_error_message() ?: 'Something went wrong.';
            $status     = $response->get_error_data()['status'] ?: 'error';

            self::update_cache_status( $object_id, 'error', $message, 0 );


            Utils::add_a_task_ready_to_tasks_overview( $object_id );

            /*
            * Prepare the response body.
            */
            $data = [
                'callback'      => 'setBlockedStatus',
                'percentage'    => 0,
                'message'       => $message,
                'status'        => $status,
            ];

            /**
             * Respond the backend.
             */
            return new WP_Error( 'set_blocked_status', $message, $data );
        }

        /**
         * Get the response code.
         */
        $response_code = wp_remote_retrieve_response_code( $response );

        /**
         * Sync api data to our plugin.
         */
        self::sync_api_data_to_plugin( $response_code, $response );

        /**
         * Handle exceeding quote.
         */
        self::handle_exceeding_quote( $response_code, $response );

        /**
         * 
         */
        if( 200 !== $response_code ) {
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );

            /**
             * Get the response message.
             */
            $message = $data['message'] ?? $response['response']['message'] ?? 'Something went wrong';

            /**
             * Update the object cache status.
             */
            self::update_cache_status( $object_id, 'error', $message, 0 );

            /*
            * Prepare the response body.
            */
            $data = [
                'callback'      => 'setBlockedStatus',
                'percentage'    => 0,
                'message'       => $message,
                'status'        => 'error',
            ];

            Utils::add_a_task_ready_to_tasks_overview( $object_id );

            /**
             * Respond the backend.
             */
            return new WP_Error( 'set_blocked_status', $message, $data );
        }

        /**
         * Update the cache status.
         */
        self::update_cache_status( $object_id, 'working-on', 'Waiting in Queue', 10 );

        return true;
    }

    public static function handle_exceeding_quote( $response_code, $response ) {
        if( 403 !== $response_code ) {
            return;
        }

        /**
         * Stop all consumers if any.
         */
        Utils::empty_consumer_table();
    }
    
    public static function tools_column_rebuild_cache() {
        self::tools_column_generate_cache();
    }

    public static function trigger_cron_job() {
        /**
         * Get data.
         */
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );

        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_trigger_cronjob_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
         * Notify the caller script that the Cron Job got 
         * triggered at this point.
         */
        self::send_json( 'Jptgb Cron Job running...' );
    }

    public static function get_ldl_data() {
        /**
         * Get the requestedd url.
         */
        $requested_url = esc_url_raw( wp_unslash( $_POST['requested_url'] ?? '' ) );
        $is_mobile = filter_var( wp_unslash( $_POST['is_mobile'] ?? '' ), FILTER_VALIDATE_BOOLEAN );

        /** 
         * Set the device name based on the mobile flag.
         *
         * @var string $device_name Either 'mobile' or 'desktop'.
         */
        $device_name = $is_mobile ? 'mobile' : 'desktop';

        /**
         * Get hashed url.
         */
        $hashed_url = md5( $requested_url );

        /**
         * Get LDL data.
         */
        $file_path  = LazyDomLoadController::get_ldl_dir_path() . '/' . $hashed_url . '-' . $device_name;
        $ldl_data   = file_get_contents( $file_path );
        $ldl_data   = json_decode( $ldl_data, true );

        /**
         * Return if we can't find the LDL data.
         */
        if( !$ldl_data ){
            self::send_json( 'We can\'t find the LDL data for this page', 'error' );
        }

        self::send_json( "Ldl Content Loaded", 'success', $ldl_data );
    }

    /**
     * Attempts to retrieve a term or post object based on a given URL.
     *
     * @param string $url The full URL to be checked.
     * @return WP_Term|WP_Post|null Depending on the URL, returns a WP_Term object if it's a taxonomy term page,
     *                               a WP_Post object if it's a single post/page/custom post type, or null if neither.
     */
    public static function get_object_by_url( $url ) {
        // Use wp_parse_url() to extract the path component of the URL
        $parsed_url = wp_parse_url( $url );
        $path = isset($parsed_url['path']) ? $parsed_url['path'] : '';

        // Normalize the path by trimming slashes
        $path = trim( $path, '/' );

        // Attempt to get a post ID first
        $post_id = url_to_postid( $url );
        if ( $post_id ) {
            /** This is a comment indicating a post object was found */
            return get_post( $post_id );
        }

        $blog_page_id = (int) get_option( 'page_for_posts' );

        if ( $blog_page_id ) {
            $blog_page = get_post( $blog_page_id );

            if ( $blog_page instanceof \WP_Post && $blog_page->post_name === $path ) {
                /** 
                 * This URL matches the slug of the Posts page 
                 * (e.g. /blog/ → page ID 1049 in your case)
                 */
                return $blog_page;
            }
        }

        // If not a post, attempt to resolve it as a term
        // Split the path into segments
        $segments = explode( '/', $path );
        foreach ( $segments as $segment ) {
            // Attempt to retrieve a term by its slug in each taxonomy
            $taxonomies = get_taxonomies( array(), 'objects' );
            foreach ( $taxonomies as $taxonomy ) {
                $term = get_term_by( 'slug', $segment, $taxonomy->name );
                if ( $term ) {
                    /** This is a comment indicating a term object was found */
                    return $term;
                }
            }
        }

        /** This is a comment indicating no matching object was found */
        return null; // No matching post or term found
    }

    /**
     * Updates the cache status for a given WP_Post object.
     *
     * Cache status could be:
     * - 'no-cache'
     * - 'cache-ready'
     * - 'working-on'
     *
     * @param WP_Post $wp_object The WP_Post object whose cache status is being updated.
     * @param string $status The cache status. Valid values are 'no-cache', 'cache-ready', 'working-on' and 'error'.
     * @param string $message The status message.
     */
    public static function update_cache_status( $wp_object, $status, $message, int $percentage = 0 ) {
        /**
         * Get the post id if available.
         */
        $post_id = is_numeric( $wp_object ) ? $wp_object : false;

        /**
         * List of valid status values.
         */
        $valid_statuses = ['no-cache', 'cache-ready', 'working-on', 'error'];

        /**
         * Check if the provided status is valid.
         */
        if ( !in_array( $status, $valid_statuses ) ) {
            /* translators: %1$s: comma-separated list of valid statuses. */
            $format = esc_html__( 'Invalid status value. Valid values are: %1$s', 'webspeed' );

            $safe_list = implode( ', ', $valid_statuses );

            // Build the final message, with translation and escaping.
            $message = sprintf( $format, $safe_list );

            throw new \InvalidArgumentException( esc_html( $message ) );
        }
        
        /**
         * In case $wp_object is WP_Post type
         */
        if( is_a( $wp_object, 'WP_Post' ) || $post_id ) {
            /**
             * Get the post id.
             */
            if( !$post_id ){
                $post_id = $wp_object->ID;
            }

            /**
             * Get current status.
             */
            $current_status = get_post_meta( $post_id, 'jptgb_cache_status', true );

            if( !$current_status ){
                $current_status = [];
            }

            /**
             * Define the new status.
             */
            $new_status = [
                'status'        => $status ? $status : $current_status['status'] ?? '',
                'message'       => $message ? $message : $current_status['message'] ?? '',
                'percentage'    => $percentage,
            ];

            /**
             * Update new status
             */
            update_post_meta( $post_id, 'jptgb_cache_status', $new_status );

            /**
             * Register post status id.
             */
            $all_posts_status   = get_option( 'jptgb_all_single_posts_status', [] );
            $all_posts_status[] = $post_id;
            update_option( 'jptgb_all_single_posts_status', $all_posts_status );
        } else {
            /**
             * For now just work with post objects.
             */
            return;
        }
    }

    public static function get_cache_status( $wp_object ) {
        /**
         * Get the post id if available.
         */
        $post_id = is_numeric( $wp_object ) ? $wp_object : false;
        
        /**
         * In case $wp_object is WP_Post type
         */
        if( is_a( $wp_object, 'WP_Post' ) || $post_id ) {
            /**
             * Get the post id.
             */
            if( !$post_id ){
                $post_id = $wp_object->ID;
            }

            /**
             * Get current status.
             */
            return get_post_meta( $post_id, 'jptgb_cache_status', true );

        } else {
            /**
             * For now just work with post objects.
             */
            return;
        }
    }

    /**
     * Sorts an array of arrays based on 'order' index in ascending order.
     *
     * @param array $array The array to be sorted.
     * @return array The sorted array.
     */
    private static function sort_array_by_order($array) {
        // Define the custom sort function
        usort($array, function ($a, $b) {
            // Compare the 'order' index of each element
            return $a['order'] <=> $b['order'];
        });

        return $array;
    }

    /**
     * Removes content enclosed between specific comment tags in HTML.
     *
     * This function searches for content enclosed between <!--jptgb-remove-script-open-->
     * and <!--jptgb-remove-script-close--> comments, including the comments themselves,
     * and removes it all from the provided HTML string.
     *
     * It uses string manipulation functions instead of regular expressions to perform
     * the removal, repeatedly searching for the start and end tags and removing the
     * content between them.
     *
     * @param string $html The HTML string to process.
     * @return string Processed HTML with specified content removed.
     */
    public static function remove_enclosed_content(string $html): string {
        $startTag = '<!--jptgb-remove-script-open-->';
        $endTag = '<!--jptgb-remove-script-close-->';

        // Continue looping as long as both start and end tags are found in the HTML.
        while (strpos($html, $startTag) !== false && strpos($html, $endTag) !== false) {
            $startPos = strpos($html, $startTag);
            $endPos = strpos($html, $endTag, $startPos);

            // If the end tag is found after the start tag
            if ($endPos !== false) {
                // Add the length of the end tag to remove it as well.
                $endPos += strlen($endTag);
                // Remove the content, including the start and end tags.
                $html = substr_replace($html, '', $startPos, $endPos - $startPos);
            }
        }

        return $html;
    }

    public static function insert_in_cache_script( $html, $is_mobile, $requested_url ){
        /**
         * Convert $is_mobile to string.
         */
        $is_mobile = $is_mobile ? 1 : 0;

        /**
         * Get current time
         */
        $current_date = '( ' . current_datetime()->format('Y/m/d H:i:s') . ' )';

        /**
         * Determines if is mobile
         */
        $device_label = $is_mobile ? ' - Mobile ' : ' - Desktop';

        if( !self::get_setting( 'mobile_cache' ) ){
            $device_label = ' - Responsive';
        }

        /**
         * Cache status string.
         */
        $status_string = 'console.log("WebSpeed Cache File created on '. $current_date . $device_label .'");';

        /**
         * Get the in-cache file contents.
         */
        $in_cache_file_contents = Utils::get_js_content( 'assets/js/controllers/in-cache.js' ); 

        /**
         * Get in-cache data.
         */
        $is_idl     = ( isset( self::$settings['jptgb_setting_immediate_dom_load'] ) && self::$settings['jptgb_setting_immediate_dom_load'] )   ? '1' : '0';
        $is_ldl     = ( isset( self::$settings['jptgb_setting_lazy_dom_load'] ) && self::$settings['jptgb_setting_lazy_dom_load'] )             ? '1' : '0';
        $jptgb_data = 'if(!document.jptgbData){ document.jptgbData = {
            isIdl           : '. $is_idl .',
            isLdl           : '. $is_ldl .',
            ajaxUrl         : "'. esc_url( admin_url( 'admin-ajax.php' ) ) .'",
            requestedUrl    : "'. $requested_url .'",
            isMobile        : '. $is_mobile .',
            loadingIcon     : "'. esc_url( JPTGB_URL . 'assets/svg/loading.svg' ) .'",
        }}';

        /**
         * Append it at the bottom of the document.
         */
        $html = str_replace( "</body>", '<script id="jptgb-in-cache" class="jptgb-script">'. $jptgb_data . $status_string . $in_cache_file_contents .'</script></body>', $html );

        return $html;
    }

    public static function enable_previously_disabled_scripts( $html_content ) {
        // Decode HTML entities
        $decoded_content = html_entity_decode($html_content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Remove custom comment markers
        $cleaned_content = preg_replace('/<!--jptgb-script-opening\s*<script/i', '<script', $decoded_content);
        $cleaned_content = preg_replace('/<\/script>\s*jptgb-script-closing-->/i', '</script>', $cleaned_content);
    
        return $cleaned_content;
    }   

    /**
     * Function to schedule a unique WP Cron job for each page that needs image compression.
     *
     * @param string $page_url The URL of the page to compress images for.
     */
    private static function schedule_generate_cache_job( $args ) {   
        if( isset( $args['is_mobile'] ) && $args['is_mobile'] ) {
            /**
             * Check if a job for this page is already scheduled to avoid duplication.
             */
            $scheduled = wp_next_scheduled( 'jptgb_generate_cache_for_page_mobile', [ $args ] );
            
            if( !$scheduled ) {
                /**
                 * Schedule the job with the unique ID as an argument.
                 */
                wp_schedule_single_event( time(), 'jptgb_generate_cache_for_page_mobile', [ $args ] );
            }
        } else {
            /**
             * Check if a job for this page is already scheduled to avoid duplication.
             */
            $scheduled = wp_next_scheduled( 'jptgb_generate_cache_for_page', [ $args ] );
            
            if( !$scheduled ) {
                /**
                 * Schedule the job with the unique ID as an argument.
                 */
                wp_schedule_single_event( time(), 'jptgb_generate_cache_for_page', [ $args ] );
            }
        }
    }

    public static function flush_critical_css() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        
        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_flush_all_cc_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
         * Flush the critical css.
         */
        CriticalCssController::flush_all_critical_css();

        /**
         * Flush all the cache.
         */
        CacheController::flush_all_cache_files();

        /**
         * Respond the backend.
         */
        // self::send_json( 'Critical Css flushed successfully.' );

                
        $data = [
            'callback' => 'refreshPageWithMessage',
        ];
    
        /**
         * Respond the backend.
         */
        self::send_json( 'Critical Css flushed successfully.', 'notice-success is-dismissible', $data );
    }

    public static function flush_all_images() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        
        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_flush_all_images_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
         * Flush all cache files.
         */
        ImagesController::flush_all_images_files();

        /**
         * Flush all the cache.
         */
        CacheController::flush_all_cache_files();

        $data = [
            'callback' => 'refreshPageWithMessage',
        ];
    
        /**
         * Respond the backend.
         */
        self::send_json( 'Compressed Images Deleted Successfully.', 'notice-success is-dismissible', $data );
    }

    public static function flush_all_cache() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        
        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_flush_all_cache_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
         * Flush all cache files
         */
        CacheController::flush_all_cache_files();
        
        $data = [
            'callback' => 'refreshPageWithMessage',
        ];
    
        /**
         * Respond the backend.
         */
        self::send_json( 'Cache Flushed Successfully.', 'notice-success is-dismissible', $data );
    }

    public static function delete_plugins_data() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        
        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_delete_plugins_data_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
         * Delte all plugins data from the database.
         */
        Utils::delete_all_plugins_data();
        
        $data = [
            'callback' => 'refreshPageWithMessage',
        ];
    
        /**
         * Respond the backend.
         */
        self::send_json( 'Plugins Data Deleted Successfully.', 'notice-success is-dismissible', $data );
    }

    public static function delete_user() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );
        $email = sanitize_email( wp_unslash( $_POST['email'] ?? '' ) );

        /**
         * Check the email integrity.
         */
        if ( ! $email || ! is_email( $email ) ) {
            self::send_json( 'Invalid or empty email address', 'error' );
        }
        
        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_deleteuser_nonce' ) ){
            self::send_json( 'Nonce not valid', 'error' );
        }

        /**
         * Get the api key.
         */
        $jptgb_settings = get_option( 'jptgb_settings', [] );
        $api_key        = $jptgb_settings['jptgb_setting_api_key'] ?? '';

        /**
         * Return if api key is empty.
         */
        if( !$api_key ){
            self::send_json( 'No API key set', 'error' );
        }

        /**
         * Build the endpoiint url.
         */
        $api_url = self::$webspeed_api_base . 'delete-user/' . urlencode( $email );

        /**
         * Call the api.
         */
        $response = wp_remote_request( $api_url, [
            'method'    => 'DELETE',
            'headers'   => [
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'sslverify' => self::$is_production,
        ] );

        /**
         * Hanlde wp errors.
         */
        if ( is_wp_error( $response ) ) {
            self::send_json( 'There was an error during the request', 'error', $response->get_error_messages() );
        }
    
        /**
         * Ge api data.
         */
        $code       = wp_remote_retrieve_response_code( $response );
        $message    = wp_remote_retrieve_response_message( $response );
    
        /**
         * Ensure response is valid
         */
        if ( 200 !== $code ) {
            self::send_json( "Request failed with status: {$code}, Message: {$message}", 'error' );
        }

        /**
         * Data to send.
         */
        $data = [
            'callback' => 'afterDeleteUser',
        ];
    
        /**
         * Respond the backend.
         */
        self::send_json( "User deleted successfully.", 'success', $data );
    }

    public static function test_api_endpoint() {
        $nonce = sanitize_key( wp_unslash( $_POST['nonce'] ?? '' ) );       
        
        /**
         * Check the wp nonce.
         */
        if( !$nonce || !wp_verify_nonce( $nonce, 'jptgb_test_api_endpoint_nonce' ) ) {
            self::send_json( 'Nonce not valid', 'error' );
            return;
        }

        /**
         * Get the api key.
         */
        $api_key = self::$settings['jptgb_setting_api_key'] ?? '';
    
        /**
         * Get the endpoint url.
         */
        $wp_endpoint = BaseController::get_endpoint_absolute_url();

        /**
         * Define arguments to send to the api.
         */
        $args = [
            'email'     => self::get_setting( 'jptgb_setting_user_email' ),
            'client'    => urlencode( sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST']  ?? 'Unknown site' ) ) ),
            'wpRestUrl' => $wp_endpoint,
        ];

        /** 
         * Payload to be sent to the API.
         */
        $body = wp_json_encode( $args );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,
            'blocking'      => true,
            'redirection'   => 2,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * API endpoint for generating critical CSS.
         */
        $api_endpoint = self::$webspeed_api_base . 'greet';

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /**
         * Handle wp errors.
         */
        if ( is_wp_error( $response ) ) {
            self::send_json( 'There was an error during the request. ' . $response->get_error_messages(), 'error' );
            return;
        }

        /**
         * Get the response code.
         */
        $response_code = wp_remote_retrieve_response_code( $response );
    
        /**
         * Handle errors.
         */
        if ( !$response_code || 200 !== $response_code ) {
            self::send_json( 'There was an error during the request.', 'error', $response );
            return;
        }

        /**
         * Sync wp data with the data from the api.
         */
        $body = wp_remote_retrieve_body( $response );

        if( $body ){
            /**
             * Decode the response.
             */
            $body = json_decode( $body, true );

            /**
             * Update the url limit variable.
             */
            if( isset( $body['allowedNumberUrls'] ) && is_numeric( $body['allowedNumberUrls'] ) && isset( $body['currentNumberUrls'] ) && is_numeric( $body['currentNumberUrls'] ) ){
                $allowed_number_urls    = intval( $body['allowedNumberUrls'] );
                $current_number_urls    = intval( $body['currentNumberUrls'] );
                $api_plan               = $body['planName'];
                $expiration_date        = $body['expirationDate'];
                $allowed_urls           = $body['allowedUrls'];

                /**
                 * Update the allowd urls array.
                 */
                BaseController::update_setting( 'jptgb_setting_allowed_urls', $allowed_urls );

                /**
                 * Update allowed numbers of urls.
                 */
                // BaseController::update_setting( 'jptgb_setting_allowed_number_of_urls', $allowed_number_urls );
                BaseController::update_setting( 'jptgb_setting_api_plan', $api_plan );
                BaseController::update_setting( 'jptgb_setting_expiration_date', $expiration_date );

                if( $current_number_urls >= $allowed_number_urls ) {
                    /**
                     * Let our base controller know that we reached the max number of 
                     * allowd urls.
                     */
                    BaseController::update_setting( 'jptgb_setting_is_url_limit_reached', true );
                } else {
                    /**
                     * Let our base controller know that we haven't reached the max number of 
                     * allowd urls.
                     */
                    BaseController::update_setting( 'jptgb_setting_is_url_limit_reached', false );
                }
            }
        }
    
        /**
         * Respond the backend.
         */
        self::send_json( "API Endpoint response: Ok | WP Rest API response: " . $body['wpRestApi'], 'success is-dismissible' );
    }

    public static function request_new_api() {
        self::send_json( 'Heyy! What are you doing?', 'error' );
        return;

        $email = sanitize_email( wp_unslash( $_POST['email'] ?? '' ) );
    
        /**
         * Check email integrity.
         */
        if ( ! $email || ! is_email( $email ) ) {
            self::send_json( 'Invalid or empty email address', 'error' );
            return;
        }
    
        /**
         * Check the wp nonce.
         */
        if ( ! $nonce || ! wp_verify_nonce( $nonce, 'jptgb_newapi_nonce' ) ) {
            self::send_json( 'Nonce not valid', 'error' );
            return;
        }
    
        /**
         * Call the api.
         */
        $response = wp_remote_post( self::$webspeed_api_base . 'register/', [
            'body'      => json_encode( [ 'email' => $email ] ),
            'headers'   => [ 'Content-Type' => 'application/json' ],
            'sslverify' => self::$is_production,
            'timeout'   => 10,
        ] );
    
        /**
         * Handle wp errors.
         */
        if ( is_wp_error( $response ) ) {
            self::send_json( 'There was an error during the request', 'error', $response->get_error_messages() );
            return;
        }
    
        /**
         * Ge api data.
         */
        $code       = wp_remote_retrieve_response_code( $response );
        $message    = wp_remote_retrieve_response_message( $response );
    
        /**
         * Ensure response is valid.
         */
        if ( 200 !== $code ) {
            self::send_json( "Request failed with status: {$code}, Message: {$message}", 'error' );
            return;
        }
    
        /**
         * Prepare the response body.
         */
        $data = [
            'callback' => 'afterNewApiRequested',
        ];

        /**
         * Respond the backend.
         */
        self::send_json( "An email with your new API key has been sent to <b>{$email}</b>.", 'success is-dismissible', $data );
    }

    private static function send_json( string $message, string $status = 'success', array $data = [], $is_dismissible = false ) {
        $response = [
            'status'    => $status,
            'message'   => $message,
            'data'      => $data,
            'dismissible' => $is_dismissible,
        ];

        wp_send_json( $response );
    }
}
