<?php 

namespace Jptgb\Controllers;

use Jptgb\Resources\Utils;
use Jptgb\Includes\Logs;
use WP_Error;

class BaseController {
    protected static $settings;
    protected static $controls;
    protected static $notifications;
    protected static $webspeed_api_base;
    protected static $changed_settings          = [];
    protected static $is_production             = false;
    protected static $is_api_request            = false;
    protected static $is_debugging_request      = false;
    protected static $is_mobile_cache_page      = null;
    protected static $scripts_loader_list       = [];
    protected static $logs                      = null;
    protected static $endpoint_namespace        = 'jptgb/v1';
    protected static $notifications_endpoint    = '/notify';
    protected static $wp_root_path              = '';
    protected static $requested_url             = '';
    protected static $post_ids_to_exclude       = [];

    public static function init() {
        /**
         * Initialize the logs object;
         */
        self::$logs = new Logs();

        /**
         * Get initial propertied status.
         */
        $settings = get_option( 'jptgb_settings', [] );

        /**
         * Get initial default values.
         */
        $default_values = self::get_settings_initial_values();

        /** 
         * Merge current settings with initial settings to ensure new settings are added.
         */
        self::$settings = array_merge( $default_values, $settings );

        /**
         * Check if the request is made from the API server (via UA) or via the 'jptgbgeneratecache' flag.
         */
        $user_agent = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ?? '' ) );

        /** @var bool $api_flag True if UA contains our API identifier. */
        $api_flag = ( strpos( $user_agent, 'JptgbApiRequest' ) !== false );

        /** 
         * @var bool $generate_cache_flag 
         * True if the 'jptgbgeneratecache' GET var is present and truthy. 
         */
        $generate_cache_flag = filter_var( wp_unslash( $_GET['jptgbgeneratecache'] ?? '' ), FILTER_VALIDATE_BOOLEAN );

        if ( $api_flag || $generate_cache_flag ) {
            self::$is_api_request = true;
        }

        /**
         * Determine whether this is a debugging request.
         *
         * @var bool $is_debugging_flag True if the 'jptgbdebugging' query var was sent and truthy.
         */
        self::$is_debugging_request = filter_var( wp_unslash( $_GET['jptgbdebugging'] ?? '' ), FILTER_VALIDATE_BOOLEAN );

        /**
         * Define the penthouse api base.
         */
        self::$webspeed_api_base = trailingslashit( BaseController::get_env_variable( 'JPTGB_API_BASE_URL' ) );

        /**
         * Initialize the notifications array.
         */
        self::$notifications = [];

        /**
         * Check if we are updating settings.
         */
        if( self::is_settings_update() ){
            /**
             * Iterate over all settings.
             */
            foreach( $_POST as $post_index => $post_value ){
                /**
                 * Check for checkbox group.
                 */
                if( 'checkboxes' === $post_index ){
                    foreach( $post_value as $checkbox_group_id => $checkboxes ){
                        if( !is_array( $checkboxes ) ){
                            continue;
                        }

                        $checboxes_data = [];

                        foreach( $checkboxes as $checkbox ){
                            $checboxes_data[] = $checkbox;
                        }

                        /**
                         * Update the changed settings array.
                         */
                        if( isset( self::$settings[$checkbox_group_id] ) && self::$settings[$checkbox_group_id] != $checkboxes ){
                            self::$changed_settings[] = $checkbox_group_id;   
                        }

                        /**
                         * Update settings object.
                         */
                        self::$settings[$checkbox_group_id] = $checboxes_data;
                    }
                }

                /**
                 * Continue if a post index do not contain our setting's prefix.
                 */
                if( strpos( $post_index, 'jptgb_setting_' ) === false ){
                    continue;
                }

                /**
                 * Store the post value as true or false.
                 */
                $post_value = ( 'button__on' === $post_value ) ? true : ( ( 'button__off' === $post_value ) ? false : $post_value );

                /**
                 * Debug can be active only for 1hr.
                 */
                if( 'jptgb_setting_debug_activate' === $post_index ){
                    if( $post_value ){
                        set_transient( 'jptbg_active_debug_expire', true, HOUR_IN_SECONDS * 1 );
                    }
                }

                /**
                 * Update the changed settings array.
                 */
                if( isset( self::$settings[$post_index] ) && self::$settings[$post_index] != $post_value ){
                    self::$changed_settings[] = $post_index;   
                }

                /**
                 * Update settings object.
                 */
                self::$settings[$post_index] = $post_value;
            }

            /**
             * Update the new settings in wp database.
             */
            update_option( 'jptgb_settings', self::$settings );

            /**
             * Check api key.
             */
            self::set_api_key_status();
        }

        $is_delete_all_logs = filter_var(
            wp_unslash( $_POST['delete_all_logs'] ?? '' ),
            FILTER_VALIDATE_BOOLEAN
        );
        
        /**
         * Check if we are deleting all logs.
         */
        if( $is_delete_all_logs ){
            $output = self::$logs->delete_all_logs();

            if( $output ){
                self::add_notification( 'All logs removed successfully' );

            } else {
                self::add_notification( 'Something went wrong while removing the logs from the database', 'error' );

            }
        }

        /**
         * Dispatch custom events.
         */
        self::dispatch_custom_events();

        /**
         * Add a custome one-minute schedule to the cron jobs.
         */
        add_filter( 'cron_schedules', [ self::class, 'add_custom_cron_schedule' ] );

        /**
         * Consume cache generation item.
         */
        add_action( 'jptgb_process_consumer_item', [ self::class, 'process_every_minute__callback' ] );

        


        // Row actions on single‐site installs:
        add_filter( 'plugin_action_links',                  [ self::class, 'disable_plugin_delete_action' ], 10, 2 );
        add_filter( 'network_admin_plugin_action_links',    [ self::class, 'disable_plugin_delete_action' ], 10, 2 );
        add_filter( 'bulk_actions-plugins',                 [ self::class, 'remove_bulk_delete_action' ], 10, 1 );

         /**
         * Note: Some hosting might serve 127.0.0.1 or localhost if the site is being served
         * behind e.g a load balancer or a proxy and HTTP_X_FORWARDED_HOST is not properly set 
         * in wp-config.php.
         */
        add_filter( 'home_url', [ self::class, 'filter_home_url' ], PHP_INT_MAX - 10, 2 );

        /**
         * Increase the nonce life so it doesn't expire before the cache files does.
         */
        add_filter( 'nonce_life', [ self::class, 'adjust_nonce_life' ], PHP_INT_MAX - 10 );
    }

    /**
     * Check if the settings form was submitted via the 'update_settings' action.
     *
     * @return bool True if the jptgb_update POST value equals 'update_settings', false otherwise.
     */
    public static function is_settings_update(): bool {
        return 'update_settings' === sanitize_text_field( wp_unslash( $_POST['jptgb_update'] ?? '' ) );
    }

    /**
     * Adjust the nonce lifetime so it never expires before cache files,
     * based on the 'cache_expiration_time' (hours) setting + 3h tolerance.
     *
     * @param int $lifetime Current nonce lifetime in seconds.
     * @return int New nonce lifetime (cache TTL + 3h), never less than the original.
     */
    public static function adjust_nonce_life( int $lifetime ): int {
        /**
         * Current cache expiration time.
         */
        $hours = absint( self::get_setting( 'cache_expiration_time' ) );

        /**
         * No custom TTL? Keep WP’s default.
         */
        if( 0 === $hours ) {
            return $lifetime;
        }

        /**
         * Compute desired TTL in seconds.
         */
        $desired = ( $hours + 3 ) * HOUR_IN_SECONDS;

        /**
         * Never shorten the lifetime below what WP already uses.
         */
        return max( $lifetime, $desired );
    }

    /**
     * Replace localhost/127.0.0.1 in any home_url() call
     * with the real site domain — preserving everything
     * after the host (path, query, slash, etc.).
     *
     * @param string $url  The complete URL including scheme, host, path.
     * @param string $path Path relative to the home URL (leading slash optional).
     * @return string      The filtered URL.
     */
    public static function filter_home_url( string $url, string $path ): string {
        // Only run if WP is still generating a local URL
        if ( false !== strpos( $url, '127.0.0.1' ) || false !== strpos( $url, 'localhost' ) ) {

            /** Get your public domain (no scheme, no slash) */
            $site_domain = Utils::get_site_doamin();

            /** Grab the original scheme (http or https) */
            $scheme = wp_parse_url( $url, PHP_URL_SCHEME );

            /** Build new host prefix, e.g. "https://wrtrading.com" */
            $replacement = $scheme . '://' . untrailingslashit( $site_domain );

            /** 
             * Replace only the scheme+host portion:
             *  - ^https?://[^/]+  matches "http(s)://{old-host}"
             * Everything after remains untouched.
             */
            $url = preg_replace(
                '#^https?://[^/]+#i',
                $replacement,
                $url
            ) ?: $url;
        }

        return $url;
    }

    /**
     * Disable the "Delete" action link for specific plugins in local environments.
     *
     * @param array  $actions     Array of row action links for the plugin.
     * @param string $plugin_file The plugin file path relative to the plugins dir.
     * @return array Modified action links.
     */
    public static function disable_plugin_delete_action( array $actions, string $plugin_file ): array {
        // Only run on local environments.
        if ( ! BaseController::is_local_env() ) {
            return $actions;
        }

        // List of plugin basenames to protect.
        $protected = array(
            'webspeed/webspeed.php',
            'webspeed-pro/webspeed.php',
        );

        if ( in_array( $plugin_file, $protected, true ) && isset( $actions['delete'] ) ) {
            unset( $actions['delete'] );
        }

        return $actions;
    }

    /**
     * Remove the bulk "Delete" action from the Plugins list when local.
     *
     * @param array $bulk_actions Array of bulk actions (key => label).
     * @return array Modified bulk actions.
     */
    public static function remove_bulk_delete_action( array $bulk_actions ): array {
        // Only run on local environments.
        if ( ! BaseController::is_local_env() ) {
            return $bulk_actions;
        }

        // The key for the Delete‐selected bulk action is 'delete-selected'.
        if ( isset( $bulk_actions['delete-selected'] ) ) {
            unset( $bulk_actions['delete-selected'] );
        }

        return $bulk_actions;
    }

    

    public static function process_every_minute__callback() {
        

        /**
         * Check cache integrity.
         */
        $cache_integrity = self::check_cache_system_integrity();

        /**
         * If the debug option has been active for more than 1 hr then lets turn it off.
         */
        $is_debug_expired = get_transient( 'jptbg_active_debug_expire' );
        if( !$is_debug_expired && self::get_setting('debug_activate') ){
            self::update_setting( 'debug_activate', false );
        }

        /**
         * If cache integrity fails then lets try the reactivation.
         */
        if( !$cache_integrity ){
            /**
             * Log the situation.
             */
            BaseController::$logs->register( 'The cache integrity check failed, attemting to reactivate the cache automatically' , 'warning' );

            /**
             * Reactivate the cache automatically
             */
            CacheController::activate_deactivate_the_cache( true, true );

            /**
             * Flush hosting's cache.
             */
            MainController::flush_hostings_cache();
        }
    }

    /**
     * Check the integrity of the cache system.
     *
     * @return bool Returns true if the cache system integrity is healthy, false otherwise.
     */
    public static function check_cache_system_integrity() {
        /**
         * If cache is deactivated, no further check is needed.
         */
        if ( ! self::get_setting( 'cache_activate' ) ) {
            return true;
        }

        /** 
         * Path to advanced-cache.php.
         */
        $advanced_cache_path = WP_CONTENT_DIR . '/advanced-cache.php';

        /**
         * Check if advanced-cache.php exists, is readable, and contains the expected phrase.
         */
        if ( ! Utils::does_file_contain_phrase( $advanced_cache_path, 'JPTGB_CACHE_DIR' ) ) {
            BaseController::$logs->register(
                'The cache is active but advanced-cache.php is either missing, unreadable, or invalid.',
                'warning'
            );
            return false;
        }

        return true;
    }

    

    /**
     * Add a custom cron schedule for every minute.
     *
     * @param array $schedules Existing schedules.
     * @return array Modified schedules.
     */
    public static function add_custom_cron_schedule( $schedules ) {
        $schedules['every_minute'] = [
            'interval' => 60,
            'display'  => __( 'Every Minute', 'webspeed' ),
        ];

        return $schedules;
    }

    public static function dispatch_custom_events() {
        /**
         * If it is not a request from the api.
         */
        if( self::$is_debugging_request ){
            // add_action( 'wp_footer', [self::class, 'custom_events__callback'], 0 );
        }
    }

    public static function get_env_variable( string $variable ){
        return isset( $_ENV[ $variable ] ) ? $_ENV[ $variable ] : null;
    }

    /**
     * Checks if the current environment is local.
     *
     * @return bool Returns true if in local environment, false otherwise.
     */
    protected static function is_local_env(): bool {
        return isset( $_ENV[ 'JPTGB_LOCAL_ENV' ] ) && $_ENV[ 'JPTGB_LOCAL_ENV' ] ? true : false;
    }

    public static function custom_events__callback(){
        $ajax_url               = esc_url( admin_url('admin-ajax.php') );
        $nonce                  = wp_create_nonce('jptgb_generate_cache_nonce');
        $trigger_cronjob_nonce  = wp_create_nonce('jptgb_trigger_cronjob_nonce');
        $exclude_scripts                = CacheController::get_script_exclusion( 'exclude_scripts' );
        $exclude_deactivation_scripts   = CacheController::get_script_exclusion( 'exclude_deactivation_scripts' );
        $page_id                = get_the_ID();
        $is_mobile              = wp_is_mobile() ? '1' : '0';
        $requested_url          = esc_js(Utils::get_current_url());
        $is_idl                 = isset(self::$settings['jptgb_setting_immediate_dom_load']) && self::$settings['jptgb_setting_immediate_dom_load'] ? '1' : '';
        $is_ldl                 = isset(self::$settings['jptgb_setting_lazy_dom_load']) && self::$settings['jptgb_setting_lazy_dom_load'] ? '1' : '';
        $is_cc                  = isset(self::$settings['jptgb_setting_critical_css_activate']) && self::$settings['jptgb_setting_critical_css_activate'] ? '1' : '';
        $is_load_scripts_on_fui = isset(self::$settings['jptgb_setting_load_scripts_on_fui']) && self::$settings['jptgb_setting_load_scripts_on_fui'] ? '1' : '';
        $idl_for_hidden_elements = isset(self::$settings['jptgb_setting_idl_for_hidden_elements']) && self::$settings['jptgb_setting_idl_for_hidden_elements'] ? '1' : '';
    
        $ldl_height = $is_mobile === '1' ? 900 : 1300;
    
        $jptgb_data = '
            <!--jptgb-remove-script-open-->
            <script id="jptgb-data" class="jptgb-script">
                /**
                 * Namespacing our data.
                 */
                if (!document.jptgbData) {
                    document.jptgbData = {
                        ldl_height: ' . $ldl_height . ',
                        isIdl: "' . $is_idl . '",
                        isLdl: "' . $is_ldl . '",
                        nonce: "' . $nonce . '",
                        triggerCronJobNonce: "' . $trigger_cronjob_nonce . '",
                        pageId: ' . $page_id . ',
                        isMobile: "' . $is_mobile . '",
                        requestedUrl: "' . $requested_url . '",
                        ajaxUrl: "' . $ajax_url . '",
                        isCriticalCss: "' . $is_cc . '",
                        scriptsToExcludeFromFui: "' . $exclude_scripts . '",
                        scriptsToAllowToExecuteOnPupetter: "' . $exclude_deactivation_scripts . '",
                        isLoadScriptsOnFui: "' . $is_load_scripts_on_fui . '",
                        ldlForHiddenElements: "' . $idl_for_hidden_elements . '",
                        generateCache: true,
                    };
                }
            </script>
            <!--jptgb-remove-script-close-->';
    
        return $jptgb_data;
    }

    public static function get_settings_initial_values() {
        return [
            'jptgb_setting_subscription_status'                     => false,
            'jptgb_setting_api_plan_status'                         => '',
            'jptgb_setting_api_plan'                                => '-',
            'jptgb_setting_expiration_date'                         => '-',
            'jptgb_setting_cache_activate'                          => false,
            'jptgb_setting_api_key'                                 => '',
            'jptgb_setting_is_api_key_valid'                        => '',
            'jptgb_setting_critical_css_activate'                   => '1',
            'jptgb_setting_scripts_activate'                        => true,
            'jptgb_setting_fonts_activate'                          => true,
            'jptgb_setting_load_fonts_on_fui'                       => '1',
            'jptgb_setting_load_fa_icons_on_fui'                    => false,
            'jptgb_setting_images_activate'                         => '1',
            'jptgb_setting_resize_images'                           => '1',
            'jptgb_setting_convert_images_to_webp'                  => '1',
            'jptgb_setting_optimize_only_atf_images'                => '0',
            'jptgb_setting_compress_images'                         => '0',
            'jptgb_setting_compress_svg'                            => false,
            'jptgb_setting_set_expire_headers'                      => false,
            'jptgb_setting_show_columntool_on_posts'                => '0',
            'jptgb_setting_later_dom_load_activate'                 => false,
            'jptgb_setting_inmediate_dom_load'                      => '0',
            'jptgb_setting_load_scripts_on_fui'                     => '1',
            'jptgb_setting_load_iframes_on_fui'                     => true,
            'jptgb_setting_generate_iframes_bg'                     => true,
            'jptgb_setting_exclude_ids_for_cache'                   => '',
            'jptgb_setting_immediate_dom_load'                      => true,
            'jptgb_setting_exclude_deactivation_scripts'            => '',
            'jptgb_setting_exclude_scripts'                         => self::get_default_scripts_exclusion_list(),
            'jptgb_setting_exclude_styles'                          => '',
            'jptgb_setting_include_in_critical_css'                 => '',
            'jptgb_setting_idl_for_hidden_elements'                 => '0',
            'jptgb_setting_lazy_dom_load'                           => false,
            'jptgb_setting_number_of_urls'                          => 0,
            'jptgb_setting_registered_urls'                         => 0,
            'jptgb_setting_is_url_limit_reached'                    => false,
            'jptgb_setting_allowed_urls'                            => [],
            'jptgb_setting_auto_cache_attempts'                     => 0,
            'jptgb_setting_time_between_scripts_load'               => 100,
            'jptgb_setting_creates_cache_on_page_visits'            => false,
            'jptgb_setting_bypass_cache_layers'                     => true,
            'jptgb_setting_custom_css_in_critical'                  => self::custom_critical_css(),
            'jptgb_setting_include_rulesets_in_critical'            => '',
            'jptgb_setting_debug_activate'                          => false,
            'jptgb_setting_allowed_domain'                          => '',
            'jptgb_setting_exclude_ids_for_ldl'                     => '',
            'jptgb_setting_modify_external_scripts'                 => '1',
            'jptgb_setting_exclude_external_scripts_from_modify'    => self::scripts_to_avoid_modification_default_list(),
            'jptgb_setting_activate_logs'                           => false,
            'jptgb_setting_webp_quality_rate'                       => '80',
            'jptgb_setting_mpa_webp_quality_rate'                   => '80',
            'jptgb_setting_activate_self_hosted'                    => '0',
            'jptgb_setting_path_puppetter_chromium'                 => '',
            'jptgb_setting_path_penthouse_chromium'                 => '',
            'jptgb_setting_flush_cache_on_posts_publish'            => false,
            'jptgb_setting_flush_cache_on_post_types_publish'       => [],
            'jptgb_setting_eligible_post_types'                     => ['post', 'page'],
            'jptgb_setting_delete_all_plugins_data_swtich'          => false,
            'jptgb_setting_allow_params'                            => false,
            'jptgb_setting_cache_expiration_time'                   => 240,
            'jptgb_setting_verify_token'                            => '',
            'jptgb_setting_free_account_created'                    => false,
            'jptgb_setting_autostart_expire_headers'                => false,
            'jptgb_setting_autostart_cache'                         => false,
            'jptgb_setting_site_url'                                => '',
            'jptgb_setting_site_domain'                             => '',
            'jptgb_setting_mobile_cache'                            => true,
            'jptgb_setting_use_custom_htaccess_path'                => false,
            'jptgb_setting_custom_htaccess_path'                    => '',
        ];
    }

    /**
     * Get the default scripts exclusion list.
     *
     * Returns a newline-separated string of MIME-type patterns
     * that should be excluded from script processing.
     *
     * @since 1.0.0
     *
     * @return string Newline-separated exclusion patterns.
     */
    private static function get_default_scripts_exclusion_list(): string {
        return 
            "{type: 'application/json'}"               . "\n" .
            "{type: 'application/xml'}"                . "\n" .
            "{type: 'text/plain'}"                     . "\n" .
            "{type: 'text/x-handlebars-template'}"     . "\n" .
            "{type: 'text/template'}"                  . "\n" .
            "{type: 'application/rss+xml'}"            . "\n" .
            "{type: 'application/ld+json'}";
    }

    /**
     * Get the default list of external script sources to exclude from modification.
     *
     * Returns a newline-separated list of JS object literals, each defining a `src`
     * URL to skip when processing scripts.
     *
     * @since 1.0.0
     *
     * @return string Newline-separated `{src: '...'}` entries.
     */
    private static function scripts_to_avoid_modification_default_list(): string {
        $entries = [
            "{src: 'https://www.paypal.com'}",
            "{src: 'https://js.stripe.com/v3/'}",
            "{src: 'https://squareup.com'}",
            "{src: 'https://www.authorize.net'}",
            "{src: 'https://www.braintreepayments.com'}",
            "{src: 'https://www.adyen.com'}",
            "{src: 'https://www.2checkout.com'}",
            "{src: 'https://www.payoneer.com'}",
            "{src: 'https://www.skrill.com'}",
            "{src: 'https://pay.google.com'}",
            "{src: 'https://www.apple.com/apple-pay/'}",
            "{src: 'https://pay.amazon.com'}",
            "{src: 'https://www.bluesnap.com'}",
            "{src: 'https://www.helcim.com'}",
            "{src: 'https://www.shopify.com/payments'}",
            "{src: 'https://www.dharmamerchantservices.com'}",
            "{src: 'https://www.worldpay.com'}",
            "{src: 'https://www.paymentcloudinc.com'}",
            "{src: 'https://squareup.com/us/en/online-store'}",
            "{src: 'https://www.payza.com'}",
            "{src: 'https://go.wepay.com'}",
        ];

        // Join entries with a newline for inline JS consumption.
        return implode( "\n", $entries );
    }

    /**
     * Generate critical CSS for below-the-fold content.
     *
     * Outputs CSS rules for the `.jptgb_btf_content` class,
     * suitable for inline critical CSS without `<style>` tags.
     *
     * @since 1.0.0
     *
     * @return string Filtered critical CSS rules.
     */
    private static function custom_critical_css(): string {
        $rules = [
            '/* Ensure below-the-fold content does not overflow horizontally */',
            '.jptgb_btf_content {',
            '    margin: 0;',
            '    padding: 0;',
            '    overflow: hidden;',
            '}',
            '',
            '/* Ensure below-the-fold content has accessible contrast */',
            '.jptgb_btf_content,',
            '.jptgb_btf_content * {',
            '    background-color: #ffffff !important;',
            '    color: #000000 !important;',
            '}',
            '',
            '/* Force static positioning for BTF elements with absolute children */',
            '.jptgb_btf_content,',
            '.jptgb_btf_content * {',
            '    position: static !important;',
            '}',
            '',
            '/* Style links within below-the-fold content for visibility */',
            'a.jptgb_btf_content,',
            '.jptgb_btf_content a {',
            '    display: block !important;',
            '    text-decoration: underline !important;',
            '    padding: 10px 30px !important;',
            '    margin: 20px !important;',
            '}',
        ];

        /** @var string $css Concatenated CSS rules. */
        $css = implode( "\n", $rules );

        /**
         * Filter the generated critical CSS.
         *
         * @param string $css The generated CSS rules.
         * @return string
         */
        return (string) apply_filters( 'jptgb_custom_critical_css', $css );
    }

    public static function set_api_key_status() {
        /**
         * Initially set the return status as success.
         */
        $api_key_status = '';

        /**
         * Check if api key is valid.
         */
        $status_response = self::validate_api_key();

        /**
         * If api key is valid
         */
        if( is_wp_error( $status_response ) ){
            $api_key_status = 'error';
            $error_message  = $status_response->get_error_message();
            self::add_notification( $error_message, 'error' );
        } 

        if( 'success' === $status_response ){
            $api_key_status = $status_response;
        }

        /**
         * Update the setting.
         */
        self::update_setting( 'jptgb_setting_is_api_key_valid', $api_key_status );

        /**
         * Sync plugin with api.
         */
        add_action('init', function () {
            self::sync_with_api();
        });
    }

    public static function sync_with_api() {
        /**
         * Get the api key.
         */
        $api_key = self::$settings['jptgb_setting_api_key'] ?? '';
    
        /**
         * Get the endpoint url.
         */
        $wp_endpoint = BaseController::get_endpoint_absolute_url();

        /**
         * Define arguments to send to the api.
         */
        $args = [
            'email'     => self::get_setting( 'jptgb_setting_user_email' ),
            'client'    => urlencode( sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST']  ?? 'Unknown site' ) ) ),
            'wpRestUrl' => $wp_endpoint,
        ];

        /** 
         * Payload to be sent to the API.
         */
        $body = wp_json_encode( $args );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,
            'blocking'      => true,
            'redirection'   => 2,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * API endpoint for generating critical CSS.
         */
        $api_endpoint = self::$webspeed_api_base . 'greet';

        /** 
         * Perform the API request.
         */
        $response       = wp_remote_post( $api_endpoint, $args );
        $response_code  = wp_remote_retrieve_response_code( $response );

        /**
         * Let's create an error object if the response code is different than 200.
         */
        if( 200 !== $response_code ) {
            /**
             * Get the error message.
             */
            $response_message = '';
            if ( is_array( $response )
                && isset( $response['response'] )
                && is_array( $response['response'] )
            ) {
                $response_message = (string) ( $response['response']['message'] ?? '' );
            }

            /**
             * Build an WP_Error object so we can handle it in the next step.
             */
            $response = new WP_Error( 'webspeed-api-connection-error', $response_message );
        }

        /**
         * Handle wp errors.
         */
        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            $error_code    = $response->get_error_code();
            $error_data    = $response->get_error_data(); // May or may not exist
    
            // Log or inspect the error details for debugging
            self::error_log( "Error message: $error_message" );
            self::error_log( print_r( $error_data, true ) );
            self::error_log( "Error code: $error_code" );

            return $response;
        }
       
        /**
         * Sync api data to our plugin.
         */
        AjaxController::sync_api_data_to_plugin( $response_code, $response );

        return true;
    }

    public static function validate_api_key() {
        /**
         * Get the api key.
         */
        $api_key = self::$settings['jptgb_setting_api_key'] ?? '';

        /**
         * Return if there is no api key declared.
         */
        if( !$api_key ){
            return '';
        }

        /**
         * Define arguments to send to the api.
         */
        $args = [
            'email'     => self::get_setting( 'jptgb_setting_user_email' ),
            'client'    => urlencode( sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST']  ?? 'Unknown site' ) ) ),
        ];

        /** 
         * Payload to be sent to the API.
         */
        $body = wp_json_encode( $args );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 10,
            'blocking'      => true,
            'redirection'   => 2,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * API endpoint for generating critical CSS.
         */
        $api_endpoint = self::$webspeed_api_base . 'validate';

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /**
         * Hanlde wp errors.
         */
        if( is_wp_error( $response ) ) {
            return new WP_Error( 'webspeed_api_connection_error', 'Can\'t communicate with the API.' );
        }
    
        /**
         * Ge api data.
         */
        $code = wp_remote_retrieve_response_code( $response );
        
        /**
         * Ensure response is valid
         */
        if( 200 !== $code ) {
            $message = wp_remote_retrieve_response_message( $response );
            return new WP_Error( 'webspeed_api_response_error', 'WebSpeed API: ' . $message );
        }

        return 'success';
    }

    public static function update_setting( string $setting_id, $value ) {
        /**
         * Ensure the setting id contains the prefix.
         */
        $setting_id = strpos( $setting_id, 'jptgb_setting_' ) === false ? 'jptgb_setting_' . $setting_id : $setting_id;

        /**
         * Update the setting property.
         */
        self::$settings[ $setting_id ] = $value;

        /**
         * Update the db.
         */
        update_option( 'jptgb_settings', self::$settings );
    }

    public static function build_custom_wp_response( $code = 200, $message = 'OK', $body = '', $headers = array(), $cookies = array(), $filename = null ) {
        return array(
            'headers'  => $headers,
            'body'     => $body,
            'response' => [
                'code'    => $code,
                'message' => $message,
            ],
            'cookies'  => $cookies,
            'filename' => $filename,
        );
    }

    public static function add_notification( string $message, string $status = 'success' ){
        /**
         * Register each notification in our interal logs.
         */
        BaseController::$logs->register( $message, $status );

        self::$notifications[] = [
            'status'    => $status,
            'message'   => $message,
        ];
    }

    public static function get_endpoint_absolute_url() {
        $absolute_url = self::$endpoint_namespace . self::$notifications_endpoint;

        return rest_url( $absolute_url );
    }

    public static function get_settings(){
        return self::$settings;
    }

    public static function get_setting( $setting_id ){
        /**
         * Ensure the setting id contains the prefix.
         */
        $setting_id = strpos( $setting_id, 'jptgb_setting_' ) === false ? 'jptgb_setting_' . $setting_id : $setting_id;

        return self::$settings[ $setting_id ] ?? null;
    }

    public static function error_log( $error_message ){
        if( self::get_setting('debug_activate') ){
            error_log( $error_message );
        }
    }
}