<?php

/**
 * The Cache Controller Class
 *
 * @author Yaidier Perez
 * 
 * */

namespace Jptgb\Controllers;

use PhpAmqpLib\Connection\AMQPStreamConnection;
use PhpAmqpLib\Message\AMQPMessage;
use PhpAmqpLib\Wire\AMQPTable;
use Jptgb\Resources\Utils;
use Jptgb\Controllers\MainController;
use Jptgb\Controllers\AjaxController;
use WP_Error;

class CacheController extends BaseController {
    /**
     * Define calss properties.
     */
    protected static $cache_dir_path        = false;
    protected static $webroot_path          = false;
    protected static $activating_exp_h      = false;
    protected static $serve_current_path    = false;
    public static $suppress_transition_post_status = false;

    public static function init() {
        self::create_the_cache_dir();
        self::define_properties();
        self::handle_cache_status();
        self::create_cache_for_visited_pages();
        self::update_cache_on_posts_modifications();
        // self::handle_set_expire_headers();
        self::handle_cache_flush_on_post_publish();
        self::handle_custom_htaccess();

        
    }

    public static function handle_custom_htaccess() {
        /**
         * Return if we are not setting a custom htaccess path.
         */
        if( ! in_array( 'jptgb_setting_custom_htaccess_path', self::$changed_settings ) ) {
            return;
        }

        /**
         * Bail if the value is empty.
        */
        $custom_htaccess_path = self::get_setting( 'jptgb_setting_custom_htaccess_path' );
        if( !$custom_htaccess_path ) {
            return;
        }
        
        /**
         * Return if file do not exist.
         */
        if ( ! Utils::does_file_exists( $custom_htaccess_path ) ) {
            /**
             * Notify the admin.
             */
            self::add_notification( 'We can\'t find the file at ' . $custom_htaccess_path . ' Please get in contact with your adminstrator.', 'warning' );

            /**
             * Reset the htacess path to blank.
             */
            self::update_setting( 'custom_htaccess_path', '' );

        } else if ( !self::is_file_writable( $custom_htaccess_path ) ) {
            /**
             * Notify the admin.
             */
            self::add_notification( 'The file at ' . $custom_htaccess_path . ' is not writable. Please get in contact with your adminstrator.', 'warning' );

            /**
             * Reset the htacess path to blank.
             */
            self::update_setting( 'custom_htaccess_path', '' );
        }
    }

    public static function handle_cache_flush_on_post_publish() {
        /**
         * Return if option is not enabled.
         */
        if( !self::get_setting( 'flush_cache_on_posts_publish' ) ){
            return;
        }

        /**
         * Hook into the transition_post_status action.
         */
        add_action( 'transition_post_status', [ self::class, 'flush_cache_on_post_publish__callback' ], 10, 3 );
    }

    public static function flush_cache_on_post_publish__callback( $new_status, $old_status, $post ) {
        /**
         * Run only if the post is being published (i.e., new_status is 'publish' and old_status is not 'publish').
         */
        if( $new_status === 'publish' && $old_status !== 'publish' ) {
            /**
             * Get the post types that triggers the cache flush.
             */
            $post_types = self::get_setting( 'flush_cache_on_post_types_publish' );

            /**
             * Check if the post is of a specific post type (e.g., 'my_custom_post_type').
             */
            if( !in_array( $post->post_type, $post_types ) ) {
                return;
            }
    
            /**
             * Make sure this is not an auto-save or a revision.
             */
            if( wp_is_post_revision( $post->ID ) || wp_is_post_autosave( $post->ID ) ) {
                return;
            }

            /**
             * Flush all cache files
             */
            CacheController::flush_all_cache_files();
        }
    }   

    public static function cache_buffer_start() {
        add_action( 'shutdown', function() {
            if (ob_get_length()) {
                ob_end_flush();
            }
        }, PHP_INT_MAX );

        ob_start( [ self::class, 'capture_buffered_output' ] ); 
    }

    public static function capture_buffered_output( $page_content ) {
        /**
         * Inject the jptgb data.
         */
        $jptgb_data =  BaseController::custom_events__callback();

        /**
         * Inject the scripts disabler if debugging is on.
         */
        $scripts_disabler_html  =  $jptgb_data . '<script>' . Utils::get_js_content( 'assets/js/debugging/scripts.js' ) . '</script>';
        $page_content           = str_replace( '</title>', '</title>' . $scripts_disabler_html, $page_content );

        /**
         * Return the cache content.
         */
        return $page_content;
    }

    /**
     * Disables and marks script tags in HTML, excluding those marked with a specific class.
     *
     * @param string $html The HTML content to process.
     * @return string Processed HTML content with scripts disabled and marked, except those excluded.
     */
    public static function disable_and_mark_scripts( $html ) {
        /**
         * Exclude scripts that contain the following keywords.
         */
        $keywords = [ 
            'class="jptgb-script"', // Our scripts.
        ];

        /**
         * Define scripts that we wanna remove from the cache file.
         */
        $scripts_to_remove = [];

        if( !self::$settings['jptgb_setting_load_fa_icons_on_fui'] ){
            $keywords[]             = 'kit.fontawesome.com';
            $scripts_to_remove[]    = 'kit.fontawesome.com';
        }

        /** 
         * Define the start and end markers for scripts.
         */
        $start_marker = '<script';
        $end_marker = '</script>';

        /** 
         * New markers to replace the original script tags.
         */
        $new_start_marker = '<!--jptgb-script-opening<script';
        $new_end_marker   = '</script>jptgb-script-closing-->';

        /**
         * Markers for scripts to be removed.
         */
        $remove_start_marker = '<!--jptgb-remove-script-open-->';
        $remove_end_marker   = '<!--jptgb-remove-script-close-->';

        /** 
         * Initialize the position for scanning the HTML content.
         */
        $pos = 0;
        
        while ( ( $pos = strpos( $html, $start_marker, $pos ) ) !== false ) {
            $exclude_script = false;
            $remove_script  = false;

            $end_pos = strpos( $html, '>', $pos + 1 );

            if ( $end_pos === false ) {
                break; /** Malformed HTML: No closing '>' found */
            }
            
            $script_tag_end = strpos( $html, $end_marker, $end_pos );
            if ( $script_tag_end === false ) {
                break; /** Malformed HTML: No closing '</script>' found */
            }

            /** 
             * Extract the script tag.
             */
            $script_tag = substr( $html, $pos, $script_tag_end + strlen( $end_marker ) - $pos );

            /**
             * Scripts to remove from the cache file.
             */
            foreach( $scripts_to_remove as $script_to_remove ) {
                if( strpos( $script_tag, $script_to_remove ) !== false ) {
                    $remove_script = true;
                    break;
                }
            }

            if( $remove_script ){
                /**
                 * Wrap the script tag with remove markers.
                 */
                $commented_script_tag = $remove_start_marker . $script_tag . $remove_end_marker;

                /**
                 * Replace the original script tag with the commented one.
                 */
                $html = substr_replace( $html, $commented_script_tag, $pos, strlen( $script_tag ) );

                // Move past this script to continue searching.
                $pos += strlen($commented_script_tag);
                continue;
            }

            /**
             * Exclude scripts that contain specifc keywords.
             */
            foreach( $keywords as $keyword ) {
                if( strpos( $script_tag, $keyword ) !== false ) {
                    $exclude_script = true;
                    break;
                }
            }

            if( $exclude_script ){
                /** 
                 * Skip this script tag and continue searching.
                 */
                $pos = $script_tag_end + strlen( $end_marker );
                continue;
            }

            /** 
             * Prepare the modified script tag with comment markers.
             */
            $modified_script_tag = $new_start_marker . substr( $script_tag, strlen( $start_marker ), -strlen( $end_marker ) ) . $new_end_marker;

            /** 
             * Replace the original script tag with the modified one.
             */
            $html = substr_replace( $html, $modified_script_tag, $pos, strlen( $script_tag ) );

            /** 
             * Update the position to the end of the modified script tag to continue searching.
             */
            $pos += strlen( $modified_script_tag );
        }

        return $html;
    }

    public static function get_cache_dir_path() {
        if( self::$cache_dir_path ) {
            return self::$cache_dir_path;
        }

        $upload_dir_info        = wp_upload_dir();
        $upload_dir_path        = $upload_dir_info['basedir'];
        self::$cache_dir_path   = $upload_dir_path . '/jptgb/cache/data';

        return self::$cache_dir_path;
    }

    public static function flush_all_cache_files() {
        /**
         * Clear all queued messages in RabbitMQ.
         */
        CacheController::remove_all_queued_messages_in_rabbitmq();

        

        /**
         * Get the cache directory path.
         */
        $cache_dir_path = self::get_cache_dir_path();              

        /**
         * Remove all static cache files.
         */
        if( is_dir( $cache_dir_path ) ) {
            Utils::remove_directory_recursively( $cache_dir_path );
        }

        /**
         * Delte all redirect transietns.
         */
        self::delete_all_has_redirect_transients();

        /**
         * Delete the post status date from meta values.
         */
        Utils::delete_all_post_meta_by_key( 'jptgb_cache_status' );

        /**
         * Clean the cache table.
         */
        Utils::clean_cache_table();

        /**
         * Signal the API that we have deleted the cache file.
         */
        $response = CacheController::request_api_to_delete_all_cache_records();

        if( false !== $response ) {
            $response_code = wp_remote_retrieve_response_code( $response );
            AjaxController::sync_api_data_to_plugin( $response_code, $response );
        }
    }

    public static function delete_has_redirect_transients( $url ){
        $list           = get_option( 'jptgb_has_redirect_list', [] );
        $transient_id   = md5( $url );

        if( isset( $list[ $transient_id ] ) ){
            delete_transient( 'jptgb_has_redirect_' . $transient_id );
            unset( $list[$transient_id ] );
            update_option( 'jptgb_has_redirect_list', $list );
        }
    }

    public static function delete_all_has_redirect_transients() {
        $list = get_option( 'jptgb_has_redirect_list', [] );

        foreach( $list as $transient_id => $value ){
            delete_transient( 'jptgb_has_redirect_' . $transient_id );
        }

        update_option( 'jptgb_has_redirect_list', [] );
    }

    public static function update_cache_on_posts_modifications() {
        /**
         * Return if cache are deactivated.
         */
        if( !BaseController::get_setting( 'jptgb_setting_cache_activate' ) ) {
            return;
        }

        /**
         * Flush cache on post update.
         */
        add_action( 'transition_post_status', function( $new_status, $old_status, $post ) {
            if ( self::$suppress_transition_post_status ) { 
                BaseController::$logs->register( 'Supress Transition Post Status is active so, early returning on transition post status for cache genreation for id: ' . esc_attr( $post->ID ), 'info' );
                return;
            }

            /**
             * Flush the cache if the post is published or updated.
             */
            if ( 'publish' === $new_status ) {
                /**
                 * Do not regenerate cache for post which do not have any cache file set.
                 */
                if( empty( Utils::get_cache_entry_by_post_id( $post->ID ) ) ){
                    BaseController::$logs->register( 'This post do not have a cache file, so lets avoid generating an unwated cache file for it, post id: ' . esc_attr( $post->ID ), 'info' );
                    return;
                }

                /**
                 * Flush the cache when a post is published or its status is updated but remains published.
                 */
                self::regenerate_cache_for_post_id( $post->ID, 'Generate Cache on post update for ' );

            /**
             * Remove the cache file if the post become unpublished.
             */
            } elseif ( 'publish' === $old_status && 'publish' !== $new_status ) {
                self::delete_cache_file_by_post_id( $post->ID );

            }
        }, 10, 3 );

        /**
         * Flush cache on comment posted.
         */
        add_action( 'comment_post', function( $comment_id, $comment_approved, $commentdata ) {
            if( !$comment_approved ){
                return;
            }

            if ( self::$suppress_transition_post_status ) {
                return;
            }

            /**
             * Get post id.
             */
            $post_id = $commentdata['comment_post_ID'];

            /**
             * Do not regenerate cache for post which do not have any cache file set.
             */
            if( empty( Utils::get_cache_entry_by_post_id( $post_id ) ) ){
                return;
            }

            self::regenerate_cache_for_post_id( $post_id, 'Flusing cache on comment posted for ' );
        }, 10, 3 );

        /**
         * Flush cache on comment approved.
         */
        add_action( 'transition_comment_status', function( $new_status, $old_status, $comment ) {
            if ( self::$suppress_transition_post_status ) {
                return;
            }

            if( $old_status != $new_status ) {
                /**
                 * Get post id.
                 */
                $post_id = $comment->comment_post_ID;

                /**
                 * Do not regenerate cache for post which do not have any cache file set.
                 */
                if( empty( Utils::get_cache_entry_by_post_id( $post_id ) ) ){
                    return;
                }

                /**
                 * Flush cache.
                 */
                self::regenerate_cache_for_post_id( $post_id, 'Flusing cache on comment approved for ' );
            }
        }, 10, 3 );
    }

    public static function delete_cache_file_by_post_id( $post_id ) {
        self::delete_cache_by_post_id( $post_id );

        /**
         * Flush hosting's cache.
         */  
        MainController::flush_hostings_cache( $post_id );
    }

    public static function delete_the_cache_files( $url ) {
        /**
         * Log the cache flush.
         */
        BaseController::$logs->register( 'Delete action - Deleting cache files for: ' . esc_url( $url ), 'info' );

        /**
         * Delete current cache file.
         */
        $parsed_url     = wp_parse_url( $url );
        $desktop_path   = CacheController::get_cache_dir_path() . '/desktop' . $parsed_url['path'];
        $mobile_path    = CacheController::get_cache_dir_path() . '/mobile' . $parsed_url['path'];

        Utils::delete_file( $desktop_path . 'index.html' );
        Utils::delete_file( $mobile_path . 'index.html' );

        /**
         * Delete param directories.
         */
        Utils::delete_prefixed_dirs( $desktop_path , 'jptgb-has-param-' );
        Utils::delete_prefixed_dirs( $mobile_path,  'jptgb-has-param-' );
    }

    public static function delete_cache_by_post_id( $post_id, bool $delete_consumer = true ) {
        /**
         * Get post url and the url hash.
         */
        $post_url = Utils::get_permalink( $post_id );

        

        /**
         * Delete the cache files.
         */
        self::delete_the_cache_files( $post_url );

        /**
         * Delete redirect transients of current url.
         */
        self::delete_has_redirect_transients( $post_url );

        /**
         * Delete cache data from the webspeed_cache table.
         */
        Utils::remove_cache_by_post_id( $post_id );

        /**
         * Log the cache flush.
         */
        BaseController::$logs->register( 'Cache Deleted for post: ' . esc_url( $post_url ), 'info' );
    }

    public static function regenerate_cache_for_post_id( $post_id, string $message = '' ){ 
        /**
         * Get post url.
         */
        $post_url = Utils::get_permalink( $post_id );

        if( !$post_url ){
            return;
        }

        AjaxController::update_cache_status( $post_id, 'working-on', 'Cache flushed', 5 ); 

        self::delete_cache_by_post_id( $post_id );

        /**
         * Let's request the api to regenerate the cache of this post.
         */
        if( self::start_the_build_cache_request( $post_id, $post_url ) ) {
            /**
            * Log the cache flush.
            */
            BaseController::$logs->register( $message . esc_url( $post_url ), 'info' );
        }
    }

    public static function define_properties() {
        self::$webroot_path = trailingslashit( Utils::get_true_wp_root_directory() );
    }

    public static function create_the_cache_dir() {
        /**
         * Return if cache are deactivated.
         */
        if( !BaseController::get_setting( 'jptgb_setting_cache_activate' ) ){
            return;
        }

        $cache_dir_path = self::get_cache_dir_path();

        if( !wp_mkdir_p( $cache_dir_path ) ){
            return false;
        }

        return true;
    }

    /**
     * Checks if the server is Apache or LiteSpeed.
     *
     * @return bool Returns true if the server is Apache or LiteSpeed, false otherwise.
     */
    public static function server_supports_htaccess(): bool {
        if( isset( $_SERVER['SERVER_SOFTWARE'] ) ) {
            if( stripos( $_SERVER['SERVER_SOFTWARE'], 'Apache' ) !== false ||
                stripos( $_SERVER['SERVER_SOFTWARE'], 'LiteSpeed' ) !== false ) {
                return true;
            }
        }

        return false;
    }

    public static function handle_cache_status() {
        /**
         * Return if we haven't changed the status of the cache.
         */
        if( !in_array( 'jptgb_setting_cache_activate', self::$changed_settings ) ){
            return;
        }

        /**
         * Check if we are activating or deactivaing the cache.
         */
        $activating_cache = self::$settings['jptgb_setting_cache_activate'] ? true : false;

        /**
         * Activate/Deactivate the cache.
         */
        self::activate_deactivate_the_cache( $activating_cache );

        /**
         * Flush hosting's cache.
         */
        MainController::flush_hostings_cache();
    }

    public static function activate_deactivate_the_cache( $is_activate, bool $is_cron = false ){
        /**
         * Initial .htaccess status.
         */
        $handle_htaccess = true;
        $handle_wp_cache = true;

        /**
         * Check if server support .htaccess.
         */
        if( self::server_supports_htaccess() ){
            /**
             * Get .htaccess file path.
             */
            $htaccess_path = self::get_htaccess_path();

            /**
             * Check if .htaccess file exist.
             */
            if( !file_exists( $htaccess_path ) ) {
                /**
                 * Log the situation.
                 */
                BaseController::$logs->register( 'We can\'t find the .htaccess file', 'warning' );

                /**
                 * Check if we can create the .htaccess file.
                 */
                if( !Utils::write_file( $htaccess_path, '' ) ){
                    /**
                     * Log the situation.
                     */
                    BaseController::$logs->register( 'We can\'t create the .htaccess file', 'warning' );

                    $handle_htaccess = false;
                } else {
                    /**
                     * Log the situation.
                     */
                    BaseController::$logs->register( 'We was able to create the .htaccess file', 'info' );
                }
            } else {
                /**
                 * Check if the file is writable
                 */
                if( !self::is_file_writable( $htaccess_path ) ){
                    /**
                     * Log the situation.
                     */
                    BaseController::$logs->register( 'The .htaccess file exist but it is not writable.', 'warning' );

                    $handle_htaccess = false;
                }
            }

            /**
             * Modify the .htaccess file.
             */
            if( $handle_htaccess ){
                $handle_htaccess = self::handle_htaccess_file( $is_activate );
            }
        } else {
            BaseController::$logs->register('Server does \'t support .htaccess', 'info');
        }

        /**
         * Define the path to the wp-config.php file.
         */
        $wp_config_path = self::$webroot_path . 'wp-config.php';

        /**
         * Verify that the wp-config.php file exists before proceeding.
         *
         * Uses WP_Filesystem::exists() instead of file_exists() for compatibility
         * with FTP/SSH filesystems.
         *
         * @var bool $wp_config_exists True if wp-config.php is found.
         */
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            /** Load the WP Filesystem API. */
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        global $wp_filesystem;

        /** @var bool $wp_config_exists */
        $wp_config_exists = $wp_filesystem->exists( $wp_config_path );

        /**
         * Check if wp-config.php file exist.
         */
        if( ! $wp_config_exists ) {
            if( !$is_cron ){
                BaseController::add_notification( 'We can\'t find the wp-config.php file in your server. Please get in contact with your adminstrator.', 'warning' );
            } else {
                BaseController::$logs->register( 'We can\'t find the wp-config.php file in your server. Please get in contact with your adminstrator.', 'warning' );
            }
            $handle_wp_cache = false;
        } 
        
        /**
         * Check if wp-config.php file is writable.
         */
        if( !wp_is_writable( $wp_config_path ) ) {
            if( !$is_cron ){
                BaseController::add_notification( 'Your wp-config.php file is not writable, please modify permissions and try again. Please get in contact with your adminstrator.', 'warning' );
            } else {
                BaseController::$logs->register( 'Your wp-config.php file is not writable, please modify permissions and try again. Please get in contact with your adminstrator.', 'warning' );
            }
            $handle_wp_cache = false;
        }
        
        /**
         * Check if wp-content is writable.
         */
        if( !wp_is_writable( WP_CONTENT_DIR ) ){
            if( !$is_cron ){
                BaseController::add_notification( 'Your wp-content directory is not writable, please allow write permissions to to wp-content and try again. Please get in contact with your adminstrator.', 'warning' );
            } else {
                BaseController::$logs->register( 'Your wp-content directory is not writable, please allow write permissions to to wp-content and try again. Please get in contact with your adminstrator.', 'warning' );
            }
            $handle_wp_cache = false;
        }

        /**
         * If we are fine to handle the wp cache.
         */
        if( $handle_wp_cache ){
            /**
             * If we have to activate the cache.
             */
            if( $is_activate ){
                /**
                 * Copy the advanced-cache.php file.
                 */
                $copy_output = self::copy_advanced_cache_file();

                /**
                 * If we successfully copied the file then lets define the 
                 * WP_CACHE to true in wp-config.php.
                 */
                if( $copy_output ){
                    BaseController::$logs->register( 'We copied the advanced-cache.php file into the wp-contents directory', 'success' );

                    /**
                     * Set the WP_CACHE contant to true.
                     */
                    $constant_output = self::add_wp_cache_definition( $wp_config_path );

                    /**
                     * In case we failed by seeting the constant. Then lets remove the advanced-cache file.
                     */
                    if( !$constant_output ){
                        /**
                         * Log the situation.
                         */
                        BaseController::$logs->register( 'We couldn\'t define the WP_CACHE = true in the wp-config.php file', 'error' );

                        /**
                         * Remove the adavanced-cache.php file.
                         */
                        self::remove_advanced_cache_file();

                        $handle_wp_cache = false;
                    } else {
                        BaseController::$logs->register( 'We defined WP_CACHE constant in the wp-config.php file', 'success' );
                    }
                
                /**
                 * The copy of the advanced-cache.php file didn't success as expected.
                 */
                } else{
                    /**
                     * Log the situation.
                     */
                    BaseController::$logs->register( 'We couldn\'t copy the advanced-cache.php file', 'error' );

                    $handle_wp_cache = false;
                }    
            } else {
                /**
                 * Remove the advacned-cache.php file.
                 */
                $remove_file_output = self::remove_advanced_cache_file();

                if( false === $remove_file_output ){
                    /**
                     * Log the situation.
                     */
                    BaseController::$logs->register( 'We couldn\'t remove the advanced-cache.php file', 'error' );

                    $handle_wp_cache = false;
                } else {
                    BaseController::$logs->register( 'File advanced-cache successfully removed', 'success' );
                }
                
                /**
                 * Remove the WP_CACHE cache definition from wp-config.php
                 */
                $remove_cosntant_output = self::remove_wp_cache_definition( $wp_config_path );                

                if( false === $remove_cosntant_output ){
                    /**
                     * Log the situation.
                     */
                    BaseController::$logs->register( 'We couldn\'t remove the constant WP_CACHE = true in the wp-config.php file', 'error' );

                    $handle_wp_cache = false;
                } else {
                    BaseController::$logs->register( 'Constant WP_CACHE successfully removed from the wp-config.php file', 'success' );
                }
            }
        }

        if( !$handle_htaccess && !$handle_wp_cache ){
            /**
             * Notify the admin.
             */
            if( !$is_cron ){
                BaseController::add_notification( 'We wasn\'t able to activate the cache on your site.', 'error' );
            } else {
                BaseController::$logs->register( 'We wasn\'t able to activate the cache on your site.', 'error' );
            }

            /**
             * Turn off the cache switch.
             */
            BaseController::update_setting( 'jptgb_setting_cache_activate', false );

            return false;
        }

        if( $is_activate ){
            if( !$is_cron ){
                BaseController::add_notification( 'Cache activated successfully.', 'success' );
            } else {
                BaseController::$logs->register( 'Cache activated successfully.', 'success' );
            }

        } else {
            if( !$is_cron ){
                BaseController::add_notification( 'Cache deactivated successfully.', 'success' );
            } else {
                BaseController::$logs->register( 'Cache deactivated successfully.', 'success' );
            }

        }

        /**
         * Activate/Deactivate expire headers.
         */
        self::handle_set_expire_headers( $is_activate );
    }

    private static function copy_file( $source, $destination ) {
        /**
         * Attempt to use the WordPress Filesystem API.
         */
        if (!function_exists('WP_Filesystem')) {
            include_once(ABSPATH . 'wp-admin/includes/file.php');
        }

        $filesystem_initialized = function_exists('WP_Filesystem') && WP_Filesystem();
        global $wp_filesystem;

        if ($filesystem_initialized && $wp_filesystem instanceof \WP_Filesystem_Base) {
            /**
             * Check if source file exists.
             */
            if ($wp_filesystem->exists($source)) {
                /**
                 * Attempt to copy the file.
                 */
                if ($wp_filesystem->copy($source, $destination, true)) {
                    return true;  // Successfully copied the file
                } else {
                    return new WP_Error('copy_failed', __('Failed to copy file.', 'webspeed'));
                }
            } else {
                return new WP_Error('no_source', __('Source file does not exist.', 'webspeed'));
            }
        } else {
            /**
             * Fallback to PHP's native copy function if WP_Filesystem is not initialized.
             */
            if (file_exists($source)) {
                if (@copy($source, $destination)) {
                    /**
                     * Successfully copied the file using native PHP copy.
                     */
                    return true;  
                } else {
                    return new WP_Error('copy_failed', __('Failed to copy file with native PHP copy.', 'webspeed'));
                }
            } else {
                return new WP_Error('no_source', __('Source file does not exist.', 'webspeed'));
            }
        }
    }

    private static function write_file( $file_path, $content ) {
        /**
         * Attempt to use the WordPress Filesystem API.
         */
        if (!function_exists('WP_Filesystem')) {
            include_once(ABSPATH . 'wp-admin/includes/file.php');
        }
        
        /**
         * Initialize the WordPress filesystem, no more using 'file-put-contents' directly.
         */
        if (function_exists('WP_Filesystem')) {
            WP_Filesystem();
        }
    
        global $wp_filesystem;
    
        /**
         * Check if the WordPress filesystem API is available and initialized.
         */
        if ( !empty( $wp_filesystem ) ) {
            /**
             * Successfully written using WP Filesystem.
             */
            if( $wp_filesystem->put_contents( $file_path, $content, FS_CHMOD_FILE ) ){
                return true;
            }
        } else {
            /**
             * Fallback to using file_put_contents if WP Filesystem is not available.
             */
            if ( function_exists( 'file_put_contents' ) ) {
                BaseController::$logs->register( '$wp_filesystem is not available using file_put_contents() to write the contents of ' . sanitize_text_field( $file_path ) , 'warning' );
                return file_put_contents( $file_path, $content ) !== false ? true : false;
            }
        }
    
        /**
         * Return false if both methods fail.
         */
        return false;
    }

    private static function is_file_writable( $file_path ){
        /**
         * Return false if the file doesn't exist.
         */
        if( !file_exists( $file_path ) ) {
            return false;
        }

        /**
         * Get the directory path.
         */
        $directory_path = dirname( $file_path );

        /**
         * Chekc if is writable.
         */
        return ( wp_is_writable( $directory_path ) && wp_is_writable( $file_path ) );
    }

    public static function handle_set_expire_headers( $set_expire_headers = false ) {
        /**
         * Return if we haven't changed the status of the expire headers.
         */
        // if( !in_array( 'jptgb_setting_set_expire_headers', self::$changed_settings ) ){
        //     return;
        // }

        /**
         * Check if we are activating or deactivaing the expire headers.
         */
        // self::$activating_exp_h = self::$settings['jptgb_setting_set_expire_headers'] ? true : false;

        self::$activating_exp_h = $set_expire_headers;

        /**
         * Get .htaccess file path.
         */
        $htaccess_file = self::get_htaccess_path();

        /**
         * Check if server support .htaccess file.
         */
        if( !self::server_supports_htaccess() ){
            BaseController::add_notification( 'Your server doesn\'t support .htaccess file but, please do contact your system administrator to set the Expire Headers manually.', 'warning' );
            return;
        }

        /**
         * Verify that the wp-config.php file exists before proceeding.
         *
         * Uses WP_Filesystem::exists() instead of file_exists() for compatibility
         * with FTP/SSH filesystems.
         *
         * @var bool $wp_config_exists True if wp-config.php is found.
         */
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            /** Load the WP Filesystem API. */
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        global $wp_filesystem;

        /** @var bool $wp_config_exists */
        $htaccess_file_exists = $wp_filesystem->exists( $htaccess_file );

        /**
         * Check if .htaccess file exist.
         */
        if( !$htaccess_file_exists ) {
            BaseController::add_notification( 'Your server seems to support .htaccess file but, we can\'t find it. We will try to add cache using wordpress built-in cache system.', 'warning' );
            return;
        }

        /**
         * Check if .htaccess file is writable.
         */
        if( !wp_is_writable( $htaccess_file ) ) {
            BaseController::add_notification( 'Your .htaccess file is not writable, please allow write permissions to .htaccess and try again. Meanwhile we will try to add cache using wordpress built-in cache system. Please get in contact with your adminstrator.', 'warning' );
            return;
        }

        /**
         * Handle the expire headers on .htaccess file.
         * Notify the admin if something has failed.
         */
        if( !self::handle_expire_headers() ){
            BaseController::add_notification( 'We couldn\'t change the expire headers.', 'error' );
        }

        /**
         * Notify the admin about the success.
         */
        if( self::$activating_exp_h ) {
            BaseController::add_notification( 'Expire Headers set successfully.', 'success' );
            
        } else {
            BaseController::add_notification( 'Expire Headers removed successfully.', 'success' );

        }
    }

    public static function remove_advanced_cache_file() {
        /**
         * Define file path.
         */
        $file_path = WP_CONTENT_DIR . '/advanced-cache.php';

        /**
         * Return true if file doesn't already exist.
         */
        if( !file_exists( $file_path ) ){
            return true;
        }

        return wp_delete_file( $file_path );
    }

    public static function copy_advanced_cache_file() {
        /**
         * Define source and destination variables.
         */
        $source         = JPTGB_PATH . 'Resources/advanced-cache.php';
        $destination    = WP_CONTENT_DIR . '/advanced-cache.php';

        /**
         * Perform the copy.
         */
        $output = self::copy_file( $source, $destination );

        /**
         * Check if we had any issue while copying the file.
         */
        if( is_wp_error( $output ) ) {
            $error_message = $output->get_error_message();

            /**
             * Log the situation.
             */
            BaseController::$logs->register( sanitize_text_field( $error_message ) , 'error' );

            return false;
        }

        return true;
    }
    
    public static function add_wp_cache_definition( $wp_config_path ) {
        /**
         * Get the files content.
         */
        $config_contents = Utils::read_file( $wp_config_path );

        /**
         * If we can't ready the contents then lets return false.
         */
        if( false === $config_contents ){
            BaseController::$logs->register( 'We can\'t read the content of the wp-config.php file', 'error' );
            return false;
        }
    
        if( strpos( $config_contents, "define('WP_CACHE', true);") !== false && strpos( $config_contents, "define('JPTGB_CACHE_DIR', '". self::get_cache_dir_path() ."');" ) !== false ) {
            return true;
        }

        /**
         * Define regex patterns.
         */
        $jptgb_cache_definination = '/define\s*\(\s*[\'"]WP_CACHE[\'"]\s*,\s*(false|true)\s*\)\s*;\s*define\s*\(\s*[\'"]JPTGB_CACHE_DIR[\'"]\s*,\s*["\'][^"\']*["\']\s*\)\s*;/';
        $cache_defination_pattern = '/define\s*\(\s*[\'"]WP_CACHE[\'"]\s*,\s*(false|true)\s*\)\s*;/';

        /**
         * Check if our cache definiation exists
         */
        if( preg_match( $jptgb_cache_definination, $config_contents ) ){
            /**
             * Remove our pattern.
             */
            $config_contents = preg_replace( $jptgb_cache_definination, '', $config_contents );

            /**
             * Remove any other advanced cache pattern declaration.
             */
            $config_contents = preg_replace( $cache_defination_pattern, '', $config_contents );

            /**
             * Add our advanced cache declaration.
             */
            $config_contents = str_replace( "<?php", "<?php\ndefine('WP_CACHE', true);\ndefine('JPTGB_CACHE_DIR', '". self::get_cache_dir_path() ."');", $config_contents );

        /**
         * In case our advanced cache declaration does not exist.
         */
        } else if ( strpos( $config_contents, "<?php" ) !== false ) {
            /**
             * Remove any other advanced cache pattern declaration.
             */
            $config_contents = preg_replace( $cache_defination_pattern, '', $config_contents );

            /**
             * Add our advanced cache declaration.
             */
            $config_contents = str_replace( "<?php", "<?php\ndefine('WP_CACHE', true);\ndefine('JPTGB_CACHE_DIR', '". self::get_cache_dir_path() ."');", $config_contents );

        } else {
            BaseController::add_notification( 'Your wp-config.php file seems to be corrupt... we cant find the "<?php" entry point. Please get in contact with your adminstrator.', 'warning' );
            return false;
        }

        return Utils::write_file( $wp_config_path, $config_contents );
    }

    public static function remove_wp_cache_definition( $wp_config_path ) {
        /**
         * Read the current contents of wp-config.php.
         */
        $config_contents = Utils::read_file( $wp_config_path );

        /**
         * If we can't read the contents then lets return false.
         */
        if ( false === $config_contents ) {
            BaseController::$logs->register( 'We can\'t read the content of the wp-config.php file', 'error' );
            return false;
        }

        /**
         * Define regex patterns.
         */
        $jptgb_cache_dir_pattern = '/^\s*define\s*\(\s*[\'"]JPTGB_CACHE_DIR[\'"]\s*,\s*[\'"][^\'"]*[\'"]\s*\)\s*;\s*$/m';
        $wp_cache_pattern        = '/^\s*define\s*\(\s*[\'"]WP_CACHE[\'"]\s*,\s*(true|false)\s*\)\s*;\s*$/m';

        /**
         * Remove the JPTGB_CACHE_DIR define line.
         */
        $config_contents = preg_replace( $jptgb_cache_dir_pattern, '', $config_contents );

        /**
         * Remove the WP_CACHE define line.
         */
        $config_contents = preg_replace( $wp_cache_pattern, '', $config_contents );

        /**
         * Write the updated contents back to wp-config.php.
         */
        return Utils::write_file( $wp_config_path, $config_contents );
    }

    public static function handle_htaccess_file( $is_activate ) {
        /**
         * Determine if we have to either activete or deactivate the cache.
         */
        if( $is_activate ){
            return self::toggle_caching_in_htaccess(true);

        } else {
            return self::toggle_caching_in_htaccess(false);

        }
    }

    private static function handle_expire_headers() {
        /**
         * Determine if we have to either activete or deactivate the cache.
         */
        if( self::$activating_exp_h ){
            return self::toggle_expire_headers_in_htaccess(true);

        } else {
            return self::toggle_expire_headers_in_htaccess(false);

        }
    }

    /**
     * Enable or disable caching in the .htaccess file.
     *
     * @param bool $enable True to enable caching, false to disable.
     */
    public static function toggle_caching_in_htaccess( $enable ) {
        // 1) Absolute disk path for the “-f” test
        // $cache_fs_path = self::get_cache_dir_path(); // e.g. /var/www/html/web/app/uploads/jptgb/cache/data
        $upload_info        = wp_upload_dir();
        $upload_base_url    = $upload_info['baseurl'];                      // e.g. https://example.com/wp-content/uploads
        $upload_base_path   = wp_parse_url( $upload_base_url, PHP_URL_PATH );  // e.g. /wp-content/uploads

        // Build the cache directory relative to document root:
        // e.g. /wp-content/uploads/jptgb/cache/data
        $cache_rel_path     = rtrim( $upload_base_path, '/' ) . '/jptgb/cache/data';
        $cache_rel_path     = ltrim( $cache_rel_path, '/' );

        /**
         * Constant to hold the caching rules.
         */
        define( 'JPTGB_CACHING_RULES', <<<HTACCESS
        # BEGIN JPTGB CACHE RULES 
        AddDefaultCharset UTF-8 
        <IfModule mod_rewrite.c> 
        RewriteEngine On 
        RewriteBase / 

        # --------------------------------------------------
        # Mobile cache (only guests, non-admin)
        # --------------------------------------------------
        RewriteCond %{REQUEST_URI} !^/wp-admin [NC]
        RewriteCond %{REQUEST_URI} !^/wp-login\.php [NC]
        RewriteCond %{HTTP_COOKIE} !wordpress_logged_in_ [NC]
        RewriteCond %{HTTP_USER_AGENT} (android|webos|iphone|ipad|ipod|blackberry|iemobile|opera\ mini|mobile) [NC]
        RewriteCond %{DOCUMENT_ROOT}/wp-content/uploads/jptgb/cache/data/mobile/$1/index.html -f
        RewriteRule ^(.*)$ /wp-content/uploads/jptgb/cache/data/mobile/$1/index.html [L]

        # --------------------------------------------------
        # Desktop cache (only guests, non-admin)
        # --------------------------------------------------
        RewriteCond %{REQUEST_URI} !^/wp-admin [NC]
        RewriteCond %{REQUEST_URI} !^/wp-login\.php [NC]
        RewriteCond %{HTTP_COOKIE} !wordpress_logged_in_ [NC]
        RewriteCond %{HTTP_USER_AGENT} !(android|webos|iphone|ipad|ipod|blackberry|iemobile|opera\ mini|mobile) [NC]
        RewriteCond %{DOCUMENT_ROOT}/wp-content/uploads/jptgb/cache/data/desktop/$1/index.html -f
        RewriteRule ^(.*)$ /wp-content/uploads/jptgb/cache/data/desktop/$1/index.html [L]
        </IfModule> 
        
        <IfModule mod_headers.c>
            <FilesMatch "^$cache_rel_path/.*\.html$">
                Header set Cache-Control "public, max-age=86400, immutable"
            </FilesMatch>
        </IfModule>
        # END JPTGB CACHE RULES
        HTACCESS
        );

        /**
         * Define the .htcaccess path.
         */
        $htaccess_file = self::get_htaccess_path();

        /**
         * Get the current file contents.
         */
        $htaccess_content = Utils::read_file( $htaccess_file );

        /**
         * If we can't read the contents then lets return false.
         */
        if( false === $htaccess_content ){
            BaseController::$logs->register( 'We can\'t read the content of the .htaccess file', 'error' );
            return false;
        }

        /**
         * Define the pattern.
         */
        $pattern = '/# BEGIN JPTGB CACHE RULES.*?# END JPTGB CACHE RULES/s';

        /** 
         * Enable or disable caching based on the parameter.
         */
        if( $enable ) {
            /**
             * If code already exist the just return true.
             */
            if( preg_match( $pattern, $htaccess_content ) ){
                return true;
            }

            /**
             * Compose the new content.
             */
            $htaccess_content = JPTGB_CACHING_RULES . $htaccess_content;

            /**
             * Attempt the write the file.
             */
            $output = Utils::write_file( $htaccess_file, $htaccess_content );

            if( false === $output ){
                BaseController::$logs->register( 'We couldn\'t enable cache rules in the .htaccess file', 'warning' );

            } else {
                BaseController::$logs->register( 'We enabled the cache rules in the .htaccess file', 'success' );
                
            }

            return $output;
        } else {
            /** 
             * Remove the rules if they are present.
             */
            $replacement = preg_replace( $pattern, '', $htaccess_content );
            
            if ( $replacement !== null && $replacement !== $htaccess_content ) {
                $output = Utils::write_file( $htaccess_file, $replacement );

                if( false === $output ){
                    BaseController::$logs->register( 'We couldn\'t remove the cache rewrite rules from the .htaccess file', 'warning' );

                } else {
                    BaseController::$logs->register( 'We removed the cache rules from the .htaccess file', 'success' );

                }

                return $output;
            } else {
                BaseController::$logs->register( 'We couldn\'t find the rules in the .htaccess file.', 'warning' );

                return false;
            }
        }
    }

    /**
     * Enable or disable expire headers in the .htaccess file.
     *
     * @param bool $enable True to enable expire headers, false to disable.
     */
    public static function toggle_expire_headers_in_htaccess( $enable ) {
        /**
         * All of our custom expiry & cache headers, in one neat nowdoc.
         */
        define( 'JPTGB_EXPIRE_HEADERS_RULES', <<<'HTACCESS'
        # BEGIN JPTGB EXPIRE HEADERS
        # Explicitly declaring the image/webp MIME type
        AddType image/webp .webp

        # Declare Expire headers.
        <IfModule mod_expires.c>
            ExpiresActive On
            ExpiresDefault                              "access plus 1 month"
            # cache.appcache needs re-requests in FF 3.6
            ExpiresByType text/cache-manifest           "access plus 0 seconds"
            ExpiresByType text/html                     "access plus 0 seconds"
            ExpiresByType text/xml                      "access plus 0 seconds"
            ExpiresByType application/xml               "access plus 0 seconds"
            ExpiresByType application/json              "access plus 0 seconds"
            ExpiresByType application/rss+xml           "access plus 1 hour"
            ExpiresByType application/atom+xml          "access plus 1 hour"
            ExpiresByType image/x-icon                  "access plus 1 week"
            ExpiresByType image/gif                     "access plus 4 months"
            ExpiresByType image/png                     "access plus 4 months"
            ExpiresByType image/jpeg                    "access plus 4 months"
            ExpiresByType image/webp                    "access plus 4 months"
            ExpiresByType video/ogg                     "access plus 4 months"
            ExpiresByType audio/ogg                     "access plus 4 months"
            ExpiresByType video/mp4                     "access plus 4 months"
            ExpiresByType video/webm                    "access plus 4 months"
            ExpiresByType image/avif                    "access plus 4 months"
            ExpiresByType image/avif-sequence           "access plus 4 months"
            ExpiresByType text/x-component              "access plus 1 month"
            ExpiresByType font/ttf                      "access plus 1 year"
            ExpiresByType font/otf                      "access plus 1 year"
            ExpiresByType font/woff                     "access plus 1 year"
            ExpiresByType font/woff2                    "access plus 1 year"
            ExpiresByType image/svg+xml                 "access plus 1 year"
            ExpiresByType application/vnd.ms-fontobject "access plus 1 year"
            ExpiresByType text/css                      "access plus 1 year"
            ExpiresByType application/javascript        "access plus 1 year"
        </IfModule>

        <IfModule mod_headers.c>
            <FilesMatch "\.(?:ttf|otf|woff2?|eot|svg)$">
                Header unset Expires
                Header unset Cache-Control
                Header set Expires "access plus 1 year"
                Header set Cache-Control "public, max-age=31536000, immutable"
            </FilesMatch>
        </IfModule>
        # END JPTGB EXPIRE HEADERS
        HTACCESS
        );

        /**
         * Define the .htcaccess path.
         */
        $htaccess_file = self::get_htaccess_path();

        /**
         * Get htaccess file contents.
         */
        $htaccess_content = Utils::read_file( $htaccess_file );

        /**
         * Return if we can't get the cotents of the file.
         */
        if( false === $htaccess_content ){
            BaseController::$logs->register( 'We can\'t read the content of the .htaccess file', 'warning' );
        }

        /**
         * Define the pattern.
         */
        $pattern = '/# BEGIN JPTGB EXPIRE HEADERS.*?# END JPTGB EXPIRE HEADERS/s';

        /** 
         * Enable or disable caching based on the parameter.
         */
        if( $enable ) {
            /**
             * If code already exist the just return true.
             */
            if( preg_match( $pattern, $htaccess_content ) ){
                return true;
            }

            $htaccess_content = JPTGB_EXPIRE_HEADERS_RULES . $htaccess_content;

            return Utils::write_file( $htaccess_file, $htaccess_content );
        } else {
            /** 
             * Remove the rules if they are present.
             */
            $replacement = preg_replace( $pattern, '', $htaccess_content );
            
            if ( $replacement !== null && $replacement !== $htaccess_content ) {
                return Utils::write_file( $htaccess_file, $replacement );
            }
        }
    }

    private static function validate_if_current_request_serves_funtional_page() {
        /**
         * Check for specific types of content. Add or remove conditions as needed for your site.
         */
        if (!is_single() && !is_page() && !is_category() && !is_author() && !is_archive() && !is_home()) {
            return false;
        }

        /**
         * Exclude feeds, search results, 404 pages, and any other non-functional pages.
         */
        if ( is_feed() || is_search() || is_404() ) {
            return false;
        }

        return true;
    }

    /**
     * Checks if the current request is a redirect check based on the custom header.
     *
     * @return bool True if the custom header is present, false otherwise.
     */
    private static function is_redirect_check() {
        // HTTP headers in PHP are prefixed with 'HTTP_' in $_SERVER.
        return isset( $_SERVER['HTTP_X_REDIRECT_CHECK'] ) && $_SERVER['HTTP_X_REDIRECT_CHECK'] === '1';
    }

        /**
     * Determine as early as possible if we should bypass the static‑cache
     * (i.e. skip serving or generating a cached file) based on the request.
     *
     * Checks for:
     *  - CLI mode (PHP_SAPI)
     *  - wp‑cron.php, xmlrpc.php, admin‑ajax.php, wp‑login.php access
     *  - Any /wp-admin or /wp-json/ URL
     *  - Logged‑in users via the raw HTTP_COOKIE header
     *
     * @return bool True to bypass cache; false to serve/cache.
     */
    public static function should_bypass_cache(): bool {
        // 1) CLI (fast constant, no function call)
        if ( PHP_SAPI === 'cli' ) {
            return true;
        }

        // Grab once, avoid repeated $_SERVER lookups        
        $script = sanitize_text_field( wp_unslash( $_SERVER['SCRIPT_NAME']  ?? '' ) );
        $uri    = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI']  ?? '' ) );

        // 1) Remove query string from URI
        $path = strstr( $uri, '?', true ) ?: $uri;

        // 2) Static‐asset extensions: bail on .css, .js, images, fonts, .map
        //    strrchr finds the last “.” and returns from there to end of string
        $ext = strtolower( strrchr( $path, '.' ) );
        if (
            $ext &&
            in_array( $ext, [
                '.css', '.js',
                '.jpg', '.jpeg', '.png', '.gif', '.webp', '.svg', '.ico',
                '.woff', '.woff2', '.ttf', '.eot',
                '.map',
            ], true )
        ) {
            return true;
        }

        // 2) Fast path script checks: wp‑cron, xmlrpc, ajax, login
        if (
            strpos( $script, 'wp-cron.php'   ) !== false ||
            strpos( $script, 'xmlrpc.php'    ) !== false ||
            strpos( $script, 'admin-ajax.php') !== false ||
            strpos( $script, 'wp-login.php'  ) !== false
        ) {
            return true;
        }

        // 3) URL‐based checks: admin pages or REST API
        if (
            strpos( $path, '/wp-admin' ) !== false ||
            strpos( $path, '/wp-json/' ) !== false
        ) {
            return true;
        }

        return false;
    }

    public static function create_cache_for_visited_pages() {
        /**
         * If cache is deactiavted, then just return.
         * 
         * @see Wp Dashboard -> Theme Settings -> Performance Settings
         */
        if( !isset( self::$settings['jptgb_setting_cache_activate'] ) || !self::$settings['jptgb_setting_cache_activate'] ) {
            return;
        }

        /**
         * Return if the option "Automatically Creates Cache On Page Visits" is deactivated.
         */
        if( !BaseController::get_setting( 'jptgb_setting_creates_cache_on_page_visits' ) ){
            return;
        }

        add_action( 'wp', function() {
            /**
             * Do not apply Optimizations for the backend or logged-in users.
             */
            if( is_admin() || is_user_logged_in() ) {
                return;
            }

            /*
             * If this is an autosave, our form has not been submitted,
             * so we don't want to do anything.
             */
            if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) || isset( $_REQUEST['bulk_edit'] ) ) {
                return;
            } 

            /**
             * Stop wp if we are checking url redirects.
             */
            if( self::is_redirect_check() ) {
                return; 
            }

            /**
             * Return if we are checking if the cache is available.
             */
            if( isset( $_GET['jptgb_check_cache_available'] ) ){
                return;
            }

            /**
             * Return if it is a request from the api. 
             */
            if( self::$is_api_request ){
                return;
            }

            /**
             * 
             */
            if( self::should_bypass_cache() ){
                return;
            }

            /**
             * Get the requested url.
             */
            $requested_url = Utils::get_current_url();

            BaseController::$logs->register( 'Automatic Cache generation requested for url: ' . esc_url( $requested_url ), 'info' );
            

            /**
             * Get cache paths.
             */
            $cache_paths = self::get_cache_file_paths_from_url( $requested_url );

            /**
             * Validate requested url.
             */
            if( !self::validate_if_current_request_serves_funtional_page() ){
                return false;
            }

            /**
             * Serve the cache file if it exist.
             */
            if( $serve_current_path = self::does_cache_file_exist( $cache_paths ) ){
                /**
                 * Log the inciedent.
                 */
                BaseController::$logs->register( wp_is_mobile() ? '[Mobile]' : '[Desktop]' . ' We got into WP execution for the url: [' . esc_url( $requested_url ) . ']' , 'warning' );

                /**
                 * Setting this value requires processing time, while the user is still waiting for the page to 
                 * get served, so we are gonna ensure to do it only once every 5 minutes.
                 */
                // self::try_activate_the_cache_automatically();

                /**
                 * Set the headers.
                 */
                header('Content-Type: text/html; charset=UTF-8');

                /**
                 * Output the file content.
                 */
                readfile( $serve_current_path );

                /**
                 * Stop the script execution.
                 */
                exit;
            }

            /**
             * Define defaults
             */
            $post_id = '';

            /**
             * If is singular, then lets check if the post id is set to be excluded.
             */
            if( !is_singular() ){
                return false;
            }

            /**
             * Get current post id.
             */
            $post_id = get_the_ID();
            
            self::start_the_build_cache_request( $post_id, $requested_url );
        } );
    }

    /**
     * Get the current request path without query string, and—if any
     * parameters exist—append their sanitized names as the last segment.
     *
     * Example:
     *   /update-test/?param1=a&param2=b  →  /update-test/param1param2
     *   (if that gets >200 chars, it uses a 12‑char MD5 hash instead)
     *
     * @return string Sanitized request path (no trailing slash).
     */
    public static function get_request_path( $requested_url, $tralingslashit = false ): string {    
        // 2) Break into components
        $parts = wp_parse_url( $requested_url );
        $path  = $parts['path'] ?? '/';

        // 3) Normalize base path (no trailing slash)
        $base = rtrim( $path, '/' );

        // 4) If there’s a query, parse out name→value pairs
        if ( ! empty( $parts['query'] ) ) {
            parse_str( $parts['query'], $pairs );

            if ( ! empty( $pairs ) ) {
                $fragments = [];

                // 5) Sanitize each name and value, then concatenate
                foreach ( $pairs as $name => $value ) {
                    /** Sanitize parameter name (A–Z, a–z, 0–9, _ and - only) */
                    $clean_name  = preg_replace( '/[^A-Za-z0-9_-]/', '', (string) $name );
                    /** Sanitize parameter value similarly */
                    $clean_value = preg_replace( '/[^A-Za-z0-9_-]/', '', (string) $value );

                    if ( '' !== $clean_name ) {
                        $fragments[] = $clean_name . $clean_value;
                    }
                }

                if ( ! empty( $fragments ) ) {
                    $combined = implode( '', $fragments );

                    // 6) If that segment gets too long, fall back to a hash
                    $max_len = 200;
                    if ( mb_strlen( $combined ) > $max_len ) {
                        // Hash of the raw name=value pairs for consistency
                        $flat_pairs = [];
                        foreach ( $pairs as $n => $v ) {
                            $flat_pairs[] = "{$n}={$v}";
                        }
                        $combined = substr( md5( implode( ',', $flat_pairs ) ), 0, 12 );
                    }

                    // 7 Prepend the param with a prefix
                    $combined = 'jptgb-has-param-' . $combined;

                    // 8) Append slash‑terminated param segment
                    $base = $base . '/' . $combined . '/';
                }
            }
        }

        if( $tralingslashit && function_exists( 'trailingslashit' ) ){
            $base = trailingslashit( $base );
        } 

        return $base;
    }

    private static function get_cache_file_paths_from_url( $requested_url ) {
        $url_path           = self::get_request_path( $requested_url, true );
        $desktop_cache_path = CacheController::get_cache_dir_path() . '/desktop' .  $url_path . 'index.html';
        $mobile_cache_path  = CacheController::get_cache_dir_path() . '/mobile' .  $url_path . 'index.html';

        return [ 'desktop' => $desktop_cache_path, 'mobile' => $mobile_cache_path ];
    }

    /**
     * Checks whether a cache file exists for the current device type.
     *
     * This function determines the appropriate cache path based on the device type (mobile or desktop)
     * and checks if the corresponding cache file exists. It utilizes a utility function to check file existence.
     *
     * @param array $cache_paths Associative array with keys 'mobile' and 'desktop' pointing to their respective cache paths.
     * @return mixed Returns the path to the cache file if it exists, false otherwise.
     */
    private static function does_cache_file_exist( array $cache_paths ) {
        /**
         * Determine the current path based on whether the device is mobile or not.
         * Uses wp_is_mobile to check device type and selects the appropriate path from the cache_paths array.
         */
        $current_path = wp_is_mobile() ? $cache_paths['mobile'] : $cache_paths['desktop'];

        /**
         * Utilizes a utility function to check if the file at the current path exists.
         * If the file does not exist, returns false.
         */
        if ( !Utils::does_file_exists( $current_path ) ) {
            return false;
        }

        /**
         * Return the current cache path if the file exists.
         */
        return $current_path;
    }

    public static function start_the_build_cache_request( $post_id = '', string $requested_url = '', $priority = 0 ) {
        /**
         * Return and avoid regenerating cache for post id if 
         * it is set to be excluded.
         */
        if( Utils::is_post_set_to_exclude( $post_id, 'exclude_ids_for_cache' ) ){ 
            return new WP_Error( 'error', 'This post/page is set to be exluded from the cache generation.', [ 'status' => 'warning' ] );
        }

        /**
         * Return if the url contains any query parameter.
         */
        if( !self::get_setting( 'allow_params' ) && Utils::has_url_parameters( $requested_url ) ){
            /**
             * Log the cache flush.
             */
            BaseController::$logs->register( 'Aborting Cache Generation (url with parameters) for' . esc_url( $requested_url ), 'warning' );

            return new WP_Error( 'error', 'The url has parameters', [ 'status' => 'warning' ] );
        }

        /**
         * Return if the requested url do not belong the the site's domain.
         */
        if( strpos( $requested_url, Utils::get_site_doamin() ) === false ){
            /**
             * Log the cache flush.
             */
            BaseController::$logs->register( 'Aborting Cache Generation. Url do not belong to site\'s domain ('. Utils::get_site_doamin() .') for: ' . esc_url( $requested_url ), 'info' );

            return new WP_Error( 'error', 'The requested url do not belong to the site\'s domain', [ 'status' => 'warning' ] );
        }

        /**
         * If the url is suspicious long, then we better avoid submitting 
         * for cache generation.
         */
        if( strlen( $requested_url ) > 250 ){
            /**
             * Log the cache flush.
             */
            $truncated_url = substr( $requested_url, 0, 250);
            $message = 'Aborting Cache Generation (url too long) for' . esc_url( $truncated_url );

            BaseController::$logs->register( $message, 'warning' );

            return new WP_Error( 'url_too_long', $message, [ 'status' => 'warning' ] );
        }

        /**
         * Get the requested device.
         */
        $requested_device = wp_is_mobile() ? '[Mobile]' : '[Desktop]';

        /**
         * Get the url hash.
         */
        $url_hash = Utils::create_url_safe_hash( $requested_url );

        /**
         * If the current chache has been requested already, then lets avoid re submit it again.
         */
        $cache_transient =  get_transient( "jptgb_cache_build_$url_hash" );
        if( $cache_transient ){
            /**
             * Log a warning.
             */
            $message = 'Aborting Cache Generation (cache requested less than 60s ago) for ' . esc_url( $requested_url );
            BaseController::$logs->register( $message, 'warning' );

            return new WP_Error( 'too_fast_request', $message, [ 'status' => 'warning' ] );
        }

        /**
         * Define a transient to avoid resubmission in the next 60 seconds.
         */
        set_transient( "jptgb_cache_build_$url_hash", true, 60 );

        /**
         * Get the transient redirect information.
         */
        delete_transient( 'jptgb_has_redirect_' . $url_hash );
        $redirect_transient = self::get_has_redirect_transient( $url_hash );

        if ( 'yes' === $redirect_transient ) {
            self::log_redirect_detected( $requested_device, $requested_url );

            return new WP_Error( 'error', 'Url has a redirect', [ 'status' => 'warning' ] );

        } else if ( false === $redirect_transient ) {
            $result             = Utils::get_url_http_code( $requested_url );
            $http_code          = $result['http_code'];
            $redirect_url       = $result['redirect_url'];
            $days_number        = self::determine_redirect_duration( $http_code );
            $has_redirect       = self::detect_redirect( $http_code );
            $has_https_redirect = $has_redirect ? self::detect_https_or_www_redirect( $requested_url, $redirect_url ) : false;

            if( $has_https_redirect ) {
                $transient_data = [
                    'redirect_url' => $redirect_url,
                ];

                self::set_has_redirect_transient( $url_hash, $transient_data, DAY_IN_SECONDS * $days_number );
                
                BaseController::$logs->register( 'Cache generation for url: ' . esc_url( $requested_url ) . ' has redirect to: ' . esc_url( $redirect_url ), 'warning' );
                $requested_url = $redirect_url;


            } else if ( $has_redirect ) {
                self::set_has_redirect_transient( $url_hash, 'yes', DAY_IN_SECONDS * $days_number );
                self::log_redirect_detected( $requested_device, $requested_url );

                /**
                 * Update cache status of current request.
                 */
                AjaxController::update_cache_status( $post_id, 'error', 'Redirect detected to: ' . $redirect_url, 0 );
                
                return new WP_Error( 'error', 'Redirect detected to: ' . $redirect_url );

            } else {
                self::set_has_redirect_transient( $url_hash, 'no', DAY_IN_SECONDS * $days_number );

            }

        } else if ( !empty( $redirect_transient[ 'redirect_url' ] ) ) {
            $requested_url = $redirect_transient[ 'redirect_url' ];

        }

        return self::request_cache_generation( $requested_url, $url_hash, $post_id, $priority );
    }

    /**
     * Detect if the redirect is due to a protocol (HTTP/HTTPS), port, or subdomain difference.
     *
     * @param string $requested_url The original URL.
     * @param string $redirect_url The redirected URL.
     * @return bool True if both URLs have the same domain and path, ignoring protocol, port, or subdomain.
     */
    private static function detect_https_or_www_redirect( $requested_url, $redirect_url ) {
        // Parse both URLs.
        $requested_parts = wp_parse_url( $requested_url );
        $redirect_parts = wp_parse_url( $redirect_url );

        // Extract domain without subdomains.
        $requested_host = self::strip_www_and_subdomain( $requested_parts['host'] ?? '' );
        $redirect_host = self::strip_www_and_subdomain( $redirect_parts['host'] ?? '' );

        // Normalize paths using trailingslashit().
        $requested_path = trailingslashit( $requested_parts['path'] ?? '/' );
        $redirect_path = trailingslashit( $redirect_parts['path'] ?? '/' );

        // Compare domain and path.
        return $requested_host === $redirect_host && $requested_path === $redirect_path;
    }

    /**
     * Normalize the domain by stripping subdomains (e.g., 'www').
     *
     * @param string $host The host (domain) to normalize.
     * @return string The normalized domain.
     */
    private static function strip_www_and_subdomain( $host ) {
        // Split the domain into parts.
        $parts = explode('.', $host);

        // Keep only the last two parts (e.g., example.com).
        if ( count( $parts ) > 2 ) {
            return implode('.', array_slice( $parts, -2 ));
        }

        return $host;
    }

    /**
     * Log a redirect detection message.
     */
    private static function log_redirect_detected( $requested_device, $requested_url ) {
        BaseController::$logs->register(
            $requested_device . ' Aborting Cache Generation: redirect detected for URL: [' . esc_url( $requested_url ) . ']',
            'warning'
        );
    }

    /**
     * Normalize a URL to HTTPS for comparison.
     *
     * @param string $url The URL to normalize.
     * @return string The normalized HTTPS URL.
     */
    private static function normalize_url_to_https( $url ) {
        $parsed_url = wp_parse_url( $url );

        // Ensure the scheme is HTTPS.
        $scheme = isset( $parsed_url['scheme'] ) && $parsed_url['scheme'] === 'http' ? 'https' : $parsed_url['scheme'];

        // Add the default port for HTTPS if it's not explicitly present.
        $host = $parsed_url['host'] ?? '';
        $port = isset( $parsed_url['port'] ) && $parsed_url['port'] !== 443 ? ':' . $parsed_url['port'] : '';

        // Rebuild the URL.
        $path = $parsed_url['path'] ?? '';
        $query = isset( $parsed_url['query'] ) ? '?' . $parsed_url['query'] : '';
        $fragment = isset( $parsed_url['fragment'] ) ? '#' . $parsed_url['fragment'] : '';

        return $scheme . '://' . $host . $port . $path . $query . $fragment;
    }

    /**
     * Determine the duration for a redirect transient based on HTTP status code.
     */
    private static function determine_redirect_duration( $http_code ) {
        switch ( $http_code ) {
            case 302:
                return 10;
            case 301:
                return 90;
            default:
                return 0;
        }
    }

    /**
     * Detect whether a redirect has occurred.
     */
    private static function detect_redirect( $http_code ) {
        if ( $http_code >= 300 && $http_code <= 399 ) {
            return true;
        }

        return false;
    }

    private static function request_cache_generation( $requested_url, $url_hash, $post_id, $priority = 0 ) {
        if( self::$is_api_request ){
            return;
        }

        /**
         * Normalize the url.
         */
        $requested_url = esc_url_raw( $requested_url );

        if( !self::get_setting( 'activate_self_hosted' ) ){
            return self::request_api_to_generate_cache( $requested_url, $url_hash, $post_id, $priority );
        } 
        
        /**
         * Get the cache data content.
         */
        $cache_file_data = self::request_self_hosted_to_generate_cache( $requested_url, $url_hash, $post_id );

        /**
         * Return false if the returned variable is not an array.
         */
        if( !is_array( $cache_file_data ) ){
            return false;
        }

        /**
         * Insert contents.
         */
        foreach( $cache_file_data as $device => $html_content ){
            MainController::insert_html_content( $requested_url, $url_hash, $html_content, $device );
        }
    }

    private static function request_self_hosted_to_generate_cache( $requested_url, $url_hash, $post_id ) {
        /**
         * Define the crhomium path for puppetter.
         */
        $chromium_path = self::get_setting('path_puppetter_chromium') ?: '';

        /**
         * Define api options.
         */
        $api_options = array_merge( [
            'url'           => $requested_url,
            'notEndpoint'   => BaseController::get_endpoint_absolute_url(),
            'urlHash'       => $url_hash,
            'post_id'       => $post_id,
            'chromiumPath'  => $chromium_path,
        ], self::get_options_for_api( $post_id ) );

        /**
         * Define the cache file contents
         */
        $cache_contents_data = [
            'desktop'   => '',
            'mobile'    => '',
        ];

        foreach( $cache_contents_data as $device => &$content ){
            /**
             * Arguments to pass.
             */
            $args = [ 
                'data'      => $api_options,
                'device'    => $device,
            ];

            /** 
             * Payload to be sent to the API.
             */
            $args = wp_json_encode( $args );
    
            /**
             * Escape the JSON string for use in the shell command.
             */
            $escaped_args = escapeshellarg( $args );
    
            /**
             * Command to execute the Node.js script with the JSON argument.
             */
            $command = 'node ' . JPTGB_PATH . 'src/js/self-hosted/api-controllers-interface.js generateCache ' . $escaped_args;
    
            /**
             * Execute the command and capture the output.
             */
            $output = shell_exec( $command );
    
            /**
             * Capture the error message.
             */
            if( $output === null ){
                /**
                 * Redirect stderr to stdout.
                 */
                $command .= ' 2>&1';
    
                /**
                 * Get the error message.
                 */
                $error_message = shell_exec( $command );
    
                /**
                 * Report the incident...
                 */
                BaseController::$logs->register( 'Self Cache generation failid for url: ' . esc_url( $error_message ), 'error' );

                return false;
            }

            $content = $output;
        }

        /**
         * Log the situation.
         */
        BaseController::$logs->register( 'Sef Generate Cache Requested for url: ' . esc_url( $requested_url ), 'info' );

        return $cache_contents_data;
    }

    public static function try_activate_the_cache_automatically() {
        /**
         * Get the transient.
         */
        // $transient = get_transient( 'jptgb_auto_activate_cache' );
        
        // if( $transient ){
        //     return false;
        // }

        /**
         * Define the transient.
         * 
         * !!!IMPORTANT Setting the transient for 10 seconds for testing purposes.
         */
        // $transient = set_transient( 'jptgb_auto_activate_cache', true, 10 );

        /**
         * Log the situation.
         */
        BaseController::$logs->register( 'Attemting to activate the cache automatically' , 'info' );

        /**
         * Activate the cache automatically.
         */
        self::activate_deactivate_the_cache( true );
    }

    public static function request_api_to_generate_cache( $requested_url, $url_hash, $post_id = '', $priority = 0 ){
        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            /**
             * Log the situation.
             */
            $message = 'Aborting Cache Generation (API Key is either missing or wrong) for url: [' . esc_url( $requested_url ) . ']';
            BaseController::$logs->register( $message , 'error' );
            return new WP_Error( 'error', $message);
        }

        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'generate-cache-v3';

        /**
         * Define api options.
         */
        $api_options = array_merge( [
            'url'           => $requested_url,
            'notEndpoint'   => BaseController::get_endpoint_absolute_url(),
            'urlHash'       => $url_hash,
            'postId'        => $post_id,
            'apiKey'        => $api_key,
            'pluginVersion' => JPTGB_VERSION,
        ], self::get_options_for_api( $post_id ) );

        /** 
         * Payload to be sent to the API.
         */
        $body = wp_json_encode( $api_options );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,    // Set a very low timeout to make the request 'non-blocking'
            'blocking'      => true,  // Non-blocking request, but not truly asynchronous
            'redirection'   => 5,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /**
         * Return if is wp error.
         */
        if( is_wp_error( $response ) ){
            /**
             * Log the situation.
             */
            BaseController::$logs->register( 'API Error: There was an error by requesting the cache generation for the url: ' . esc_url( $requested_url ), 'error' );

            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if( isset( $data['status'] ) && 'error' !== $data['status'] ){
            $rabbitmq_args          = $data['rabbitmq_credentials'];
            $subscription_data      = $data['subscription_data'];
            $sanitized_domain       = preg_replace( '/[^a-zA-Z0-9_\-]/', '_', Utils::get_site_doamin() );
            $rabbitmq_args['queue'] = 'generate_cache_' . $sanitized_domain;

            $api_options['apiUserId'] = $subscription_data['apiUserId'] ?? '';
           
            // Enqueue the task.
            self::enqueue_rabbitmq_task( $rabbitmq_args, $api_options, $priority );
        }

        /**
         * Log the situation.
         */
        BaseController::$logs->register( 'Generate Cache Requested for url: ' . esc_url( $requested_url ), 'info' );

        return $response;
    }

    /**
     * Converts an image to WebP format via API.
     *
     * Sends a request to the API endpoint to convert an image to WebP format.
     * 
     * @param string $image_content Base64 encoded string of the image.
     * @param string $image_format  The format of the image being converted.
     *
     * @return string|false The Base64 encoded string of the WebP image or false on failure.
     */
    public static function request_api_to_delete_cache( $requested_url, $url_hash ) {
        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'delete-cache';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /** 
         * Payload to be sent to the API.
         * Adjusted to send image content and format.
         */
        $body = wp_json_encode( [
            'url'               => $requested_url,
            'urlHash'           => $url_hash,
            'email'             => self::get_setting( 'user_email' ),
            'isDebug'           => self::get_setting( 'debug_activate' ),
        ]);

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 30,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting an url delete: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( 200 !== $status_code ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }
        
        /**
         * Return the received data.
         */
        return $response;
    }

    public static function request_api_to_delete_all_cache_records() {
        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'delete-all-cache-records';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /** 
         * Payload to be sent to the API.
         * Adjusted to send image content and format.
         */
        $body = wp_json_encode( [
            'email'             => self::get_setting( 'user_email' ),
            'isDebug'           => self::get_setting( 'debug_activate' ),
        ]);

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 30,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting an url delete: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( 200 !== $status_code ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }
        
        /**
         * Return the received data.
         */
        return $response;
    }

    public static function remove_all_queued_messages_in_rabbitmq() {
        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            /**
             * Log the situation.
             */
            $message = 'Aborting Removing all Queued messages in RabbitMQ, either API key is missing or it status is not valid';
            BaseController::$logs->register( $message , 'error' );
            return new WP_Error( 'error', $message);
        }

        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'get-rmq-credentials';

        /**
         * Define api options.
         */
        $api_options = [ 'apiKey' => $api_key ];

        /** 
         * Payload to be sent to the API.
         */
        $body = wp_json_encode( $api_options );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,    // Set a very low timeout to make the request 'non-blocking'
            'blocking'      => true,  // Non-blocking request, but not truly asynchronous
            'redirection'   => 5,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /**
         * Return if is wp error.
         */
        if( is_wp_error( $response ) ){
            /**
             * Log the situation.
             */
            // BaseController::$logs->register( 'API Error: There was an error by requesting the cache generation for the url: ' . esc_url( $requested_url ), 'error' );

            return false;
        }

        $body                   = wp_remote_retrieve_body( $response );
        $data                   = json_decode( $body, true );
        $rabbitmq_args          = $data['rabbitmq_credentials'];
        $sanitized_domain       = preg_replace( '/[^a-zA-Z0-9_\-]/', '_', Utils::get_site_doamin() );
        $rabbitmq_args['queue'] = 'generate_cache_' . $sanitized_domain;
       
        /**
         * Enqueue the task.
         */
        self::clear_rabbitmq_queue( $rabbitmq_args );
    }

    /**
     * Clear all messages from a RabbitMQ queue.
     *
     * @param string $queue The name of the queue to clear.
     * @param array $rabbitmq_args RabbitMQ connection details.
     * @return void
     */
    public static function clear_rabbitmq_queue( $rabbitmq_args ) {
        try {
            // RabbitMQ connection details
            $queue              = $rabbitmq_args['queue'];
            $rabbitmq_host      = BaseController::get_env_variable( 'JPTGB_RMQ_URL' );
            $rabbitmq_port      = BaseController::get_env_variable( 'JPTGB_RMQ_PORT' );
            $rabbitmq_user      = $rabbitmq_args['username']; // Replace with your RabbitMQ username
            $rabbitmq_password  = $rabbitmq_args['password']; // Replace with your RabbitMQ password

            // Establish connection
            $connection = new AMQPStreamConnection( $rabbitmq_host, $rabbitmq_port, $rabbitmq_user, $rabbitmq_password );
            $channel = $connection->channel();

            // Purge the queue (remove all messages)
            $channel->queue_purge( $queue );

            // Close the channel and connection
            $channel->close();
            $connection->close();

            BaseController::$logs->register( 'RabbitMQ Queue Purged Successfully' , 'success' );
        } catch ( \Exception $e ) {
            /**
             * Log the situation.
             */
            $message = 'Failed to clear queue in RabbitMQ | Error message: ' . $e->getMessage();
            BaseController::$logs->register( $message , 'error' );
        }
    }

    /**
     * Enqueue a task in RabbitMQ with an optional priority.
     *
     * @param array $rabbitmq_args RabbitMQ connection details.
     * @param array $payload The data to enqueue.
     * @param int $priority The priority of the task (default is 0, higher is more priority).
     * @return void
     */
    public static function enqueue_rabbitmq_task( $rabbitmq_args, $payload, $priority = 0 ) {
        try {
            // RabbitMQ connection details
            $queue              = $rabbitmq_args['queue'];
            $rabbitmq_host      = BaseController::get_env_variable( 'JPTGB_RMQ_URL' );
            $rabbitmq_port      = BaseController::get_env_variable( 'JPTGB_RMQ_PORT' );
            $rabbitmq_user      = $rabbitmq_args['username'];
            $rabbitmq_password  = $rabbitmq_args['password'];

            // Establish connection
            $connection = new AMQPStreamConnection( $rabbitmq_host, $rabbitmq_port, $rabbitmq_user, $rabbitmq_password );
            $channel = $connection->channel();

            // Set the queue arguments
            $arguments = new AMQPTable([
                'x-max-priority' => 10 // Maximum priority for this queue
            ]);           

            // Declare the queue with support for priorities
            $channel->queue_declare( $queue, false, true, false, false, false, $arguments);

            // Create the task message
            $messageBody = json_encode( $payload );
            $message = new AMQPMessage( $messageBody, [
                'delivery_mode' => AMQPMessage::DELIVERY_MODE_PERSISTENT, // Make the message persistent
                'priority'      => $priority, // Set the message priority
            ]);

            // Publish the message to the queue
            $channel->basic_publish( $message, '', $queue );

            // Close the channel and connection
            $channel->close();
            $connection->close();

            self::error_log( 'Task enqueued successfully: ' . $messageBody );
        } catch ( \Exception $e ) {
            $error_message = 'Failed to enqueue task: ' . $e->getMessage();

            self::error_log( $error_message );
        }
    }

    /**
     * Enqueue multiple tasks in RabbitMQ with optional priorities.
     *
     * @param array $rabbitmq_args RabbitMQ connection details.
     * @param array $payloads Array of payloads to enqueue. Each payload can have its own data and priority.
     *                        Example: [ ['data' => [...], 'priority' => 5], ['data' => [...], 'priority' => 10] ]
     * @return void
     */
    public static function enqueue_multiple_rabbitmq_tasks( $rabbitmq_args, $payloads ) {
        try {
            // RabbitMQ connection details
            $queue              = $rabbitmq_args['queue'];
            $rabbitmq_host      = BaseController::get_env_variable( 'JPTGB_RMQ_URL' );
            $rabbitmq_port      = BaseController::get_env_variable( 'JPTGB_RMQ_PORT' );
            $rabbitmq_user      = $rabbitmq_args['username'];
            $rabbitmq_password  = $rabbitmq_args['password'];

            // Establish connection
            $connection = new AMQPStreamConnection( $rabbitmq_host, $rabbitmq_port, $rabbitmq_user, $rabbitmq_password );
            $channel = $connection->channel();

            // Set the queue arguments
            $arguments = new AMQPTable([
                'x-max-priority' => 10 // Maximum priority for this queue
            ]);

            // Declare the queue with support for priorities
            $channel->queue_declare( $queue, false, true, false, false, false, $arguments );

            // Loop through the payloads and publish each one
            foreach ( $payloads as $payload_item ) {
                $message_body = json_encode( $payload_item['data'] ); // Convert the payload data to JSON

                // Set the message properties
                $message = new AMQPMessage( $message_body, [
                    'delivery_mode' => AMQPMessage::DELIVERY_MODE_PERSISTENT, // Make the message persistent
                    'priority'      => isset( $payload_item['priority'] ) ? $payload_item['priority'] : 0, // Set priority, default to 0
                ]);

                // Publish the message to the queue
                $channel->basic_publish( $message, '', $queue );
            }

            // Close the channel and connection
            $channel->close();
            $connection->close();

            self::error_log( 'All tasks enqueued successfully.' );
        } catch ( \Exception $e ) {
            $error_message = 'Failed to enqueue tasks: ' . $e->getMessage();

            self::error_log( $error_message );
        }
    }

    public static function get_ldl_setting( $ldl_setting, $post_id = false ) {
        /**
         * Check if there is a specific value for this post id.
         */
        if( $post_id ){
            return Utils::is_post_set_to_exclude( $post_id, 'exclude_ids_for_ldl' ) ? '' : '1';
        }

        /**
         * Return the global value.
         */
        return BaseController::get_setting( $ldl_setting ) ? '1' : '';
    }

    public static function get_script_exclusion( $exclusion_list ){
        /**
         * Get the seting list.
         */
        $exclusion_list = self::get_setting( $exclusion_list );

        /**
         * If not list is set, then return an empty string.
         */
        if( !$exclusion_list ){
            return '{}';
        }

        /**
         * Remove all spaces.
         */
        $exclusion_list = preg_replace( '/\s+/', '', $exclusion_list );
    
        /**
         * Replace all double quotes with single quotes.
         */
        $exclusion_list = str_replace( '"', "'", $exclusion_list );

        /**
         * Normalize data before sending it to the API.
         */
        return $exclusion_list;

    }

    public static function get_options_for_api( $post_id ) {   
        $ajax_url                       = esc_url( admin_url( 'admin-ajax.php' ) );
        $is_mobile                      = wp_is_mobile() ? '1' : '0';
        $exclude_scripts                = self::get_script_exclusion( 'exclude_scripts' );
        $exclude_styles                 = self::get_script_exclusion( 'exclude_styles' );
        $exclude_deactivation_scripts   = self::get_script_exclusion( 'exclude_deactivation_scripts' );
        $is_idl                         = self::get_ldl_setting( 'immediate_dom_load', $post_id );
        $is_ldl                         = self::get_ldl_setting( 'lazy_dom_load', $post_id );
        $load_iframes_on_fui            = self::get_setting( 'load_iframes_on_fui' );
        $generate_iframes_bg            = self::get_setting( 'generate_iframes_bg' );
        $is_debug                       = self::get_setting( 'debug_activate' );
        $is_mobile_cache                = self::get_setting( 'mobile_cache' );
        $is_bypass_cache_layers         = self::get_setting( 'bypass_cache_layers' );
        $is_cc                          = ( isset( self::$settings['jptgb_setting_critical_css_activate'] )     && self::$settings['jptgb_setting_critical_css_activate'] )     ? '1' : '';
        $is_load_scripts_on_fui         = ( isset( self::$settings['jptgb_setting_load_scripts_on_fui'] )       && self::$settings['jptgb_setting_load_scripts_on_fui'] )       ? '1' : '';
        $idl_for_hidden_elements        = ( isset( self::$settings['jptgb_setting_idl_for_hidden_elements'] )  && self::$settings['jptgb_setting_idl_for_hidden_elements'] )    ? '1' : '';

        $options = [
            'ldl_height'                        => $is_mobile ? 900 : 1300,
            'isIdl'                             => $is_idl,
            'isLdl'                             => $is_ldl,
            'ajaxUrl'                           => $ajax_url,
            'isCriticalCss'                     => $is_cc,
            'isLoadScriptsOnFui'                => $is_load_scripts_on_fui,
            'isLoadIframesOnFui'                => $load_iframes_on_fui,
            'isGenerateIframesBg'               => $generate_iframes_bg,
            'ldlForHiddenElements'              => $idl_for_hidden_elements,
            'stylesToExcludeFromFui'            => $exclude_styles,
            'scriptsToExcludeFromFui'           => $exclude_scripts,
            'scriptsToAllowToExecuteOnPupetter' => $exclude_deactivation_scripts,
            'generateMobileCache'               => $is_mobile_cache,
            'bypassCacheLayers'                 => $is_bypass_cache_layers,
            'isDebug'                           => $is_debug,
        ];

        return $options;
    }

    public static function get_has_redirect_transient( $id ) {
        return get_transient( 'jptgb_has_redirect_' . $id );
    }

    public static function set_has_redirect_transient( $id, $value, $expires ) {
        /**
         * Set the transient.
         */
        set_transient( 'jptgb_has_redirect_' . $id, $value, $expires );

        /**
         * Add this transient to the jptgb_has_redirect list.
         */
        $list       = get_option( 'jptgb_has_redirect_list', [] );
        $list[$id]  = [ 'value' => $value, 'expires' => $expires ];

        /**
         * Update the list.
         */
        update_option( 'jptgb_has_redirect_list', $list );
    }

    

    public static function create_cache_file( $page_content, $current_url, $cache_url_path ) {
        try {
            //code...
            /**
             * Try to create the the cache url path.
             */
            $response = wp_mkdir_p( $cache_url_path );
    
            /**
             * Return if we failed creating the cache file path.
             */
            if( !$response ){
                return new WP_Error( 'jptgb_error', "Unable to create the cache file: {$cache_url_path}" );
            }
    
            /**
             * Ensure the integrety of the content, if cache 
             * content do not contains </html>, then just return.
             */
            if ( strpos( $page_content, '</html>' ) === false ) {
                return $page_content;
            }
            
            /**
             * Modify the cache page.
             */
            $content_for_cache_file = self::insert_canonical_tag( $page_content, $current_url );
    
            return $content_for_cache_file;
        } catch (\Throwable $th) {
            $error_message = $th->getMessage();
            BaseController::$logs->register( 'Something failed while trying create the cache file | Error message: ' . $error_message, 'error' );
            
            throw $th;
        }
    }

    /**
     * Insert a canonical URL tag into the HTML content if not already present.
     *
     * @param string $html_content The HTML content of the page.
     * @param string $canonical_url The URL to set as the canonical link.
     * @return string The HTML content with the canonical link added, if necessary.
     */
    public static function insert_canonical_tag( $html_content, $canonical_url ) {
        /**
         * Check if a canonical tag already exists (either with single or double quotes).
         */
        if ( strpos( $html_content, 'rel="canonical"' ) === false && strpos( $html_content, "rel='canonical'" ) === false ) {
            /**
             * Find the closing </head> tag.
             */
            $head_pos = strpos( $html_content, '</head>' );
            if ( $head_pos !== false ) {
                /**
                 * Prepare the canonical tag.
                 */
                $canonical_tag = '<link rel="canonical" href="' . esc_url( $canonical_url ) . '"/>' . "\n";

                /**
                 * Insert the canonical tag before the closing </head> tag.
                 */
                $html_content = substr_replace( $html_content, $canonical_tag, $head_pos, 0 );
            }
        }

        return $html_content;
    }
    
    /**
     * Get the path to the htaccess file either custom or defautl.
     * 
     * @return string Absolute path to the intended .htaccess file.
     */
    public static function get_htaccess_path(): string {
        /**
         * Return custom path if it is set and the file exist.
         */
        $use_custom_htaccess_path   = self::get_setting( 'use_custom_htaccess_path' );
        $custom_htaccess_path       = self::get_setting( 'custom_htaccess_path' );
        if( $use_custom_htaccess_path && $custom_htaccess_path ) {
            return $custom_htaccess_path;
        }

        return self::$webroot_path . '.htaccess';
    }
}