<?php

/**
 * The Cache Controller Class
 *
 * @author Yaidier Perez
 * */

namespace Jptgb\Controllers;

use Jptgb\Controllers\CacheController;
use Sabberworm\CSS\Parser;
use Sabberworm\CSS\RuleSet\DeclarationBlock;
use Sabberworm\CSS\CSSList\Document;
use Jptgb\Resources\Utils;

class CriticalCssController extends BaseController {
    /**
     * Define calss properties.
     */
    public static $cc_data  = [];
    public static $cc_flush = 0;

    public static function init() {
        /**
         * If we just turn on the critical css,
         * and the cache was active then lets
         * clear up the cache then.
         */
        if( in_array( 'jptgb_setting_critical_css_activate', self::$changed_settings ) ) {
            CacheController::flush_all_cache_files();
        }
    }

    public static function flush_all_critical_css() {
        /**
         * Get the cache directory path.
         */
        $cc_base_dir_path = self::get_critical_css_directory(); 

        /**
         * Remove all critical files recursively.
         */
        if( is_dir( $cc_base_dir_path ) ) {
            Utils::remove_directory_recursively( $cc_base_dir_path );
        }

        /**
         * Get the critical css records. (!IMPORTANT DEPRECATED)
         */
        $cc_records = get_option( 'jptgb_cc_records', [] );

        /**
         * Itereate over each record. (!IMPORTANT DEPRECATED)
         */
        foreach( $cc_records as $id => $value ){
            /**
             * Get the critical css data.
             */
            $cc_data = get_option( $id, false );

            /**
             * Continue if there is no data set.
             */
            if( false == $cc_data ){
                continue;
            }

            /**
             * Update the critical css data.
             */
            delete_option( $id );
        }
    }

    /**
     * Removes @font-face declarations from a CSS string, excluding those containing specified keywords.
     *
     * @param string $css The CSS content as a string.
     * @return string The CSS content with specific @font-face declarations removed.
     */
    public static function remove_font_face_statements( $css ) {
        /**
         * Define font face keyworks to exclude.
         */
        $excludeKeywords = [];
        
        if( !self::$settings['jptgb_setting_load_fa_icons_on_fui'] ){
            $excludeKeywords[] = 'FontAwesome';
            $excludeKeywords[] = 'Font Awesome';
            $excludeKeywords[] = 'awb-icons';
        }

        /**
         * Initial font face position.
         */
        $fontFaceStart = strpos($css, '@font-face');

        while ($fontFaceStart !== false) {
            /**
             * Find the opening curly brace of the @font-face declaration.
             */
            $curlyStart = strpos($css, '{', $fontFaceStart);
            if ($curlyStart === false) {
                /**
                 * If no opening curly brace is found, break out of the loop.
                 */
                break;
            }

            /**
             * Find the closing curly brace of the @font-face declaration.
             */
            $curlyEnd = strpos($css, '}', $curlyStart);
            if ($curlyEnd === false) {
                /**
                 * If no closing curly brace is found, break out of the loop.
                 */
                break;
            }

            /**
             * Extract the @font-face declaration.
             */
            $fontFaceDeclaration = substr($css, $fontFaceStart, $curlyEnd - $fontFaceStart + 1);

            /**
             * Check if the current @font-face declaration contains any of the exclude keywords.
             */
            $excludeThisFontFace = false;
            foreach ($excludeKeywords as $keyword) {
                if (strpos($fontFaceDeclaration, $keyword) !== false) {
                    $excludeThisFontFace = true;
                    break;
                }
            }

            if ($excludeThisFontFace) {
                /**
                 * Move the start position past this @font-face declaration to continue the search.
                 */
                $fontFaceStart = strpos($css, '@font-face', $curlyEnd);
                continue; // In a while loop, this will just proceed to the next iteration of the loop
            }

            /**
             * Remove the @font-face declaration from CSS.
             */
            $css = str_replace($fontFaceDeclaration, '', $css);

            /**
             * Look for the next @font-face declaration starting from the previous start position.
             */
            $fontFaceStart = strpos($css, '@font-face', $fontFaceStart);
        }

        return $css;
    }

    /**
     * Removes @import statements from a CSS string.
     *
     * @param string $css The CSS content as a string.
     * @return string The CSS content with @import statements removed.
     */
    public static function remove_import_statements($css) {
        $importStart = strpos($css, '@import');
        while ($importStart !== false) {
            // Find the end of the URL in the import statement (closing parenthesis)
            $urlEnd = strpos($css, ')', $importStart);
            if ($urlEnd === false) {
                // If no closing parenthesis is found, break out of the loop
                break;
            }

            // Find the end of the import statement (semicolon after the URL end)
            $importEnd = strpos($css, ';', $urlEnd);
            if ($importEnd === false) {
                // If no semicolon is found, break out of the loop
                break;
            }

            // Extract the @import statement
            $importStatement = substr($css, $importStart, $importEnd - $importStart + 1);

            // Remove the @import statement from CSS
            $css = str_replace($importStatement, '', $css);

            // Look for the next @import statement
            $importStart = strpos($css, '@import');
        }
        return $css;
    }

    public static function update_critical_data( $url_hash, $cc_data ){
        $critical_data = get_option( 'jptgb_critical_data', [] );
        $critical_data[ 'jptgb_critical_css_' . $url_hash ] = $cc_data;

        return update_option( 'jptgb_critical_data', $critical_data );
    }

    public static function get_critical_data( $url_hash ){
        $critical_data = get_option( 'jptgb_critical_data', [] );

        return $critical_data[ 'jptgb_critical_css_' . $url_hash ] ?? [];
    }

    private static function get_rulsets_to_include() {
        $key_sentences          = self::get_setting( 'include_rulesets_in_critical' );
        $key_sentences_array    = Utils::textarea_content_to_array__breaklines( $key_sentences );

        return $key_sentences_array;
    }

    /**
     ** Build a non‑blocking preload <link> tag + <noscript> fallback.
     **
     ** @param array $attributes Original <link> attributes (rel, id, href, type, media, etc).
     ** @return string           The HTML for the deferred tag + fallback.
     */
    protected static function build_deferred_link_tag( array $attributes ): string {
        // 1) Copy over every original attr except rel, type, media
        $filtered = array_filter(
            $attributes,
            function( $value, $key ) {
                return ! in_array( $key, [ 'rel', 'type', 'media' ], true );
            },
            ARRAY_FILTER_USE_BOTH
        );

        // 2) Start building our <link> tag
        $html = '<link';
        foreach ( $filtered as $name => $value ) {
            $html .= sprintf(
                ' %1$s="%2$s"',
                esc_attr( $name ),
                esc_attr( $value )
            );
        }

        // 3) Append non‑blocking attributes
        $html .= ' rel="preload" as="style"';
        $html .= ' onload="this.onload=null;this.rel=\'stylesheet\';this.removeAttribute(\'as\')"';
        $html .= '>' . "\n";

        // 4) Add <noscript> fallback
        $id   = isset( $attributes['id'] )   ? esc_attr( $attributes['id'] )   : '';
        $href = isset( $attributes['href'] ) ? esc_url(   $attributes['href'] ) : '';

        $html .= sprintf(
            '<noscript><link%s rel="stylesheet" href="%s"></noscript>',
            $id   ? ' id="' . $id . '"' : '',
            $href
        );

        return $html;
    }

    /**
     * IMPORTANT!!! 
     * 
     * @todo The $css_bundle_hash is always returning different values therefore we are generating critical css everytime 
     * that a cache file is requested to generate, the issue arises because we are generating the bundle css in the api in the
     * generateCacheController using pupetteer coverage built in function, whichs extract the css in random order, so is pending 
     * to either improve the way the bundle css gets extracted in the api or to extract it from wp site as we initially use to do.
     */
    public static function critical_css_page_content( $cache_file_content, $html_with_scripts_disabled, $requested_url, $url_hash, $css_tags_data, $device, $critical_css_content = '' ) {  
        try {
            /**
             * Initally set the api call to false.
             */
            $cc_api_call = false;

            /**
             * Get critical css data.
             */
            $cc_data            = self::get_critical_data( $url_hash );
            $last_bundle_hash   = $cc_data['bundle_hash_' . $device] ?? false; 
    
            /**
             * IMPORTANT!!! There is an issue with the critical cache. 
             * Check before consider to activate it again. Otherwise 
             * let it set to true, so we regenerate critical every time
             * we request a cache of a page.
             */
            $do_flush = $cc_data['flush_' . $device] ?? true;
               
            /**
             * Define tha base url and paths.
             */
            $cc_dir_data        = self::get_critical_css_path_and_url();
            $cc_base_dir_path   = $cc_dir_data['path'];
            $cc_base_dir_url    = $cc_dir_data['url'];
    
            /**
             * Get the critical css file path.
             */
            $critical_css_path  = $cc_base_dir_path . "/critical-css-{$url_hash}-{$device}.css";
    
            /**
             * Get the local file.
             */
            $local_critical_css_content = Utils::read_file( $critical_css_path );
    
            /**
            * Bundle all css into a single file.
            */ 
            $bundle_css = self::get_bundle_css( $css_tags_data, $cache_file_content, $requested_url );
    
            /**
             * Get the current bundle hash.
             */
            $css_bundle_hash = md5( $bundle_css );
    
            /**
             * If we can't find a local file with the content
             * or the file is empty, then lets command the api 
             * to regenerate the critical css by sending an empty
             * hash.
             */
            if( !$local_critical_css_content || $do_flush || ( $last_bundle_hash !== $css_bundle_hash ) ){
                /**
                 * Get the critical css rulesets to include.
                 */
                $rulsets_to_include = self::get_rulsets_to_include();
    
                /**
                 * Check if we have to load the fonts on FUI.
                 */
                if( !self::get_setting( 'load_fonts_on_fui' ) ) {
                    $font_faces_data = Utils::extract_font_faces( $bundle_css );
                    // BaseController::$logs->register( 'Font faces extracted for url ' . $requested_url . ' | Data: ' . wp_json_encode( $font_faces_data, JSON_PRETTY_PRINT ), 'info' );
                } 
    
                /**
                 * Remove @import statements from the bundle css.
                 */
                $bundle_css = self::remove_import_statements( $bundle_css );
        
                /**
                 * Remove @font-face statements from the bundle css.
                 */
                $bundle_css = self::remove_font_face_statements( $bundle_css );
    
                /**
                 * Penthouse do not recognize the ":-" tailwind selectors used i.e before:-top-3,
                 * that is why we need to temporally replace them.
                 */
                // $bundle_css            = str_replace( '\:-', 'COLONDASHCOLONDASH', $bundle_css );
                // $modified_page_content = str_replace( ':-', 'COLONDASHCOLONDASH', $html_with_scripts_disabled );
                $modified_page_content = $html_with_scripts_disabled;

                /**
                 * Inject the bundle css in the html.
                 */
                $modified_page_content = str_replace( '<!-- JPTGBBUNDLECSSSTART -->', '<!-- JPTGBBUNDLECSSSTART --><style>' . $bundle_css . '</style>', $modified_page_content );
        
                /**
                 * Define the file urls and paths.
                 */
                $target_html_url    = $cc_base_dir_url  . "/tmp-html-{$url_hash}.html";
                $target_html_path   = $cc_base_dir_path . "/tmp-html-{$url_hash}.html";
                $bundle_css_url     = $cc_base_dir_url  . "/tmp-all-css-{$url_hash}.css";
                $bundle_css_path    = $cc_base_dir_path . "/tmp-all-css-{$url_hash}.css";
    
                /**
                 * Write local files.
                 */
                Utils::write_file( $target_html_path, $modified_page_content );
                Utils::write_file( $bundle_css_path, $bundle_css );
    
                /**
                 * Call the api.
                 */
                if( !$critical_css_content ){
                    $cc_api_call            = true;
                    $critical_css_content   = self::request_critical_css( $requested_url, $target_html_url, $bundle_css_url, $url_hash, $device );
                }
    
                /**
                 * Return if response is not valid.
                 */
                if( !$critical_css_content ){
                    /**
                     * Return 
                     */
                    return [ 
                        'cc_api_call'           => $cc_api_call,
                        'cache_file_content'    => $cache_file_content 
                    ];
                }
    
                /**
                 * Get the critical css.
                 */
                // $critical_css_content = str_replace( 'COLONDASHCOLONDASH', '\:-', $critical_css_content );
                
                /**
                 * Decide to include font-faces or not
                 */
                if( !empty( $font_faces_data ) ){
                    $atf_fonts_data = Utils::extract_fonts_data_from_html( $cache_file_content );
                    $critical_css_content = self::maybe_prepend_font_faces( $critical_css_content, $font_faces_data, $atf_fonts_data );
    
                }
        
                /**
                 * Store the new css_bundle_hash
                 */
                $cc_data['bundle_hash_' . $device] = $css_bundle_hash;
    
            } else {
                $critical_css_content = $local_critical_css_content;
    
            }
    
            /**
             * Command the styles to load on FUI.
             */
            foreach( $css_tags_data as $css_tag ){
                $style_id   = $css_tag['styleId'];
                $attributes = $css_tag['attributes'];
    
                /**
                 * Extract the tag from the cache file content.
                 */
                $tag_content = self::get_wrapped_content_by_id( $cache_file_content, $style_id );
    
                // if we should defer this stylesheet
                if ( !empty( $css_tag['excludeCss'] ) ) {
                    /**
                     * Target placeholder.
                     */
                    $target_placeholder = '<!--JPTGB-SCRIPT-'. $style_id .'-->';
    
                    /**
                     * Generate our deferred markup.
                     */
                    if( !empty( $css_tag['shouldDefer'] ) && 'link' === ( $css_tag['tag'] ?? '' ) ){
                        $tag_content = self::build_deferred_link_tag( $attributes );
                    }
    
                    /**
                     * replace the old tag in the HTML with the deferred version.
                     */
                    $cache_file_content = str_replace( $target_placeholder, $tag_content, $cache_file_content );
                }
                else {
                    // normal FUI loading
                    self::load_css_tag_on_fui( $tag_content, $attributes );
                }
            }
    
            /**
             * Remove the css tags and css data from the cache file content.
             */
            $cache_file_content = self::remove_all_wrapped_content_general( $cache_file_content );
    
            /**
             * Store the critical css in local file.
             */
            Utils::write_file( $critical_css_path, $critical_css_content );
    
            /**
             * Update the info in the critical css data.
             */
            $cc_data[ 'flush_'          . $device ] = false;
            $cc_data[ 'cc_path_'        . $device ] = $critical_css_path;
            $cc_data[ 'bundle_hash_'    . $device ] = $css_bundle_hash;
    
            /**
             * Update critical css data.
             */
            self::update_critical_data( $url_hash, $cc_data );
    
            /**
             * Append custom css content.
             */
            $critical_css_content .= BaseController::get_setting( 'jptgb_setting_custom_css_in_critical' );
    
            /**
             * Inline the critical css in the cache file.
             */
            $cache_file_content = str_replace( '</head>', '<style id="jptgb-critical-css" data-ccflush="0">' . $critical_css_content . '</style></head>', $cache_file_content );
    
            /**
             * Return the page content.
             */
            return [
                'cc_api_call'           => $cc_api_call,
                'cache_file_content'    => $cache_file_content 
            ];
        } catch (\Throwable $th) {
            $error_message = $th->getMessage();
            BaseController::$logs->register( 'Something failed while trying to generate the critical css | Error message: ' . $error_message, 'error' );
            
            throw $th;
        }
    }

    /**
     * Conditionally prepend any required @font-face rulesets into the critical CSS,
     * but only once per weight–style (e.g. “400-normal”, “400-italic”).
     *
     * @param string $critical_css_content The critical‐only CSS string.
     * @param array<string,array{rulesets:array<string,string>}> $font_faces_data
     *   Associative array of font families to their rulesets, keyed by weight-style:
     *   [
     *     'Barlow' => [
     *       'rulesets' => [
     *         '400-normal' => '@font-face{…}',
     *         '400-italic' => '@font-face{…}',
     *         …
     *       ],
     *     ],
     *     …
     *   ]
     * @return string The critical CSS with necessary @font-face blocks prepended.
     */
    public static function maybe_prepend_font_faces( string $critical_css_content, array $font_faces_data, array $atf_fonts_data ): string {
        // Bail early if there are no font-face definitions
        if ( empty( $font_faces_data ) ) {
            return $critical_css_content;
        }

        $atf_fonts_families = array_keys( $atf_fonts_data );

        // Keep track of which weight-style keys have already been inserted
        $inserted = [];
        $inserted_rulset = [];

        foreach ( $font_faces_data as $font_family => $font_face_data ) {
            // BaseController::$logs->register( 'Including Font Family in the critical css: ' . $font_family, 'info' );
            /**
             * Skip this family if it’s not referenced in the critical CSS at all.
             * 
             * @todo Look not only for the font family name, but also font-wieght, type, etc
             * on that way we narrow down the fonts inserted to thouse only solely in the 
             * above the fold section. Maybe we can extract the information about what
             * fonts, weight, types are used in above the fold in the API...
             */
            if ( false === stripos( $critical_css_content, $font_family ) ) {
                // BaseController::$logs->register( 'Skipt ' . $font_family . ' is not present in the critical css', 'info' );
                continue;
            }

            /**
             * Continue if the current font family is not being used in the above the fold content.
             */
            if( !in_array( $font_family, $atf_fonts_families ) ){
                continue;
            }

            // Make sure we have a proper 'rulesets' array
            if ( empty( $font_face_data['rulesets'] ) || ! is_array( $font_face_data['rulesets'] ) ) {
                // BaseController::$logs->register( 'Skipt ' . $font_family . ' has empty rulsets', 'info' );
                continue;
            }

            $atf_styles = array_values( $atf_fonts_data[ $font_family ]['styles'] ?? [] );

            /**
             * Each ruleset is keyed by its weight-style, e.g. '400-normal'.
             * By looping with the key in hand we can prevent duplicates.
             */
            foreach ( $font_face_data['rulesets'] as $weight_style => $ruleset ) {   
                 
                // Skip invalid or empty rulesets
                if ( ! is_string( $ruleset ) || '' === trim( $ruleset ) ) {
                    // BaseController::$logs->register( 'Skipt ' . $font_family . ' ' . $weight_style .' is empty', 'info' );
                    continue;
                }

                /**
                 * Continue if the current weight - style is not being used in the above the fold content.
                 */
                if( !in_array( $weight_style, $atf_styles ) ){
                    continue;
                }

                // If we’ve already added this exact weight-style, skip it
                if ( isset( $inserted[ $font_family . ' - ' . $weight_style ] ) ) {
                    // BaseController::$logs->register( 'Skipt ' . $font_family . ' ' . $weight_style .' was already inserted', 'info' );
                    continue;
                }

                $rulset_hash = md5( (string) $ruleset );

                if( isset( $inserted_rulset[$rulset_hash] ) ){
                    continue;
                }

                $inserted_rulset[$rulset_hash] = true;

                // Mark it as inserted and prepend its ruleset
                $inserted[ $font_family . ' - ' . $weight_style ] = true;
                $critical_css_content = Utils::insert_font_face_into_critical_css(
                    $critical_css_content,
                    $ruleset
                );

                // BaseController::$logs->register( $font_family . ' ' . $weight_style .' successfully inserted', 'info' );
            }
        }

        return $critical_css_content;
    }

    /**
     * Removes all instances of content wrapped between custom start and end comment markers,
     * regardless of the presence of an ID.
     *
     * @param string $html The full HTML document as a string.
     * @return string The HTML content after all specified wrapped contents have been removed.
     */
    private static function remove_all_wrapped_content_general( string $html ): string {
        // Pattern to match and remove any content between the specified start and end comments
        $pattern = "/<!--JPTGBREMOVESTYLESTART-[\s\S]*?-JPTGBREMOVESTYLEEND-->/";

        // Perform the regex replacement to remove matched content
        $updatedHtml = preg_replace($pattern, '', $html);

        // Return the updated HTML content
        return $updatedHtml;
    }

    /**
     * Removes all content enclosed between the specific start and end comment tags for CSS data.
     *
     * @param string $html The full HTML document as a string.
     * @return string The HTML content after all specified wrapped contents have been removed.
     */
    private static function remove_css_data_content( string $html ): string {
        // Pattern to match and remove any content between the specified start and end comments for CSS data
        $pattern = "/JPTGB-CSS-DATA-START-[\s\S]*?-JPTGB-CSS-DATA-END/";

        // Perform the regex replacement to remove matched content
        $updatedHtml = preg_replace($pattern, '', $html);

        // Return the updated HTML content
        return $updatedHtml;
    }

    /**
     * Extracts the content wrapped within specified start and end comments by ID.
     *
     * @param string $html The full HTML document as a string.
     * @param string $id The unique identifier used within the comment markers.
     * @return string|null The extracted content or null if not found.
     */
    private static function get_wrapped_content_by_id( string $html, string $id ): ?string {
        // Pattern to match the wrapped content based on the provided ID
        $pattern = "/<!--JPTGBREMOVESTYLESTART-$id-->(.*?)<!--JPTGBREMOVESTYLEEND-->/s";

        // Perform the regex search
        if ( preg_match( $pattern, $html, $matches ) ) {
            // If a match is found, return the content (without the comment markers)
            return $matches[1];
        }

        // Return null if no content is found
        return null;
    }

    public static function load_css_tag_on_fui( $css_tag, $attributes ) {
        if( strpos( $css_tag, '<style' ) !== false ){
            $style_tag_data = self::extract_data_from_style_tag( $css_tag );
            $content        = $style_tag_data['content'];

        } else if ( strpos( $css_tag, '<link' ) !== false ) {
            $content    = '';
        }

        $order = isset( $attributes['data-jptgb-order'] ) ? (int) $attributes['data-jptgb-order'] : 99999;
    
        self::$scripts_loader_list[] = [
            'handle'            => false,
            'dependencies'      => [],
            'attributes'        => $attributes,
            'content'           => $content,
            'type'              => 'style',
            'in_footer'         => false,
            'order'             => $order
        ];
    }   

    private static function get_css_content_to_insert_manually( $css_contents, $bundle_array ){
        $css_properties_to_add_manually = '';

        foreach( $css_contents as $index => $css_data ){
            if( !isset( $css_data['add_manually'] ) || !$css_data['add_manually'] ){
                continue;
            }

            $css_properties_to_add_manually .= $bundle_array[$index] ?? '';
        }

        return $css_properties_to_add_manually;
    }

    private static function extract_nested_rulesets( $bundle_array ) {
        /**
         * Define the rulesets string.
         */
        $special_rulesets = '';
        
        /**
         * Iterate over each css content in the bundle array.
         */
        foreach ( $bundle_array as $css_rules ) {
            /**
             * Continue if the css content do not contains nested rulsets.
             */
            if( !self::hasNestedBlocks( $css_rules ) ){
                continue;
            }
            
            /**
             * Append the rulesets.
             */
            $special_rulesets .= $css_rules;
        }

        return $special_rulesets;
    }

    /**
     * Checks if a CSS content string contains nested blocks, excluding @media and other at-rules.
     *
     * @param string $cssContent CSS content as a string.
     * @return bool Returns true if nested blocks are found, false otherwise.
     */
    private static function hasNestedBlocks(string $cssContent): bool {
        // Remove CSS comments to simplify parsing
        $cssContent = preg_replace('!/\*.*?\*/!s', '', $cssContent);

        // Remove at-rules including nested blocks inside them
        $cssContent = preg_replace('/@[^{]*\{(?:[^{}]*\{[^{}]*\})*[^{}]*\}/', '', $cssContent);

        // Look for nested blocks using simple string manipulation
        $braceLevel = 0;  // Track depth of nesting
        $length = strlen($cssContent);
        for ($i = 0; $i < $length; $i++) {
            $char = $cssContent[$i];
            if ($char === '{') {
                $braceLevel++;
                if ($braceLevel > 1) {
                    // More than one level of braces means there's nesting
                    return true;
                }
            } elseif ($char === '}') {
                $braceLevel--;
            }
        }

        return false;
    } 


    private static function extract_data_from_style_tag( $inline_style ) {
        /**
         * Extract attributes and CSS content from the <style> tag.
         */
        $styleStart         = strpos($inline_style, '<style');
        $styleEndTagStart   = strpos($inline_style, '>', $styleStart) + 1;
        $styleEnd           = strpos($inline_style, '</style>', $styleEndTagStart);
    
        /**
         * Extract attributes from the style tag.
         */
        $attributesString   = substr($inline_style, $styleStart, $styleEndTagStart - $styleStart);
        $attributesString   = trim(str_replace(['<style', '>'], '', $attributesString));
        $attributes         = [];
        preg_match_all('/([a-zA-Z-]+)\s*=\s*(?:\"([^"]*)\"|\'([^\']*)\')/', $attributesString, $matches, PREG_SET_ORDER);
        foreach ($matches as $match) {
            $attributes[$match[1]] = $match[2] ?? $match[3];
        }

        /**
         * Extract CSS content.
         */
        $cssContent = substr($inline_style, $styleEndTagStart, $styleEnd - $styleEndTagStart);
        $cssContent = trim($cssContent);

        return ['attributes' => $attributes, 'content' => $cssContent];
    }

    /**
     * Extracts all attributes from a given <link> tag using string manipulation.
     *
     * @param string $link_tag The <link> tag from which to extract attributes.
     * @return array An associative array of attributes and their values.
     */
    private static function extract_data_from_link_tag($link_tag) {
        $attributes = [];
        $tagContent = trim($link_tag);

        // Remove the starting '<link ' and the ending '>' or '/>' from the tag
        $start = strpos($tagContent, ' ') + 1;
        $end = strrpos($tagContent, '>');
        $tagContent = trim(substr($tagContent, $start, $end - $start - (substr($tagContent, -2) == '/>' ? 1 : 0)));

        while ($tagContent !== '') {
            // Find the position of the first "=" (attribute=value)
            $equalPos = strpos($tagContent, '=');
            if ($equalPos === false) {
                // No more attributes to process
                break;
            }

            $attrName = trim(substr($tagContent, 0, $equalPos));

            // Move past the "=" symbol
            $tagContent = ltrim(substr($tagContent, $equalPos + 1));

            // Check what the attribute value is wrapped in (single or double quotes)
            $quote = $tagContent[0];
            if ($quote !== '"' && $quote !== "'") {
                // Attribute value is not quoted properly
                break;
            }

            // Find the end of the attribute value
            $quotePos = strpos($tagContent, $quote, 1);
            if ($quotePos === false) {
                // Ending quote not found
                break;
            }

            $attrValue = substr($tagContent, 1, $quotePos - 1);

            // Add attribute to the array
            $attributes[$attrName] = $attrValue;

            // Move to the next attribute, trimming any leading whitespace or ending tags
            $tagContent = ltrim(substr($tagContent, $quotePos + 1), " />\t\n\r\0\x0B");
        }

        return $attributes;
    }

    public static function load_inline_style_on_fui( $handle, $inline_style, $in_footer ) {
        if( strpos( $inline_style, '<style' ) !== false ){
            $style_tag_data = self::extract_data_from_style_tag( $inline_style );
            $attributes     = $style_tag_data['attributes'];
            $content        = $style_tag_data['content'];

        } else if ( strpos( $inline_style, '<link' ) !== false ) {
            $attributes = self::extract_data_from_link_tag( $inline_style );
            $content    = '';
        }


        $order = isset( $attributes['data-jptgb-order'] ) ? (int) $attributes['data-jptgb-order'] : 99999;
    
        self::$scripts_loader_list[] = [
            'handle'            => $handle,
            'dependencies'      => [],
            'attributes'        => $attributes,
            'content'           => $content,
            'type'              => 'style',
            'in_footer'         => $in_footer,
            'order'             => $order
        ];
    }    

    /**
     * Extracts all the stylesheet link tags from an HTML document and indicates their location (head or footer).
     *
     * @param string $html The HTML document as a string.
     * @return array An array of arrays, each containing the stylesheet link tag and its location.
     */
    public static function extract_stylesheet_tags_data( $html ) {
        $stylesheet_links = array();
        $start = 0;

        // Find the position of the opening <body> tag
        $body_start = strpos( $html, '<body' );

        while ( ( $start = strpos( $html, '<link', $start ) ) !== false ) {
            $end_tag = strpos( $html, '>', $start );
            if ( false === $end_tag ) {
                break;
            }

            $link_tag = substr( $html, $start, $end_tag - $start + 1 );

            // Check if the link tag has rel="stylesheet"
            if ( false !== strpos( $link_tag, 'rel="stylesheet"' ) || false !== strpos( $link_tag, "rel='stylesheet'" ) ) {
                // Determine the location of the tag based on its position relative to <body>
                $location = ( $start > $body_start && false !== $body_start ) ? 'footer' : 'head';
                $stylesheet_links[] = array(
                    'tag'       => $link_tag,
                    'location'  => $location
                );
            }

            $start = $end_tag + 1;
        }

        return $stylesheet_links;
    }

    /**
     * Request critical CSS from API.
     *
     * Sends a request to the API endpoint to generate critical CSS.
     * 
     * @param string $css_url     The URL of the CSS file.
     * @param string $target_url  The target URL for which critical CSS is to be generated.
     *
     * @return string|null The URL of the generated critical CSS file or null on failure.
     */
    private static function request_critical_css( $requested_url, $target_html_url, $bundle_css_url, $url_hash, $device ) {
        /**
         * Check if we request the critical css to the api or self hosted script.
         */
        if( !self::get_setting( 'activate_self_hosted' ) ){
            return self::request_critical_css_to_api( $requested_url, $target_html_url, $bundle_css_url, $url_hash, $device );
        }

        return self::request_self_hosted_critical_css( $requested_url, $target_html_url, $bundle_css_url, $device );
    }

    private static function request_self_hosted_critical_css( $requested_url, $target_html_url, $bundle_css_url, $device ){
        /**
         * Get chromium custom path if any.
         */
        $chromium_path = self::get_setting( 'path_penthouse_chromium' );

        /**
         * Define arguments to send.
         */
        $cache_contents_data = [
            'requestedUrl'  => $requested_url,
            'targetHtmlUrl' => $target_html_url,
            'bundleCssUrl'  => $bundle_css_url,
            'chromiumPath'  => $chromium_path,
            'device'        => $device,
        ];

        /** 
         * Encode the data.
         */
        $args = wp_json_encode( $cache_contents_data );

        /**
         * Escape the JSON string for use in the shell command.
         */
        $escaped_args = escapeshellarg( $args );

        /**
         * Command to execute the Node.js script with the JSON argument.
         */
        $command = 'node ' . JPTGB_PATH . 'src/js/self-hosted/api-controllers-interface.js generateCriticalCss ' . $escaped_args;

        /**
         * Execute the command and capture the output.
         */
        $output = shell_exec( $command );

        /**
         * Capture the error message.
         */
        if( $output === null ){
            /**
             * Redirect stderr to stdout.
             */
            $command .= ' 2>&1';

            /**
             * Get the error message.
             */
            $error_message = shell_exec( $command );

            /**
             * Report the incident...
             */
            BaseController::$logs->register( 'Self Critical Css generation failid for url: ' . esc_url( $error_message ), 'error' );

            return false;
        }

        /**
         * Log the situation.
         */
        BaseController::$logs->register( 'Self Critical Css Generation requested for url: ' . esc_url( $requested_url ), 'info' );

        return $output;
    } 

    private static function request_critical_css_to_api( $requested_url, $target_html_url, $bundle_css_url, $url_hash, $device ) {
        /** 
         * API endpoint for generating critical CSS.
         */
        $api_endpoint = self::$webspeed_api_base . 'generate-critical-css';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /** 
         * Payload to be sent to the API.
         */
        $body = wp_json_encode( [
            'email'                 => self::get_setting( 'user_email' ),
            'isDebug'               => self::get_setting( 'debug_activate' ),
            'generateMobileCache'   => self::get_setting( 'mobile_cache' ),
            'notEndpoint'           => BaseController::get_endpoint_absolute_url(),
            'requestedUrl'          => $requested_url,
            'targetHtmlUrl'         => $target_html_url,
            'bundleCssUrl'          => $bundle_css_url,
            'urlHash'               => $url_hash,
            'device'                => $device, 
            'apiKey'                => $api_key,
        ] );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting critical CSS: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( 200 !== $status_code ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }

        /**
         * Get the body response.
         */
        $body = wp_remote_retrieve_body( $response );
        
        /**
         * Return the received data.
         */
        return $body;
    }

    /**
     * Creates a 'critical-css' directory inside the WordPress 'uploads' directory and returns its path and URL.
     * Adheres to WordPress coding standards.
     *
     * @return array|bool An associative array with 'path' and 'url' of the critical-css directory, or false on failure.
     */
    public static function get_critical_css_directory() {
        /**
         * Get the path and URL to the WordPress 'uploads' directory.
         */
        $upload_dir_info   = wp_upload_dir();
        $upload_path       = $upload_dir_info['basedir'];

        /** 
         * Define the path and URL for the 'critical-css' directory.
         */
        $critical_css_dir  = $upload_path . '/jptgb/critical-css';

        /**
         * Check if the directory already exists.
         */
        if( ! Utils::does_file_exists( $critical_css_dir ) ) {
            /**
             * Attempt to create the directory.
             */
            if ( !Utils::create_directory_recursively( $critical_css_dir, 0755 ) ) {
                /**
                 * Handle the error, e.g., log or notify.
                 */
                self::error_log( 'Failed to create critical-css directory.' );
                return false;
            }
        }

        return $critical_css_dir;
    }

    /**
     * Creates a 'critical-css' directory inside the WordPress 'uploads' directory and returns its path and URL.
     * Adheres to WordPress coding standards.
     *
     * @return array|bool An associative array with 'path' and 'url' of the critical-css directory, or false on failure.
     */
    public static function get_critical_css_path_and_url() {
        /**
         * Get the path and URL to the WordPress 'uploads' directory.
         */
        $upload_dir_info   = wp_upload_dir();
        $upload_path       = $upload_dir_info['basedir'];
        $upload_url        = $upload_dir_info['baseurl'];

        /** 
         * Define the path and URL for the 'critical-css' directory.
         */
        $critical_css_dir  = $upload_path . '/jptgb/critical-css';
        $critical_css_url  = $upload_url . '/jptgb/critical-css';

        /**
         * Check if the directory already exists.
         */
        if( !Utils::does_file_exists( $critical_css_dir ) ) {
            /**
             * Attempt to create the directory.
             */
            if( !Utils::create_directory_recursively( $critical_css_dir, 0755 ) ) {
                /**
                 * Handle the error, e.g., log or notify.
                 */
                self::error_log( 'Failed to create critical-css directory.' );
                return false;
            }
        }

        return array(
            'path' => $critical_css_dir,
            'url'  => $critical_css_url
        );
    }

    /**
     * Get the body of a remote URL.
     *
     * Tries WP_HTTP first (and logs any WP_Error), then falls back to
     * file_get_contents(), relaxing SSL / switching to http in non-prod.
     *
     * @param  string           $url  The URL to fetch.
     */
    public static function get_file_contents( string $url ) {
        // --- 1) Try WP HTTP API ---
        $args = [
            'timeout'   => 30,
            // In dev, skip SSL verification; in prod, leave it on.
            'sslverify' => self::$is_production,
            // Force streams transport if you want to avoid cURL entirely:
            // 'transport' => 'stream',
        ];

        $response = wp_remote_get( $url, $args );

        if ( is_wp_error( $response ) ) {
            self::error_log( sprintf(
                'WP_HTTP failed for %s: %s',
                $url,
                $response->get_error_message()
            ) );
        } else {
            $code = wp_remote_retrieve_response_code( $response );
            $body = wp_remote_retrieve_body( $response );

            if ( 200 === $code && '' !== $body ) {
                return $body;
            }

            self::error_log( sprintf(
                'WP_HTTP non-200 or empty for %s: HTTP %d, %d bytes',
                $url,
                $code,
                strlen( $body )
            ) );
        }

        // --- 2) Fallback to file_get_contents() ---
        // Production: strict
        if ( self::$is_production ) {
            $content = @file_get_contents( $url );
            if ( false === $content ) {
                self::error_log( 'file_get_contents failed for ' . $url );
            }
            return $content;
        }

        // Dev: relax SSL AND try http if https fails
        $scheme    = wp_parse_url( $url, PHP_URL_SCHEME );
        $try_urls  = [ $url ];
        if ( 'https' === $scheme ) {
            $try_urls[] = preg_replace( '#^https:#', 'http:', $url );
        }

        $context = stream_context_create([
            'ssl' => [
                'verify_peer'      => false,
                'verify_peer_name' => false,
            ],
        ]);

        foreach ( $try_urls as $u ) {
            $content = @file_get_contents( $u, false, $context );
            if ( false !== $content ) {
                return $content;
            }
            self::error_log( 'file_get_contents(dev) failed for ' . $u );
        }

        return false;
    }

    /**
     * Bundles CSS contents into a single CSS file.
     * Fetches external CSS from URLs and combines with inline styles.
     * Adheres to WordPress coding standards.
     *
     * @param array  $css_elements     Array of CSS URLs and inline style contents.
     * @param string $output_file_path Path to the output CSS file.
     */
    private static function get_bundle_css( $css_elements, $cache_file_content, $requested_url ) {
        $bundled_css = '';
        $bundle_array = [];

        foreach ( $css_elements as $element ) {
            /**
             * Get the tag type.
             */
            $tag = $element['tag'] ?? false;

            /**
             * Get the element attributes.
             */
            $attributes = $element['attributes'] ?? [];

            /**
             * Check if the element is a URL.
             */
            if ( 'link' === $tag ) {
                /**
                 * Get the link url.
                 */
                $url = $attributes['href'] ?? '';

                /**
                 * Get the content of the file.
                 */
                $css_content = self::get_file_contents( $url );

                /**
                 * If content is not empty,
                 * then add it to the css bundle.
                 */
                if ( $css_content !== false ) {
                    /**
                     * Convert all file relative urls to domain relative.
                     */
                    $css_content = self::convert_urls_to_domain_relative( $url, $css_content );

                    $bundled_css .= $css_content;
                }

            } else if( 'style' === $tag ) {
                /**
                 * Get the style custom id added by the api.
                 */
                $style_custom_id = $element['styleId'] ?? '';

                /**
                 * Get the full style tag, including its content.
                 */
                $full_tag = self::get_wrapped_content_by_id( $cache_file_content, $style_custom_id );

                /**
                 * Assume it's inline CSS.
                 */
                $css_content = self::extract_content_from_style_tag( $full_tag );

                /**
                 * Convert all file relative urls to domain relative.
                 */
                $css_content = self::convert_urls_to_domain_relative( $requested_url, $css_content );

                $bundled_css .= $css_content;
            }
        }

        return $bundled_css;
    }

    /**
     * Converts file relative URLs in a CSS file to domain relative URLs.
     *
     * @param string $css_file_url The full URL to the CSS file.
     * @param string $css_content The content of the CSS file.
     * @return string The modified CSS content with domain relative URLs.
     */
    private static function convert_urls_to_domain_relative(string $css_file_url, string $css_content): string {
        /** 
         * Extract the domain and path from the CSS file URL.
         */
        $parsed_url = wp_parse_url( $css_file_url );
        $base_path  = isset( $parsed_url['path'] ) ? dirname( $parsed_url['path'] ) : '';

        /** 
         * Regex to find relative URLs 
         */
        $regex = '/url\(\s*([\'"]?)(?!http|\/)([^\'"\)]+)\1\s*\)/i';

        /** 
         * Callback function to convert file path to domain relative path
         */
        $callback = function ($matches) use ($base_path) {
            /** 
             * If it starts with data:image, leave it alone.
             */
            if ( 0 === strpos( $matches[2], 'data:image' ) ) {
                return $matches[0];
            }

            /**
             * Build the new url.
             */
            $new_path = rtrim($base_path, '/') . '/' . ltrim($matches[2], '/');

            /**
             * Return the new url wrapped in the url selector.
             */
            return 'url(' . $new_path . ')';
        };

        /** 
         * Replace file relative URLs with domain relative URLs 
         */
        $new_css_content = preg_replace_callback($regex, $callback, $css_content);

        return $new_css_content;
    }

    /**
     * Extracts the CSS content from a style tag.
     *
     * @param string $style_tag The complete style tag.
     * @return string The extracted CSS content.
     */
    public static function extract_content_from_style_tag( $style_tag ) {
        $start = strpos( $style_tag, '>' ) + 1;
        $end = strrpos( $style_tag, '</style>' );

        return $end > $start ? substr( $style_tag, $start, $end - $start ) : '';
    }

    /**
     * Extracts CSS URLs and inline style content from an HTML document.
     * Uses string functions for extraction, adhering to WordPress coding standards.
     * Determines if the CSS is in the head or footer of the document, identifies its type (stylesheet or inline),
     * and provides the full tag along with the URL for stylesheets.
     *
     * @param string $html The HTML document as a string.
     * @return array An array of CSS URLs and inline style contents with their locations, types, and full tags.
     */
    private static function extract_css_from_html($html) {
        $css_elements = [];
        $body_pos = strpos($html, '<body');

        /**
         * Extract CSS URLs from link tags and complete style tags.
         */
        $start = 0;
        while (($start = strpos($html, '<', $start)) !== false) {
            /**
             * Find the end of the current tag.
             */
            $end = strpos($html, '>', $start);
            if ($end === false) {
                break;
            }

            $tag = substr($html, $start, $end - $start + 1);

            // Determine location of the tag
            $location = ($start < $body_pos || $body_pos === false) ? 'head' : 'footer';

            /**
             * Common function to extract attribute values.
             */
            $extract_attribute = function($attribute, $tag) {
                $pos = strpos($tag, $attribute . '=');
                if ($pos !== false) {
                    $delimiter = $tag[$pos + strlen($attribute) + 1];
                    $attr_start = $pos + strlen($attribute) + 2;
                    $attr_end = strpos($tag, $delimiter, $attr_start);
                    if ($attr_end !== false) {
                        return substr($tag, $attr_start, $attr_end - $attr_start);
                    }
                }
                return null;
            };

            /** 
             * Check if it's a link tag with a stylesheet:
             */
            if (strpos($tag, '<link') === 0 && (strpos($tag, 'rel="stylesheet"') !== false || strpos($tag, "rel='stylesheet'") !== false)) {
                $href   = $extract_attribute('href', $tag);
                $id     = $extract_attribute('id', $tag);
                $class  = $extract_attribute('class', $tag);
                $add_style_manually = self::is_style_set_to_be_added_manually( $id, $class );
    
                if ($href !== null) {
                    $css_elements[] = [
                        'tag'           => $tag,
                        'url'           => $href,
                        'location'      => $location,
                        'type'          => 'stylesheet',
                        'id'            => $id,
                        'class'         => $class,
                        'add_manually'  => $add_style_manually,
                    ];
                }
            }

            /**
             * Check if it's a style tag.
             */
            if (strpos($tag, '<style') === 0) {
                $style_end = strpos($html, '</style>', $end);
                if ($style_end !== false) {
                    $id                 = $extract_attribute('id', $tag);
                    $class              = $extract_attribute('class', $tag);
                    $add_style_manually = self::is_style_set_to_be_added_manually( $id, $class );

                    $css_elements[] = [
                        'tag'           => substr($html, $start, $style_end - $start + 8),
                        'location'      => $location,
                        'type'          => 'inline',
                        'id'            => $id,
                        'class'         => $class,
                        'add_manually'  => $add_style_manually,
                    ];
                }
                /**
                 * Move past the end of the style tag.
                 */
                $start = $style_end + 8;
                continue;
            }

            /**
             * Move to the next tag.
             */
            $start = $end + 1;
        }

        return $css_elements;
    }

    private static function is_style_set_to_be_added_manually( $id, $class ) {
        /**
         * Return false if neither $id not $class is set.
         */
        if( !$id && !$class ){
            return false;
        }

        /**
         * Get selectors to exlcude.
         */
        $selectors_to_exclude = BaseController::get_setting( 'jptgb_setting_include_in_critical_css' );
        $selectors_to_exclude = Utils::convert_selectors_to_array( $selectors_to_exclude );

        /**
         * Return if there are not selectors to exlucde.
         */
        if( !$selectors_to_exclude || empty( $selectors_to_exclude ) ){
            return false;
        }

        /**
         * Initially assume we don't have to exclude the array.
         */
        $is_to_exclude = false;

        /**
         * Check if any of the attributes names are set to exlcude.
         */
        foreach( $selectors_to_exclude as $selector_to_exclude ){
            /**
             * Check for id.
             */
            if( ( strpos( $selector_to_exclude, '#' ) !== false ) && $id ){
                /**
                 * Get the selector name.
                 */
                $selector_name = ltrim( $selector_to_exclude, '#' );

                /**
                 * Check if the selector match the id of the script.
                 */
                if( $id == $selector_name ){
                    $is_to_exclude = true;
                    break; 
                }
            } 

            /**
             * Check for classes.
             */
            if( ( strpos( $selector_to_exclude, '.' ) !== false ) && $class ){
                /**
                 * Get the selector name.
                 */
                $selector_name = ltrim( $selector_to_exclude, '.' );

                /**
                 * Get all class names of the selector.
                 */
                $all_classes = explode( ' ', $class );

                /**
                 * Iterate over each class
                 */
                foreach( $all_classes as $class_name ){
                    if( $class_name == $selector_name ){
                        $is_to_exclude = true;
                        break 2; 
                    }
                }
            } 
        }

        return $is_to_exclude;
    }
}