<?php

/**
 * The Images Controller Class.
 *
 * @author Yaidier Perez
 * */

namespace Jptgb\Controllers;

use Jptgb\Resources\Utils;

class ImagesController extends BaseController {
    /**
     * Define calss properties.
     */
    protected static $tmp_dir_path = false;

    protected static $largest_paint_area = null;

    protected static $original_tags = [];

    public static function init() {
        /**
         * If we just turn on/off the images,
         * and the cache was active then lets
         * clear up the cache then.
         */
        if( in_array( 'jptgb_setting_resize_images',            self::$changed_settings ) ||
            in_array( 'jptgb_setting_compress_svg',             self::$changed_settings ) ||
            in_array( 'jptgb_setting_convert_images_to_webp',   self::$changed_settings ) ) {

            /**
             * Flush all images.
             */
            self::flush_all_images_files();

            /**
             * Flush all cache.
             */
            CacheController::flush_all_cache_files();
            
        /**
         * Just flush the cache if setting 
         * explicit width and height has changed.
         */
        } elseif ( in_array( 'jptgb_setting_set_explicit_w_and_h', self::$changed_settings ) ) {
            /**
             * Flush all cache.
             */
            CacheController::flush_all_cache_files();
        }

        /**
         * Return if Load Fonts on FUI is deactivated.
         * 
         * @see Wp Dashboard -> Jptgb -> Fonts
         */
        if( !isset( self::$settings['jptgb_setting_resize_images'] ) || !self::$settings['jptgb_setting_resize_images'] ) {
            return;
        }

        /**
         * Create the tmp directory.
         */
        self::create_the_tmp_dir();

        
    }

    public static function flush_all_images_files() {
        $tmp_dir_path = self::get_tmp_dir_path();              

        if( is_dir( $tmp_dir_path ) ) {
            Utils::remove_directory_recursively( $tmp_dir_path );
        }

        /**
         * Flush Images data.
         */
        delete_option( 'jptgb_images_info' );

        /**
         * Remove alla 404 transient stored.
         */
        Utils::delete_jptgb_image_url_404_transients();
    }

    public static function create_the_tmp_dir() {
        $tmp_dir_path = self::get_tmp_dir_path();

        if( !wp_mkdir_p( $tmp_dir_path ) ){
            return false;
        }

        return true;
    }

    public static function get_tmp_dir_path() {
        if( self::$tmp_dir_path ) {
            return self::$tmp_dir_path;
        }

        $upload_dir_info        = wp_upload_dir();
        $upload_dir_path        = $upload_dir_info['basedir'];
        self::$tmp_dir_path     = $upload_dir_path . '/jptgb/images';

        return self::$tmp_dir_path;
    }

    

    

    /**
     * Extracts img tags from an HTML document using string manipulation.
     *
     * @param string $html The HTML content to process.
     * @return array An array of img tag strings.
     */
    public static function extractImgTagsFromString($html) {
        $tags = [];
        $lowercaseHtml = strtolower($html); // Convert to lowercase for case-insensitive search
        $start = 0;

        while (($start = strpos($lowercaseHtml, '<img', $start)) !== false) {
            $end = strpos($lowercaseHtml, '>', $start);
            if ($end === false) {
                break; // No closing '>' found for the img tag
            }

            $imgTag = substr($html, $start, $end - $start + 1);
            $tags[] = [ 'tag' => $imgTag ];
            $start = $end + 1;
        }

        return $tags;
    }

    /**
     * Extracts opening parts of tags with the `data-jptgb-bgimage="1"` attribute using string manipulation.
     *
     * @param string $html The HTML content to process.
     * @return array An array of opening tag strings with the specified attribute.
     */
    public static function extractTagsWithDataAttributeFromString($html) {           
        /** 
         * Regex breakdown:
         * - `<`                     : opening angle bracket
         * - `[^>]+?`                : one or more chars that are not '>', non-greedy
         * - `data-jptgb-bgimage="1"`: the attribute we care about
         * - `[^>]*`                 : any remaining attributes
         * - `>`                     : closing angle bracket
         * - `i`                     : case-insensitive
         */
        $pattern = '/<[^>]+?data-jptgb-bgimage="1"[^>]*>/i';
        $tags    = [];

        if ( preg_match_all( $pattern, $html, $matches ) ) {
            foreach ( $matches[0] as $tag ) {
                $tags[] = [ 'tag' => $tag ];
            }
        }

        return $tags;
    }

    /**
     * Parse data-jptgb, src and srcset on a list of image tags.
     *
     * For any tag whose data-jptgb decodes to a numeric array,
     * it will emit one entry per item (replacing the original).
     * Otherwise it augments the tagInfo in-place.
     *
     * @param array  $img_tags Array of [ 'tag' => '<img …>' ] entries.
     * @param string $base_url Optional base URL for resolving relative URLs.
     * @return array           The new array of tagInfo entries.
     */
    public static function parseDataJptgbAttributesAndSrc( array $img_tags, string $base_url = '' ): array {
        // ** Determine the absolute base/domain URL once */
        $base = empty( $base_url ) ? site_url() : $base_url;
        $parts = wp_parse_url( $base );
        $domain = $parts['scheme'] . '://' . $parts['host'];

        $out = [];
        $tmp_out = [];
        $tmp_count = count( $img_tags );

        // ** Use a for‐loop so we can safely push into $out without altering $img_tags */
        for ( $i = 0; $i < $tmp_count; $i++ ) {
            $tagInfo    = $img_tags[ $i ];
            $opening_tag = $tagInfo['tag'] ?? '';

            // 1) Handle data-jptgb
            if ( preg_match( '/data-jptgb="([^"]+)"/', $opening_tag, $m ) ) {
                $json = html_entity_decode( $m[1], ENT_QUOTES, 'UTF-8' );
                $data = json_decode( $json, true );

                // 1a) If it's a numeric array, emit one entry per item
                if ( !wp_is_numeric_array( $data ) ) {
                    $tmp_out[] = $tagInfo;
                    continue;
                }

                $unique_class_selector = uniqid();

                /**
                 * Preserve the original tag.
                 */
                self::$original_tags[$unique_class_selector] = $opening_tag;

                foreach ( $data as $item ) {
                    // ** Re-encode just this item as the new data-jptgb **
                    $item['unique_class_selector'] = $unique_class_selector;
                    // $item['original_opening_tag'] = $opening_tag;
                    $item_json = wp_json_encode( $item );
                    $escaped   = esc_attr( $item_json );
                    $pattern   = '/\sdata-jptgb="[^"]*"/';
                    $replace   = ' data-jptgb="' . $escaped . '"';

                    if ( preg_match( $pattern, $opening_tag ) ) {
                        $new_tag = preg_replace( $pattern, $replace, $opening_tag, 1 );
                    } else {
                        $new_tag = preg_replace( '/>$/', $replace . '>', $opening_tag, 1 );
                    }

                    $tmp_out[] = [ 'tag' => $new_tag ];
                }

                // Skip all the rest of the processing for this original tag
                continue;
            }
        }

        $count = count( $tmp_out );

        // ** Use a for‐loop so we can safely push into $out without altering $img_tags */
        for ( $i = 0; $i < $count; $i++ ) {
            $tagInfo    = $tmp_out[ $i ];
            $opening_tag = $tagInfo['tag'] ?? '';

            // 1) Handle data-jptgb
            if ( preg_match( '/data-jptgb="([^"]+)"/', $opening_tag, $m ) ) {
                $json = html_entity_decode( $m[1], ENT_QUOTES, 'UTF-8' );
                $data = json_decode( $json, true );

                // 1b) Otherwise merge its properties onto $tagInfo
                foreach ( $data as $key => $val ) {
                    $tagInfo[ $key ] = $val;
                }

                // Track largest paint area if you need it
                $paint = $data['paintArea'] ?? 0;
                if ( $paint > self::$largest_paint_area ) {
                    self::$largest_paint_area = $paint;
                }
            }

            /**
             * Convert url to absolute.
             */
            $tagInfo['src'] = self::convertToAbsoluteUrl($tagInfo['src'] ?? '', $domain );

            // 3) Process srcset → array of absolute URLs
            if ( preg_match( '/srcset="([^"]+)"/', $tagInfo['tag'], $ss_m ) ) {
                $tagInfo['srcset_tag_value'] = $ss_m[1];
                $pieces = array_map(
                    function( $part ) use ( $domain ) {
                        $url_part = trim( explode( ' ', trim( $part ), 2 )[0] );
                        $abs      = self::convertToAbsoluteUrl( $url_part, $domain );
                        return [ 'url' => $abs ];
                    },
                    explode( ',', $ss_m[1] )
                );
                $tagInfo['srcset'] = $pieces;
            }

            // Finally push this single, augmented tagInfo
            $out[] = $tagInfo;
        }

        return $out;
    }

    /**
     * Converts a given URL to an absolute URL based on the provided domain URL.
     * 
     * @param string $url The URL to convert.
     * @param string $domainUrl The domain URL to use for conversion.
     * @return string The absolute URL.
     */
    private static function convertToAbsoluteUrl($url, $domainUrl) {
        // Check if the URL is already absolute
        if (!preg_match('/^https?:\/\//', $url)) {
            // Prepend domain URL to convert to absolute URL
            $url = rtrim($domainUrl, '/') . '/' . ltrim($url, '/');
        }
        return $url;
    }

    /**
     * Convert a URL to its local file path in WordPress, preserving case sensitivity, including theme directories.
     *
     * @param string $url The URL of the file.
     * @return string The local file path of the file.
     */
    public static function convert_media_library_url_to_path( $url ) {
        // Helper function to remove the protocol (http or https) from a URL
        $remove_protocol = function($url) {
            return preg_replace("(^https?://)", "", $url);
        };

        // Normalize and decode the URL for comparison, remove protocol
        // Preserve the original case sensitivity of the URL
        $normalized_url = trailingslashit( urldecode( $remove_protocol($url) ) );

        // Get the upload directory information and normalize the base URL, remove protocol
        // Preserve the case of the upload directory URL
        $upload_dir = wp_upload_dir();
        $upload_baseurl = trailingslashit( $remove_protocol($upload_dir['baseurl']) );
        $upload_basedir = trailingslashit( $upload_dir['basedir'] ); // Ensure trailing slash

        // Replace the base URL of the uploads directory with the case-sensitive URL
        $path = str_replace( $upload_baseurl, $upload_basedir, $normalized_url );

        // Check if the replacement has occurred, if not, try the theme directory
        if( $path === $normalized_url ) {
            // Preserve the case of the theme directory URL
            $theme_dir_url = trailingslashit( $remove_protocol(get_stylesheet_directory_uri()) );
            $theme_dir_path = trailingslashit( get_stylesheet_directory() ); // Ensure trailing slash
            $path = str_replace( $theme_dir_url, $theme_dir_path, $normalized_url );
        }

        // Return the path with the trailing slash removed
        return untrailingslashit( $path );
    }

    /**
     * Checks if an image URL is local to the current WordPress installation.
     *
     * This function considers both absolute URLs (with a host) and relative URLs.
     * It returns true if the image URL is local to the site, which includes all relative URLs,
     * as they do not specify a host and are therefore inherently local.
     *
     * @param string $image_url The URL of the image to check.
     * @return bool True if the image URL is local, false if it is external.
     */
    public static function is_image_url_local( $image_url ) {
        /**
         * Directly return true for relative URLs (those that don't include a host).
         */
        if ( !wp_parse_url( $image_url, PHP_URL_HOST ) ) {
            return true;
        }

        /**
         * Get the site URL and parse it for the host.
         */
        $site_url = wp_parse_url( get_site_url(), PHP_URL_HOST );

        /**
         * Parse the image URL for the host.
         */
        $image_url_host = wp_parse_url( $image_url, PHP_URL_HOST );

        /**
         * Compare the hosts.
         */
        return $site_url === $image_url_host;
    }

    /**
     * Generates a hash from a URL.
     *
     * @param string $url The URL to hash.
     * @param string $algorithm The hashing algorithm to use (e.g., "md5", "sha256").
     * @return string The generated hash.
     */
    public static function generate_hash( $url, $length = 16, $algorithm = 'sha256' ) {
        $full_hash  = hash( $algorithm, $url );
        $short_hash = substr( $full_hash, 0, $length );

        return $short_hash;
    }

    public static function store_image_in_local( $image ) {
        /**
         * Check if images is either local or external.
         */
        if( self::is_image_url_local( $image['src'] ) ){
            /**
             * Convert the url path.
             */
            $destination_path = Utils::convert_url_to_absolute_path( $image['src'] );
        
        /**
         * In case the url is external, then lets
         * fetch it and save it in local storage.
         */
        } else {
            /**
             * Get destination path.
             */
            $destination_path = self::get_destination_path( $image, 'tmp' );

            /**
             * Return if destination path already exist.
             */
            if( Utils::does_file_exists( $destination_path ) ) {
                $image['src_path'] = $destination_path;
                return $image;
            }

            /** 
             * Fetch the image content from the URL.
             */
            $file_content = Utils::read_file_from_url( $image['src'] );
            
            /**
             * Return the image item if we can't read the file.
             */
            if( !$file_content ) {
                return $image;
            }
            
            /**
             * Skip if writing fails.
             */
            if( !Utils::write_file( $destination_path, $file_content ) ) {
                return $image;
            }
        }

        /**
         * Assign source path to the image array.
         */
        $image['src_path'] = $destination_path;

        return $image;
    }

    // /**
    //  * Appends a specified size suffix to the filename in a given path.
    //  *
    //  * This function modifies the provided file path by inserting a predefined
    //  * size suffix before the file's extension, typically used for denoting
    //  * resized image files.
    //  *
    //  * @param string $path The original file path.
    //  * @return string The file path including the new size suffix.
    //  */
    // public static function append_size_suffix_to_path( $path, $image ) {
    //     /**
    //      * Define the suffix to be appended.
    //      */
    //     $size_suffix = '-resized';
    //     $size_suffix .= $image['width']  ? '-' . $image['width'] : '';
    //     $size_suffix .= $image['height'] ? 'x' . $image['height'] : '';

    //     /**
    //      * Use pathinfo to get information about the original file path.
    //      */
    //     $info = pathinfo( $path );

    //     /**
    //      * Reconstruct the path with the size suffix before the file extension.
    //      */
    //     $new_path = $info['dirname'] . '/' . $info['filename'] . $size_suffix . '.' . $info['extension'];

    //     return $new_path;
    // }

    public static function get_destiantion_paths( $url, $image, $add_resize_info = false ) {
        /**
         * Use basename to get the filename with extension.
         */
        $filename_with_extension = basename( wp_parse_url( $url, PHP_URL_PATH ) );

        /**
         * Use pathinfo to separate the filename and its extension.
         */
        $file_parts = pathinfo( $filename_with_extension );

        /**
         * Prepare the result.
         */
        $file_name = $file_parts['filename'];
        $file_ext  = $file_parts['extension'];

        /**
         * Add dimensions to the filename only if it is not svg.
         */
        if( $add_resize_info ){
            $file_name .= '-' .  $image['width'];
            $file_name .= 'x' .  $image['height'];
        }

        /**
         * Get the hashed url.
         */
        $hashed_url = self::generate_hash( $url );

        /**
         * Partial path.
         */
        $partial_path = self::get_tmp_dir_path() . '/' . $file_name . '-'. $hashed_url;

        /**
         * Define the final path.
         */
        if( isset( self::$settings['jptgb_setting_convert_images_to_webp'] ) && self::$settings['jptgb_setting_convert_images_to_webp'] && 'svg' != $file_ext ){
            $final_path = $partial_path . '.webp';
        } else {
            $final_path = $partial_path . '.' . $file_ext;
        }

        /**
         * Define the source path.
         */
        $path = $partial_path . '.' . $file_ext;

        return [ 'destination_path' => $path, 'final_destination_path' => $final_path ];
    }

    /**
     * Resize and save images from an array of image data.
     *
     * @param array $images Array of image data with src, width, and height.
     */
    public static function reszie_and_optimize_images( $images, $url_hash ) {
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        /**
         * Remove old image information in the db.
         * 
         * IMPORTANT!!! Revmove this in the future.
         */
        delete_option( 'jptgb_images_info' );

        /**
         * Check if we have to convert images to webp.
         */
        $is_convert_to_webp = BaseController::get_setting( 'convert_images_to_webp' ) ? true : false; 

        /**
         * Define images groups.
         */
        $images_ready       = [];
        $images_to_process  = [];

        /**
         * Iterate over each image.
         */
        foreach( $images as &$image_item ) { 
            try {               
                /**
                 * Get the image data.
                 */
                $image_tag = $image_item['tag'] ?? '';
                $image_url = $image_item['src'] ?? '';
                

                if( strpos( $image_url, 'ding-course-yt-vide' ) !== false ){
                    $stop = 1;
                }

                /**
                 * Get the tag name.
                 */
                $tag_name = Utils::get_tag_name_from_string( $image_tag );
                $image_item['tag_name'] = $tag_name;

                /**
                 * If the image is the bg of an iframe lets crop it and it comes from our api
                 * then lets crop it a bit on the edges.
                 */
                if( 'iframe' === $tag_name && strpos( $image_url, BaseController::get_env_variable( 'JPTGB_API_BASE_URL' ) ) !== false ){
                    $image_item['width'] =  intval( $image_item['width'] ) - 4;
                    $image_item['height'] = intval( $image_item['height'] ) - 4;
                    $image_item['crop'] = true;
                }

                /**
                 * Get width and height data.
                 */
                $width  = $image_item['width'] ?? '';
                $height = $image_item['height'] ?? '';


                /**
                 * Set the image hash.
                 */
                $image_item = self::set_the_image_hash( $image_item, $image_tag . $width . $height );

                /**
                 * Get the image hash.
                 */
                $image_hash = $image_item['image_hash'] ?? '';
                
                /**
                 * Skip invalid URLs.
                 */
                if( !Utils::is_url_valid( $image_url ) ) {
                    continue;
                }
                
                /**
                 * Skip if the url do not return 200.
                 */
                if( !self::is_image_url_successful( $image_url, $image_hash ) ){
                    continue;
                }
                
                /**
                 * Set the image format.
                 */
                $image_item = self::set_the_image_format( $image_item, $image_url );

                /**
                 * Skip if we can't define the image format.
                 */
                if( isset( $image_item['format'] ) && !$image_item['format'] ){
                    continue;
                }

                /**
                 * Continue if image is svg and we are set not to compress svg.
                 */
                if( 'svg' === $image_item['format'] && !self::get_setting( 'compress_svg' ) ){
                    continue;
                }

                /**
                 * Get image path and filename.
                 */
                $path_and_filename                  = Utils::extract_filename_and_path_from_url( $image_url );
                $image_item['path_segment']         = $path_and_filename['path']            ?? '';
                $image_item['filename']             = $path_and_filename['filename']        ?? '';
                $image_item['filename_no_ext']      = $path_and_filename['filename_no_ext'] ?? '';

                /**
                 * Get the path hash.
                 */
                $image_item['path_hash'] = Utils::create_url_safe_hash( $image_item['path_segment'] );

                /**
                 * Save the original src.
                 */
                $image_item['original_src'] = $image_url;

                /**
                 * Get the image path.
                 */
                $image_item = self::store_image_in_local( $image_item );

                /**
                 * If the path does not exist, then continue
                 */
                if( !isset( $image_item['src_path'] ) || !Utils::does_file_exists( $image_item['src_path'] ) ){
                    continue;
                }

                /**
                 * Determine if the image is the largest paint area.
                 */
                $image_item['ismaxpaintarea'] = ( !empty( $image_item['paintArea'] ) && $image_item['paintArea'] === self::$largest_paint_area ) ? true : false;

                /**
                 * Handle svg format.
                 */
                switch( $image_item['format'] ) {
                    case 'svg': {
                        /**
                         * Define the destiantion path where to storage the image in local.
                         */
                        $destination_path = self::get_tmp_dir_path() . '/' . $image_item['path_hash'] . '/' . $image_item['filename_no_ext'] . '.svg';

                        /**
                         * Check if destination path already exist.
                         */
                        if( Utils::does_file_exists( $destination_path ) ){
                            /**
                             * Update the src_path with the new location of the image coverted to webp in our local server.
                             */
                            $image_item['src_path'] = $destination_path;
                
                            /**
                             * Update the image item.
                             */
                            $image_item = self::update_image_item( $image_item );

                            /**
                             * Add the current image to the ready list.
                             */
                            $images_ready[] = $image_item;

                        } else {
                            /**
                             * Get the url from where we are gonna fetch the image in the api
                             */
                            $image_item['url_for_api'] = Utils::convert_absolute_path_to_url( $image_item['src_path'] );
                            
                            /**
                             * Add the image to the $images_to_process group.
                             */
                            $images_to_process[] = $image_item;

                        }
                    }
                    break; 
                    default:{
                        /**
                         * Check if we have to resize the image.
                         */
                        if( BaseController::get_setting( 'resize_images' ) ){
                            $image_item = self::resize_image( $image_item );
                        }

                        /**
                         * Check if we have to convert the image to webp.
                         */
                        if( $is_convert_to_webp ){
                            /**
                             * Get the resized segment of the url.
                             */
                            $resized_url_segment = isset( $image_item['resized_segment'] ) ? $image_item['resized_segment'] . '/' : '';

                            /**
                             * Define the destiantion path where to storage the image in local.
                             */
                            $destination_path = self::get_tmp_dir_path() . '/' . $image_item['path_hash'] . '/' . $resized_url_segment . $image_item['filename_no_ext'] . '.webp';

                            /**
                             * Check if destination path already exist.
                             */
                            if( Utils::does_file_exists( $destination_path ) ){
                                /**
                                 * Update the src_path with the new location of the image coverted to webp in our local server.
                                 */
                                $image_item['src_path'] = $destination_path;
                    
                                /**
                                 * Update the image item.
                                 */
                                $image_item = self::update_image_item( $image_item );

                                /**
                                 * Add the current image to the ready list.
                                 */
                                $images_ready[] = $image_item;

                            } else {
                                /**
                                 * Get the url from where we are gonna fetch the image in the api
                                 */
                                $image_item['url_for_api'] = Utils::convert_absolute_path_to_url( $image_item['src_path'] );
    
                                /**
                                 * Add the image to the $images_to_process group;
                                 */
                                $images_to_process[] = $image_item;
                            }

                        /**
                         * In case we do not need to convert them to webp.
                         */
                        } else {
                            /**
                             * Update the image item.
                             */
                            $image_item = self::update_image_item( $image_item );

                            /**
                             * Add the image to the $images_ready group.
                             */
                            $images_ready[] = $image_item;
                        }
                    }
                    break;
                }
            } catch (\Throwable $th) {
                continue;
            }
        }

        /**
         * Send images to the api only if $images_to_process group is not empty.
         */
        if( !empty( $images_to_process ) ){
            /**
             * Get the converted images urls from the api.
             */
            $processed_images = self::process_images( $images_to_process, $url_hash );

            if( false === $processed_images ){
                $processed_images = $images_ready;
            }
        }
        
        /**
         * Get all images group into only one.
         */
        $all_images = isset( $processed_images ) ? array_merge( $images_ready, $processed_images ) : $images_ready;

        return $all_images;
    }

    private static function process_images( $images_to_process, $url_hash ) {
        /**
         * Check if we have to send the images to the api or is self hosted.
         */
        if( !self::get_setting( 'activate_self_hosted' ) ){
            $images_from_api = self::send_images_to_api( $images_to_process, $url_hash );

            /**
             * If the response was not positive, the return the current images data.
             */
            if( !$images_from_api ){
                /**
                 * Report the incident.
                 */
                BaseController::$logs->register( 'We couldn\'t convert the images to webp, we got a bad response from the API...', 'error' );
    
                return false;
            }
            
            /**
             * Decode the API response
             */
            $images_from_api = json_decode( $images_from_api, true );
    
            /**
             * Collect all converted images.
             */
            foreach( $images_from_api as &$image_from_the_api ){
                /**
                 * Get the image process status.
                 */
                $image_compression_status = $image_from_the_api['api_success'];

                /**
                 * Continue if the process in the API failed somehow.
                 */
                if( !$image_compression_status ){
                    continue;
                }

                /**
                 * Get the api from where to fetch the converted image in the API.
                 */
                $api_fetch_url  = self::$webspeed_api_base . $image_from_the_api['api_fetch_url'];
    
                /**
                 * Get the image contents.
                 */
                $image_contents = Utils::read_file_from_url( $api_fetch_url );
    
                /**
                 * Continue if we can't read the image contents.
                 */
                if( !$image_contents ){
                    continue;
                }
    
                /**
                 * Get the resized segment of the url.
                 */
                $resized_url_segment = isset( $image_from_the_api['resized_segment'] ) ? $image_from_the_api['resized_segment'] . '/' : '';

                /**
                 * Destination format.
                 */
                $destination_format = 'svg' === $image_from_the_api['format'] ? 'svg' : 'webp';

                /**
                 * Define the destiantion path where to storage the image in local.
                 */
                $destination_path = self::get_tmp_dir_path() . '/' . $image_from_the_api['path_hash'] . '/' . $resized_url_segment . $image_from_the_api['filename_no_ext'] . '.' . $destination_format;
   
                /**
                 * Save the image in the local server.
                 */
                Utils::write_file( $destination_path, $image_contents );
    
                /**
                 * Update the src_path with the new location of the image coverted to webp in our local server.
                 */
                $image_from_the_api['src_path'] = $destination_path;
    
                /**
                 * Update the image item.
                 */
                $image_from_the_api = self::update_image_item( $image_from_the_api );
            }

            return $images_from_api;
        } else {
            /**
             * Get the compression rates.
             */
            $webp_compression_rate      = self::get_setting( 'webp_quality_rate' );
            $mpa_webp_compression_rate  = self::get_setting( 'mpa_webp_quality_rate' );

            foreach( $images_to_process as &$image_to_compress ){
                /**
                 * Get the url for the interface.
                 */
                $url_for_api = $image_to_compress['url_for_api'];

                /**
                 * Get the resized segment of the url.
                 */
                $resized_url_segment = isset( $image_to_compress['resized_segment'] ) ? $image_to_compress['resized_segment'] . '/' : '';

                /**
                 * Destination format.
                 */
                $destination_format = 'svg' === $image_to_compress['format'] ? 'svg' : 'webp';

                /**
                 * Define the destiantion path where to storage the image in local.
                 */
                $destination_path = self::get_tmp_dir_path() . '/' . $image_to_compress['path_hash'] . '/' . $resized_url_segment . $image_to_compress['filename_no_ext'] . '.' . $destination_format;

                /**
                 * Define the compression rate.
                 */
                $compression_rate = ( isset( $image_to_compress['ismaxpaintarea'] ) && $image_to_compress['ismaxpaintarea'] ) ? $webp_compression_rate : $mpa_webp_compression_rate;

                /**
                 * Get the image format.
                 */
                $format = $image_to_compress['format'];

                /**
                 * Define args to pass.
                 */
                $args = [
                    'urlForApi'         => $url_for_api,
                    'destinationPath'   => $destination_path,
                    'compressionRate'   => $compression_rate,
                    'format'            => $format,
                ];

                /** 
                 * Encode the data.
                 */
                $args = wp_json_encode( $args );

                /**
                 * Escape the JSON string for use in the shell command.
                 */
                $escaped_args = escapeshellarg( $args );

                /**
                 * Command to execute the Node.js script with the JSON argument.
                 */
                $command = 'node ' . JPTGB_PATH . 'src/js/self-hosted/api-controllers-interface.js optimizeImage ' . $escaped_args;

                /**
                 * Execute the command and capture the output.
                 */
                $output = shell_exec( $command );

                /**
                 * Capture the error message.
                 */
                if( $output === null ){
                    /**
                     * Redirect stderr to stdout.
                     */
                    $command .= ' 2>&1';

                    /**
                     * Get the error message.
                     */
                    $error_message = shell_exec( $command );

                    /**
                     * Report the incident...
                     */
                    BaseController::$logs->register( 'Self Critical Css generation failid for url: ' . esc_url( $error_message ), 'error' );

                    continue;
                }

                /**
                 * Update the src_path with the new location of the image coverted to webp in our local server.
                 */
                $image_to_compress['src_path'] = $destination_path;
    
                /**
                 * Update the image item.
                 */
                $image_to_compress = self::update_image_item( $image_to_compress );

            }

            return $images_to_process;
        }        
    }

    private static function set_the_image_format( $image_item, $image_url ){
        /**
         * Get initial image format.
         */
        $format = $image_item['format'] ?? false;

        /**
         * Attempt to set the image format if not already defined or known.
         */
        if( false === $format || empty( $format ) || 'unknown' === $format ) {
            /**
             * Get the format from the image url.
             */
            $format = self::get_image_extension_from_url( $image_url );
                        
            /**
             * Define the image format.
             */
            $image_item['format'] = empty( $format ) ? false : $format;
        }

        return $image_item;
    }

    private static function set_the_image_hash( $image_item, $hash_string ) {
        /**
         * Create the image hash.
         */
        $image_hash = md5( $hash_string );

        /**
         * Update the image hash in the image item.
         */
        $image_item['image_hash'] = $image_hash;

        return $image_item;
    }

    public static function is_image_url_successful( $image_url, $image_hash ) {
        /**
         * Continue if the image file do not exist.
         */
        $image_url_code = get_transient( 'jptgb_image_url_404_' . $image_hash );

        /**
         * If the transient does not exist or has expired, then lets check again.
         */
        if( !$image_url_code ){
            /**
             * If url does not exist.
             */
            if( !self::check_image_url_exists( $image_url ) ){
                /**
                 * Set the transient and allow to check again in 1 hour.
                 */
                set_transient( 'jptgb_image_url_404_' . $image_hash, '404', HOUR_IN_SECONDS * 1 );

                /**
                 * Skip proccessing this url.
                 */
                return false;

            /**
             * If the url does exist.
             */
            } else {
                /**
                 * Set the transient and allow to check again in 30 days.
                 */
                set_transient( 'jptgb_image_url_404_' . $image_hash, '200', DAY_IN_SECONDS * 30 );
            }
    
        /**
         * Return false in case there is a transient code stored and is 404.
         */
        } else if( '404' === $image_url_code ) {
            return false;
        }

        return true;
    }

     /**
     * Converts an image to WebP format via API.
     *
     * Sends a request to the API endpoint to convert an image to WebP format.
     * 
     * @param string $image_content Base64 encoded string of the image.
     * @param string $image_format  The format of the image being converted.
     *
     * @return string|false The Base64 encoded string of the WebP image or false on failure.
     */
    private static function send_images_to_api( $images_data, $url_hash ) {
        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'image-to-webp';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /**
         * Get the compression rates.
         */
        $webp_compression_rate      = self::get_setting( 'webp_quality_rate' );
        $mpa_webp_compression_rate  = self::get_setting( 'mpa_webp_quality_rate' );
        $is_debug                   = self::get_setting( 'debug_activate' );

        /** 
         * Payload to be sent to the API.
         * Adjusted to send image content and format.
         */
        $body = wp_json_encode( [
            'email'             => self::get_setting( 'user_email' ),
            'urlHash'           => $url_hash,
            'webpCompRate'      => $webp_compression_rate,
            'mpaWebpCompRate'   => $mpa_webp_compression_rate,
            'imagesData'        => $images_data,
            'isDebug'           => $is_debug,
        ]);

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting the SVG Compresssion: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( 200 !== $status_code ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }

        /**
         * Get the body response.
         */
        $body = wp_remote_retrieve_body( $response );
        
        /**
         * Return the received data.
         */
        return $body;
    }

    /**
     * Check if an image URL exists.
     *
     * Uses a HEAD request to check for the image's existence based on the HTTP status code.
     * A status code in the 200 range indicates the resource exists. This function includes
     * conditional SSL verification based on the environment.
     *
     * @param string $url The URL of the image to check.
     * @return bool True if the image exists, false otherwise.
     */
    public static function check_image_url_exists( string $url ): bool {
        /**
         * Perform a HEAD request to get the headers of the URL.
         * Conditionally set 'sslverify' based on the environment.
         */
        $response = wp_remote_head($url, [
            'timeout'   => 5,
            'sslverify' => self::$is_production,
        ]);

        if (is_wp_error($response)) {
            /**
             * There was an error making the request.
             */
            return false;
        }

        /**
         * Check the HTTP status code.
         */
        $status_code = wp_remote_retrieve_response_code($response);

        /**
         * Return true if the status code is in the 200 range, indicating the image exists.
         */
        return $status_code >= 200 && $status_code < 400;
    }

    /**
     * Get an image extension from its URL.
     *
     * First attempts to extract and validate the file extension from the URL path.
     * If that yields nothing or isn’t a recognized image, falls back to a HEAD request
     * to read the Content-Type header.
     *
     * @param string $url The URL of the image.
     * @return string|null Lower-case extension (e.g. 'jpg', 'png'), or null if undeterminable.
     */
    public static function get_image_extension_from_url( string $url ): ?string {
        /** Try parsing the extension directly from the URL path. */
        $path = wp_parse_url( $url, PHP_URL_PATH );
        if ( $path ) {
            $file_info = wp_check_filetype( $path );
            /** Return it only if WP knows it’s an image type. */
            if ( ! empty( $file_info['ext'] ) && strpos( $file_info['type'], 'image/' ) === 0 ) {
                return strtolower( $file_info['ext'] );
            }
        }

        /** Fallback: make a safe HEAD request to read the Content-Type header. */
        $response = wp_safe_remote_head(
            esc_url_raw( $url ),
            [
                'timeout'     => 5,
                'redirection' => 5,
                'blocking'    => true,
            ]
        );

        /** Bail if there was an error or no header. */
        if ( is_wp_error( $response ) ) {
            return null;
        }

        $content_type = wp_remote_retrieve_header( $response, 'content-type' );
        if ( empty( $content_type ) ) {
            return null;
        }

        /** Strip any charset or extra parameters. */
        $mime = strtok( $content_type, ';' );

        /** Map known image MIME types to extensions. */
        $map = [
            'image/jpeg'    => 'jpg',
            'image/png'     => 'png',
            'image/webp'    => 'webp',
            'image/gif'     => 'gif',
            'image/bmp'     => 'bmp',
            'image/svg+xml' => 'svg',
        ];

        return $map[ $mime ] ?? null;
    }


    private static function get_destination_path( $image, $sufix = false, $file_extension = false ) {
        /**
         * Get the url.
         */
        $url = $image['src'];

        /**
         * Use basename to get the filename with extension.
         */
        $filename_with_extension = basename( wp_parse_url( $url, PHP_URL_PATH ) );

        /**
         * Use pathinfo to separate the filename and its extension.
         */
        $file_parts = pathinfo( $filename_with_extension );

        /**
         * Prepare the result.
         */
        $file_name  = $file_parts['filename'];
        $file_ext   = $file_parts['extension'];
        $sufix      = $sufix ? "-$sufix" : '';

        /**
         * Determine if need to add the hash or not.
         */
        if( isset( $image['has_hash'] ) && $image['has_hash'] ){
            $hashed_url = '';
            
        } else {
            $hashed_url = '-' . self::generate_hash( $url );

        }

        /**
         * Crop the length of the filename if it exceeds 73 characters. 
         */
        if( strlen( $file_name ) > 73 ) {
            $file_name = substr( $file_name, 0, 73 );
        }

        /**
         * If file extension is not set, then lets try to get the extension 
         * from the image type.
         */
        if( !$file_ext ) {
            $file_ext = $image['format'] ?? '';
        }

        /**
         * If a custom extension is defined, then lets overwrite the file 
         * extensin with it.
         */
        if( $file_extension ){
            $file_ext = $file_extension;
        }

        /**
         * Prepend a dot (.) if file extension is set, 
         * leave it blank otherwise.
         */
        $file_ext  = $file_ext ? ".$file_ext" : '';

        /**
         * Build the path.
         */
        $destination_path = self::get_tmp_dir_path() . '/' . $file_name . $sufix . $hashed_url . $file_ext;

        return $destination_path;
    }

    private static function handle_srcset_urls( $image ) {
        /**
         * Return if srcset is not set.
         */
        if( !isset( $image['srcset'] ) ){
            return $image;
        }

        /**
         * Iterate over each srcset url.
         */
        foreach( $image['srcset'] as &$srcset_data ){
            /**
             * Get the url.
             */
            $url = $srcset_data['url'];

            /**
             * Use basename to get the filename with extension.
             */
            $filename_with_extension = basename( wp_parse_url( $url, PHP_URL_PATH ) );

            /**
             * Use pathinfo to separate the filename and its extension.
             */
            $file_parts = pathinfo( $filename_with_extension );

            /**
             * Prepare the result.
             */
            $file_name  = $file_parts['filename'];
            $file_ext   = $file_parts['extension'];
            $hashed_url = '-' . self::generate_hash( $url );

            /**
             * Crop the length of the filename if it exceeds 73 characters. 
             */
            if( strlen( $file_name ) > 73 ) {
                $file_name = substr( $file_name, 0, 73 );
            }

            /**
             * If file extension is not set, then lets try to get the extension 
             * from the image type.
             */
            if( !$file_ext ) {
                $file_ext = $image['format'] ?? '';
            }

            /**
             * Prepend a dot (.) if file extension is set, 
             * leave it blank otherwise.
             */
            $file_ext = $file_ext ? ".$file_ext" : '';

            /**
             * Build the path.
             */
            $destination_path = self::get_tmp_dir_path() . '/' . $file_name . '-webp' . $hashed_url . '.webp';

            /**
             * Continue if url already exist.
             */
            if( file_exists( $destination_path ) ){
                $srcset_data['modified_url'] = Utils::convert_absolute_path_to_url( $destination_path );
                continue;
            }

            /**
             * Return if the url is not local.
             */
            if( !self::is_image_url_local( $url ) ) {
                continue;
            }

            /**
             * Get the image content.
             */
            $image_content = Utils::file_get_contents( $url );

            /**
             * Return if we can't find the image content.
             */
            if( !$image_content ) {
                continue;
            }

            /**
             * Call the API to convert the image to Webp.
             */
            $webp_image = self::convert_image_to_webp( $url, $image_content );

            if ( false === $webp_image ) {
                continue;
            }

            /**
             * Return if failed while writing the file.
             */
            if( !file_put_contents( $destination_path, $webp_image ) ){
                continue;
            }

            /**
             * Insert the modified url.
             */
            $srcset_data['modified_url'] = Utils::convert_absolute_path_to_url( $destination_path );
        }

        return $image;
    }

    private static function handle_convert_image_to_webp( $image ) {
        /**
         * Get destination path;
         */
        $destination_path = self::get_destination_path( $image, 'webp', 'webp' );

        /**
         * Return the destiantion path if it already exist.
         */
        if( file_exists( $destination_path ) ){
            $image['src_path'] = $destination_path;
            return $image;
        }

        /**
         * Get the src path.
         */
        $src_path = $image['src_path'];

        /**
         * Get the destiantion url
         */
        $src_url = Utils::convert_absolute_path_to_url( $src_path );

        /**
         * Get the image content.
         */
        $image_content = file_get_contents( $src_path );

        /**
         * Return if we can't find the image content.
         */
        if( !$image_content ) {
            return $image;
        }

        /**
         * Call the API to compress the SVG.
         */
        $webp_image = self::convert_image_to_webp( $src_url, $image_content );

        if ( false === $webp_image ) {
            self::error_log( "Webp compression failed." );
            return $image;
        }

        /**
         * Return if failed while writing the file.
         */
        if( !file_put_contents( $destination_path, $webp_image ) ){
            return $image;
        }

        $image['src_path'] = $destination_path;

        /**
         * Command to update the image.
         */
        $image['update_it'] = true;

        return $image;
    }

    /**
     * Modifies the file extension of a given path.
     *
     * Adheres to WordPress coding standards for PHP.
     *
     * @param string $path The original file path.
     * @param string $new_extension The new extension without the dot.
     * @return string The modified file path with the new extension.
     */
    public static function modify_file_extension( $path, $new_extension ) {
        /**
         * Extract the path information.
         */
        $info = pathinfo( $path );

        /**
         * Rebuild the path without the original extension and append the new extension.
         */
        $new_path = $info['dirname'] . DIRECTORY_SEPARATOR . $info['filename'] . '.' . $new_extension;

        return $new_path;
    }

    /**
     * Handle SVG compression and ensure compressed SVG is smaller.
     * 
     * @param mixed $image The image object or identifier (not used in this example).
     * @param string $src_path Path to the source SVG file.
     * @param string $destination_path Path where the compressed SVG should be saved, if smaller.
     */
    public static function handle_svg_optimization( $image ) {
        /**
         * Save original image data.
         */
        $original_image_data = $image;

        try {
            /**
             * Return if the compress svg switch is off.
             */
            if( !isset( self::$settings['jptgb_setting_compress_svg'] ) || !self::$settings['jptgb_setting_compress_svg'] ){
                return $image;                        
            }
    
            /**
             * Get destination path.
             */
            $destination_path = self::get_destination_path( $image, 'svg' );
    
            /**
             * Return the image array if file already exist.
             */
            if( file_exists( $destination_path ) ){
                $image['src_path'] = $destination_path;
                return $image;
            }
    
            /**
             * Get the source path.
             */
            $src_path = $image['src_path'];
    
            /**
             * Get svg content.
             */
            $svg_content = file_get_contents( $src_path );
    
            if ( false === $svg_content ) {
                self::error_log( "Failed to read SVG content from: $src_path" );
                return $image; // Exit if the SVG content could not be read.
            }
    
            /**
             * Get the svg url.
             */
            $url = $image['src'] ?? '';
    
            /**
             * Call the API to compress the SVG.
             */
            $compressed_svg = self::compress_svg( $svg_content, $url );
    
            if ( false === $compressed_svg ) {
                return $image;
            }
    
            /**
             * Return if compressed SVG is bigger than the original.
             */
            if( strlen( $compressed_svg ) > strlen( $svg_content ) ){
                return $image;
            }
    
            /**
             * Return if failed while writing the file.
             */
            if( !file_put_contents( $destination_path, $compressed_svg ) ){
                return $image;
            }
    
            $image['src_path'] = $destination_path;
        } catch (\Throwable $th) {
            return $original_image_data;
        }

        /**
         * Command to update the image.
         */
        $image['update_it'] = true;

        return $image;
    }

    public static function resize_image( $image ) {
        /**
         * Get device name.
         */
        $device_name = self::$is_mobile_cache_page ? 'mobile' : 'desktop';

        /**
         * Save the original image data.
         */
        $original_image_data = $image;

        try {
            /**
             * Get destination path.
             */
            $resized_url_segment = 'resized';

            /**
             * Append the device name to the resized_url if image width is 'full'
             */
            if('full' === $image['width']){
                $resized_url_segment .= '-' . $device_name;
            }

            $resized_url_segment .= $image['width']  ? '-' . $image['width'] : '';
            $resized_url_segment .= $image['height'] ? 'x' . $image['height'] : '';

            /**
             * Update the resized segment in the image item.
             */
            $image['resized_segment'] = $resized_url_segment;

            /**
             * Get the destination path.
             */
            $destination_path = self::get_tmp_dir_path() . '/' . $image['path_hash'] . '/' . $resized_url_segment . '/' . $image['filename_no_ext'] . '.' . $image['format'];

            /**
             * Update and return the image item if the destination path already exist.
             */
            if( Utils::does_file_exists( $destination_path ) ){
                /**
                 * Get the url of the new modified image.
                 */
                $modified_url = Utils::convert_absolute_path_to_url( $destination_path );
        
                /**
                 * Update the src path in the array.
                 */
                $image['src_path'] = $destination_path;

                /**
                 * Update the image src with the new modified image url.
                 */
                $image['src'] = $modified_url;

                return $image;
            }
        
            /**
             * Return if the sizes are not defined.
             */
            if( !isset( $image['width'] ) ){
                return $image;
            }
    
            /**
             * Get the src path.
             */
            $src_path = $image['src_path'];
    
            /**
             * Get the image editor.
             */
            $editor = wp_get_image_editor( $src_path );
    
            /**
             * Return if editor is not available.
             */
            if( is_wp_error( $editor ) ){
                return $image;
            }
    
            /**
             * Define the full width if is set.
             */
            if( 'full' === $image['width'] ){
                $image['width'] = self::$is_mobile_cache_page ? '430' : '1920';
    
                /**
                 * Get the original dimensions of the image.
                 */
                $size               = $editor->get_size();
                $original_width     = $size['width'];
                $original_height    = $size['height'];
        
                /**
                 * Define new height according to aspect ratio.
                 * Ensure the height is not zero to avoid division by zero error.
                 */
                if( $original_height > 0 ) {
                    $aspect_ratio       = $original_width / $original_height;
                    $aspect_ratio       = round( $aspect_ratio, 2 );
                    $image['height']    = round( $image['width'] / $aspect_ratio );
                }
        
                /**
                 * Do not optimize if the original dimensions are smaller than the target dimensions.
                 */
                if( $original_width < $image['width'] || $original_height < $image['height'] ) {
                    return $image;
                }
            }

            if( !is_numeric( $image['width'] ) ) {
                return $image;
            }
            
            /**
             * Resize the image.
             */
            $resized_file = self::resize_the_image( $editor, $image, $destination_path );
    
            /**
             * Check for errors.
             */
            if( is_wp_error( $resized_file ) ){
                return $image;
            }

            /**
             * Get the url of the new modified image.
             */
            $modified_url = Utils::convert_absolute_path_to_url( $destination_path );
    
            /**
             * Update the src path in the array.
             */
            $image['src_path'] = $destination_path;

            /**
             * Update the image src with the new modified image url.
             */
            $image['src'] = $modified_url;

            /**
             * Set the has_hash flag.
             */
            $image['has_hash'] = true;
        }
        catch (\Throwable $th) {
            return $original_image_data;
        }

        /**
         * Command to update the image.
         */
        $image['update_it'] = true;

        return $image;
    }

    /**
     * Request critical CSS from API.
     *
     * Sends a request to the API endpoint to generate critical CSS.
     * 
     * @param string $css_url     The URL of the CSS file.
     * @param string $target_url  The target URL for which critical CSS is to be generated.
     *
     * @return string|null The URL of the generated critical CSS file or null on failure.
     */
    private static function compress_svg( $svg_content, $url ) {
        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'svg-compressor';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /** 
         * Payload to be sent to the API.
         * Adjusted to send SVG content.
         */
        $body = wp_json_encode( [
            'svgContent'    => $svg_content,
            'originalUrl'   => $url,
        ] );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting the SVG Compresssion: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( 200 !== $status_code ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }

        /**
         * Get the body response.
         */
        $body = wp_remote_retrieve_body( $response );
        
        /**
         * Return the received data.
         */
        return $body;
    }

    /**
     * Converts an image to WebP format via API.
     *
     * Sends a request to the API endpoint to convert an image to WebP format.
     * 
     * @param string $image_content Base64 encoded string of the image.
     * @param string $image_format  The format of the image being converted.
     *
     * @return string|false The Base64 encoded string of the WebP image or false on failure.
     */
    private static function convert_image_to_webp( $image_url, $image_content ) {
        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'image-to-webp';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /**
         * Encode the image content as Base64.
         */
        $base64_image_content = base64_encode( $image_content );

        /** 
         * Payload to be sent to the API.
         * Adjusted to send image content and format.
         */
        $body = wp_json_encode( [
            'imageUrl'      => $image_url,
            'imageContent'  => $base64_image_content,
        ] );

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 60,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting the SVG Compresssion: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( 200 !== $status_code ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }

        /**
         * Get the body response.
         */
        $body = wp_remote_retrieve_body( $response );
        
        /**
         * Return the received data.
         */
        return $body;
    }

    public static function update_image_item( $image ) {
        $resized_path = $image['src_path'] ?? false;

        if( !$resized_path ){
            return $image;
        }

        /**
         * Get the image type.
         */
        $image_type = $image['type'] ?? '';
        $bg_image_types = [ 'bg', 'before', 'after' ];

        // if( 'bg' === $image_type ){

        // if( isset( $image_item['isbg'] ) ){
        if( in_array( $image_type, $bg_image_types ) ){
            /**
             * Get the originial and modified url.
             */
            $resized_url    = Utils::convert_absolute_path_to_url( $resized_path );
            $original_url   = $image['original_src'];
            
            /**
             * If the image url is local, then lets update only the 
             * path section.
             */
            if( self::is_image_url_local( $original_url ) ) {
                $parsed_url         = wp_parse_url( $original_url );
                $relative_url       = $parsed_url['path'];
                $b_parsed_url       = wp_parse_url( $resized_url );
                $b_relative_url     = $b_parsed_url['path'];
    
                $image['original_bg_url'] = $relative_url;
                $image['modified_bg_url'] = $b_relative_url;
                

            /**
             * In case the url is external, then lets update the full url.
             */
            } else {
                $image['original_bg_url'] = $image['original_src'];
                $image['modified_bg_url'] = $resized_url;

            }

        } else {
            $resized_url            = Utils::convert_absolute_path_to_url( $resized_path );
            $modified_tag           = self::update_image_src( $image['tag'], $resized_url );
            $modified_tag           = self::replaceSrcsetAttribute( $modified_tag );
            $image['modified_tag']  = $modified_tag;
        }

        return $image;

        /**
         * Update srcset.
         */
        if( isset( $image['srcset'] ) ){
            foreach( $image['srcset'] as $src_data ){
                /**
                 * Continue to next loop if there is no url.
                 */
                if( !isset( $src_data['url'] ) ){
                    continue;
                }

                /**
                 * Check if we have to update the modified url.
                 */
                if( isset( $src_data[ 'modified_url' ] ) ) {
                    $image['modified_tag'] = str_replace( $src_data[ 'url' ], $src_data[ 'modified_url' ], $image['modified_tag'] );
                }

                /**
                 * Check if we have to add a new url.
                 */
                if( isset( $src_data['to_add'] ) ){
                    $image['modified_tag'] = self::add_srcset_version( $image['modified_tag'], $src_data['url'], $src_data['width'] );
                }
            }
        }

        return $image;
    }

    /**
     * Detects the srcset attribute in an img tag and replaces it with jptgb-data-srcset.
     *
     * This function takes an HTML img tag as input, checks for the presence of the srcset attribute,
     * and if found, renames it to jptgb-data-srcset. The updated img tag is then returned.
     *
     * @param string $imgTag The HTML img tag as a string.
     * @return string The updated img tag with srcset replaced by jptgb-data-srcset, if srcset was present.
     */
    public static function replaceSrcsetAttribute($imgTag) {
        // Check if 'srcset' attribute is present in the img tag
        if (strpos($imgTag, 'srcset') !== false) {
            // Replace 'srcset' with 'jptgb-data-srcset'
            $updatedImgTag = preg_replace('/\ssrcset\s*=\s*("[^"]*"|\'[^\']*\')/', ' data-jptgb-srcset=$1', $imgTag);
            return $updatedImgTag;
        }

        // Return the original img tag if 'srcset' is not found
        return $imgTag;
    }

    /**
     * Adds an additional image version with a specified width to the srcset attribute of an img tag.
     *
     * This function appends a new image source and its width descriptor to the existing srcset attribute
     * of the provided img tag string. It assumes that the srcset attribute is
     * always present and properly formatted in the input string.
     *
     * @param string $img_tag The HTML img tag as a string.
     * @param string $new_src The new image source URL to add.
     * @param int $width The width of the new image source in pixels.
     * @return string The modified img tag with the added srcset source.
     */
    public static function add_srcset_version( $img_tag, $new_src, $width ) {
        // Find the position of the closing quote for the srcset attribute
        $srcset_pos = strpos( $img_tag, ' srcset="' ) + 9;
        $end_pos = strpos( $img_tag, '"', $srcset_pos );

        // Construct the new srcset segment with the width descriptor
        $new_srcset_segment = $new_src . ' ' . $width;

        // Insert the new source at the end of the srcset attribute
        $before = substr( $img_tag, 0, $end_pos );
        $after = substr( $img_tag, $end_pos );

        $new_img_tag = $before . ', ' . $new_srcset_segment . $after;

        return $new_img_tag;
    }

    public static function resize_the_image( $editor, $image, $destination_path ) {
        /**
         * Check if we need to crop or not.
         */
        if( isset( $image['fitStyle'] ) && 'contain' === $image['fitStyle'] ){
            /**
             * Resize the image.
             */
            $resized_file = self::resize_image_contain( $editor, $image['width'], $image['height'], $destination_path );

        } else {
            /**
             * Should we crop the image.
             */
            $crop = isset( $image['crop'] ) ? (bool) $image['crop'] : true;

            /**
             * Resize the image.
             */
            $editor->resize( $image['width'], $image['height'], $crop );

            /**
             * Save the resized image.
             */
            $resized_file = $editor->save( $destination_path );
        }

        return $resized_file;
    }

    /**
     * Resize an image to fit within specified dimensions, simulating "contain" behavior.
     *
     * @param WP_Image_Editor $editor The image editor instance.
     * @param int $max_width The maximum width of the image.
     * @param int $max_height The maximum height of the image.
     * @return array|WP_Error The result of the resizing operation.
     */
    public static function resize_image_contain( $editor, $max_width, $max_height, $destination_path ) {
        // Get the current dimensions of the image
        $size = $editor->get_size();
        $original_width = $size['width'];
        $original_height = $size['height'];

        // Calculate the aspect ratio of the desired dimensions and the original image
        $aspect_ratio = $original_width / $original_height;
        $desired_aspect_ratio = $max_width / $max_height;

        // Calculate the new dimensions based on aspect ratio
        if ($aspect_ratio > $desired_aspect_ratio) {
            // Image is wider than the desired aspect ratio
            $new_width = $max_width;
            $new_height = intval($max_width / $aspect_ratio);
        } else {
            // Image is as wide or narrower than the desired aspect ratio
            $new_height = $max_height;
            $new_width = intval($max_height * $aspect_ratio);
        }

        // Resize the image
        $editor->resize($new_width, $new_height, false);

        // Save the resized image
        $resized_file = $editor->save( $destination_path );

        return $resized_file;
    }


    /**
     * Compresses an image using GD or Imagick extensions.
     *
     * Attempts to compress the given image using the GD library first, and falls back to Imagick if GD is unavailable.
     * Updates the image item on successful compression.
     *
     * @param string $image Path to the source image.
     * @param string $destination_path Path where the compressed image should be saved.
     * @return bool True on success, false on failure.
     */
    public static function compress_image_with_php( $image ) {
        /**
         * Get source path.
         */
        $src_path = $image['src_path'];

        /**
         * Get destination path;
         */
        $destination_path = self::get_destination_path( $image, 'compressed' );

        /**
         * First attempt to compress using GD library.
         */
        if( extension_loaded( 'gd' ) ) {
            $output = self::compress_with_gd( $src_path, $destination_path );

        /**
         * Fallback to Imagick if GD is not available.
         */
        } else if( extension_loaded( 'imagick' ) ) {
            $output = self::compress_with_imagick( $src_path, $destination_path );
        }

        /**
         * Update the src path if the compression was successful.
         */
        if( $output ){
            /**
             * Command to update the image.
             */
            $image['update_it'] = true;
            $image['src_path'] = $destination_path;
        }

        return $image;
    }

    public static function compress_with_imagick( $src_path, $destination_path, $quality = 75 ) {
        /*
         * Ensure Imagick is loaded.
         */
        if ( ! extension_loaded( 'imagick' ) ) {
            throw new \Exception( 'Imagick extension is not loaded.' );
        }
    
        /*
         * Create a temporary file.
         */
        $tmp_file = tempnam( sys_get_temp_dir(), 'compressed_' );
    
        try {
            $image = new \Imagick( $src_path );
    
            /*
             * Check if the necessary methods exist.
             */
            if ( ! method_exists( $image, 'setImageCompressionQuality' ) || ! method_exists( $image, 'writeImage' ) ) {
                throw new \Exception( 'Required Imagick methods are not available.' );
            }
    
            /*
             * Detect the MIME type of the source image.
             */
            $mime = $image->getImageMimeType();
    
            switch ( $mime ) {
                case 'image/jpeg':
                    $image->setImageFormat( 'jpeg' );
                    break;
                case 'image/png':
                    $image->setImageFormat( 'png' );
                    break;
                case 'image/gif':
                    $image->setImageFormat( 'gif' );
                    break;
                case 'image/webp':
                    /*
                     * Additional check for WebP support.
                     */
                    if ( ! in_array( 'WEBP', $image->queryFormats(), true ) ) {
                        throw new \Exception( 'WebP format is not supported by this version of Imagick.' );
                    }
                    $image->setImageFormat( 'webp' );
                    break;
                default:
                    throw new \Exception( 'Unsupported image type: ' . $mime );
            }
    
            /*
             * Set the compression quality.
             */
            $image->setImageCompressionQuality( $quality );
    
            /*
             * Save the image to the temporary path instead of the output_path.
             */
            $image->writeImage( $tmp_file );
    
            /*
             * Compare source and temporary file sizes.
             */
            $source_size    = filesize( $src_path );
            $temp_size      = filesize( $tmp_file );
    
            /*
             * If the temporary file is smaller, replace the original file.
             */
            if ( $temp_size < $source_size ) {

                // Initialize WP_Filesystem if needed.
                if ( ! function_exists( 'WP_Filesystem' ) ) {
                    /** Load the filesystem API */
                    require_once ABSPATH . 'wp-admin/includes/file.php';
                }
                WP_Filesystem();

                /** @var WP_Filesystem_Base $wp_filesystem */
                global $wp_filesystem;

                if ( !$wp_filesystem->move( $tmp_file, $destination_path, true ) ) {
                    /** translators: Error replacing original image with compressed version. */
                    throw new \Exception( 'Failed to replace the original image with the compressed one.' );
                }
            } else {
                /*
                 * Delete the temporary file if it's not smaller.
                 */
                wp_delete_file( $tmp_file );
            }

            /**
             * If the temporary file is smaller, replace the original image with the compressed one.
             */
            if ( $temp_size < $source_size ) {
                // Initialize WP_Filesystem if needed.
                if ( ! function_exists( 'WP_Filesystem' ) ) {
                    /** Load the filesystem API */
                    require_once ABSPATH . 'wp-admin/includes/file.php';
                }
                WP_Filesystem();

                /** @var WP_Filesystem_Base $wp_filesystem */
                global $wp_filesystem;

                if ( !$wp_filesystem->move( $tmp_file, $destination_path, true ) ) {
                    /** translators: Error replacing original image with compressed version. */
                    throw new \Exception( 'Failed to replace the original image with the compressed one.' );
                }
            } else {
                /**
                 * Delete the temporary file if it's not smaller.
                 */
                wp_delete_file( $tmp_file );
            }
    
            $image->clear();
            $image->destroy();
    
            return true;
        } catch ( \Exception $e ) {
            /*
             * Remove the temporary file if it exists.
             */
            if ( file_exists( $tmp_file ) ) {
                wp_delete_file( $tmp_file );
            }
    
            /*
             * Return failure.
             */
            return false; 
        }
    }   
    
    public static function compress_with_gd( $src_path, $destination_path, $quality = 75 ) {
        /**
         * Ensure GD is loaded.
         */
        if ( ! extension_loaded( 'gd' ) ) {
            throw new \Exception( 'GD extension is not loaded.' );
        }
    
        /** 
         * Create a temporary file name. 
         */
        $temp_path = tempnam( sys_get_temp_dir(), 'compressed_' );
    
        try {
            /** 
             * Detect the MIME type of the source image. 
             */
            $info = getimagesize( $src_path );
            if ( ! $info ) {
                throw new \Exception( 'Unable to get image size. Make sure the source path is correct.' );
            }
            $mime = $info['mime'];
    
            /**
             *Load the image based on its MIME type.
             */
            switch ( $mime ) {
                case 'image/jpeg':
                    $image = imagecreatefromjpeg( $src_path );
                    if ( ! $image ) {
                        throw new \Exception( 'Failed to load JPEG image.' );
                    }
                    break;
                case 'image/png':
                    $image = imagecreatefrompng( $src_path );
                    if ( ! $image ) {
                        throw new \Exception( 'Failed to load PNG image.' );
                    }

                    /**
                     *Preserve alpha transparency for PNG. 
                     */
                    imagealphablending( $image, false );
                    imagesavealpha( $image, true );

                    /**
                     * Adjust for PNG quality (0 - no compression, 9 - maximum). 
                     */
                    $quality = 9 - intval( $quality / 11.1111111 );
                    break;
                case 'image/gif':
                    $image = imagecreatefromgif( $src_path );
                    if ( ! $image ) {
                        throw new \Exception( 'Failed to load GIF image.' );
                    }
                    break;
                case 'image/webp':
                    $image = imagecreatefromwebp( $src_path );
                    if ( ! $image ) {
                        throw new \Exception( 'Failed to load WebP image.' );
                    }
                    break;
                default:
                    throw new \Exception( 'Unsupported image type: ' . $mime );
            }
    
            $success = false; // Assume failure until proven otherwise
            switch ( $mime ) {
                case 'image/jpeg':
                    $success = imagejpeg( $image, $temp_path, $quality );
                    break;
                case 'image/png':
                    $success = imagepng( $image, $temp_path, $quality );
                    break;
                case 'image/gif':
                    $success = imagegif( $image, $temp_path );
                    break;
                case 'image/webp':
                    $success = imagewebp( $image, $temp_path, $quality );
                    break;
            }
    
            if ( ! $success ) {
                throw new \Exception( 'Failed to save the compressed image.' );
            }
    
            /**
             * Compare source and temporary file sizes. 
             */
            $source_size    = filesize( $src_path );
            $temp_size      = filesize( $temp_path );
    
            /**
             * If the temporary file is smaller, replace the original file. 
             */
            if ( $temp_size < $source_size ) {
                // Initialize WP_Filesystem if needed.
                if ( ! function_exists( 'WP_Filesystem' ) ) {
                    /** Load the filesystem API */
                    require_once ABSPATH . 'wp-admin/includes/file.php';
                }
                WP_Filesystem();

                /** @var WP_Filesystem_Base $wp_filesystem */
                global $wp_filesystem;

                if ( !$wp_filesystem->move( $temp_path, $destination_path, true ) ) {
                    /** translators: Error replacing original image with compressed version. */
                    throw new \Exception( 'Failed to replace the original image with the compressed one.' );
                }
            } else {
                wp_delete_file( $temp_path ); // Delete the temporary file if it's not smaller
            }
    
            /**
             * Free up memory.
             */
            imagedestroy( $image );
    
            /** 
             * Return success. 
             */
            return true;
        } catch ( \Exception $e ) {
            /**
             * Clean up. 
             */
            if ( file_exists( $temp_path ) ) {
                wp_delete_file( $temp_path );
            }

            /**
             * Free up memory.
             */
            if ( isset( $image ) ) {
                imagedestroy( $image );
            }
    
            /** 
             * Indicate failure. 
             */
            return false;
        }
    }     
    
    public static function update_image_srcset( $img_tag, $target_url, $new_url ) {
        $img_tag = str_replace( $target_url, $new_url, $img_tag );

        return $img_tag;
    }

    /**
     * Updates the src attribute of an image tag string.
     *
     * @param string $img_tag The original image tag.
     * @param string $new_src The new value for the src attribute.
     * @return string The image tag with the updated src attribute.
     */
    public static function update_image_src($img_tag, $new_src) {
        // Find the start of the src attribute
        $src_pos = strpos($img_tag, 'src="');
        if ($src_pos === false) {
            // If src attribute not found, return the original tag
            return $img_tag;
        }

        // Find the start of the URL in src attribute
        $start_url_pos = $src_pos + 5; // 5 is the length of 'src="'

        // Find the end of the URL in src attribute
        $end_url_pos = strpos($img_tag, '"', $start_url_pos);
        if ($end_url_pos === false) {
            // If the end of the URL not found, return the original tag
            return $img_tag;
        }

        // Extract the original src value
        $original_src = substr($img_tag, $start_url_pos, $end_url_pos - $start_url_pos);

        if( strpos( $original_src, 'Q2-2023-Report-Live-Streaming-Trends-768x432.jpg' ) !== false ){
            $stop = 1;
        }

        // Reconstruct the image tag with the new src
        $new_img_tag = substr($img_tag, 0, $start_url_pos) . $new_src . substr($img_tag, $end_url_pos);

        // Insert the origina tag.
        $new_img_tag = str_replace( ' src="', ' data-jptgb-original="' . $original_src . '" src="', $new_img_tag );

        return $new_img_tag;
    }

    /**
     * Updates the background-image URL in the style attribute of an HTML element.
     *
     * @param string $element_html The original HTML element.
     * @param string $new_url The new URL for the background-image.
     * @return string The HTML element with the updated background-image URL.
     */
    public static function update_background_image_url($element_html, $new_url) {
        // Find the start of the style attribute
        $style_pos = strpos($element_html, 'style="');
        if ($style_pos === false) {
            // If style attribute not found, return the original HTML
            return $element_html;
        }

        // Find the start of the background-image property in the style attribute
        $bg_image_pos = strpos($element_html, 'background-image:', $style_pos);
        if ($bg_image_pos === false) {
            // If background-image not found, return the original HTML
            return $element_html;
        }

        // Find the start of the URL in the background-image property
        $start_url_pos = strpos($element_html, 'url(', $bg_image_pos) + 4; // 4 is the length of 'url('

        // Find the end of the URL in the background-image property
        $end_url_pos = strpos($element_html, ')', $start_url_pos);
        if ($end_url_pos === false) {
            // If the end of the URL not found, return the original HTML
            return $element_html;
        }

        // Reconstruct the HTML element with the new background-image URL
        $new_element_html = substr($element_html, 0, $start_url_pos) . $new_url . substr($element_html, $end_url_pos);

        return $new_element_html;
    }

    /**
     * Convert a local file path to its corresponding URL in WordPress.
     *
     * @param string $path The local file path.
     * @return string The URL corresponding to the file path.
     */
    public static function convert_path_to_url( $path ) {
        /**
         * Get the upload directory information.
         */
        $upload_dir     = wp_upload_dir();
        $upload_baseurl = trailingslashit( is_ssl() ? str_replace('http://', 'https://', $upload_dir['baseurl']) : $upload_dir['baseurl'] );
        $upload_basedir = trailingslashit($upload_dir['basedir']);

        /**
         * Check if the path is in the uploads directory.
         */
        if (strpos($path, $upload_basedir) !== false) {
            return str_replace($upload_basedir, $upload_baseurl, $path);
        }

        /**
         * Get the theme directory information.
         */
        $theme_dir_url = trailingslashit(get_stylesheet_directory_uri());
        $theme_dir_path = trailingslashit(get_stylesheet_directory());

        /**
         * Check if the path is in the theme directory.
         */
        if (strpos($path, $theme_dir_path) !== false) {
            return str_replace($theme_dir_path, $theme_dir_url, $path);
        }

        /**
         * Return original path if it doesn't match uploads or theme directories.
         */
        return $path;
    }

    /**
     * Updates or inserts the background image URL in the style attribute of a given HTML element string.
     *
     * @param string $element The HTML element as a string.
     * @param string $backgroundImageUrl The URL of the background image to insert or update.
     * @return string The modified HTML element string.
     */
    public static function updateElementBackgroundImage($element, $backgroundImageUrl) {
        // Find the position of the opening tag end
        $tagEndPos = strpos($element, '>');
        if ($tagEndPos === false) {
            return $element; // Malformed HTML 
        }

        // Extract the opening tag
        $openingTag = substr($element, 0, $tagEndPos + 1);

        // Check if the style attribute is present
        $styleAttrPos = strpos($openingTag, 'style=');
        if ($styleAttrPos !== false) {
            // Style attribute is present, update or append the background-image
            $styleStartPos = strpos($openingTag, '"', $styleAttrPos) + 1;
            $styleEndPos = strpos($openingTag, '"', $styleStartPos);
            $styleContent = substr($openingTag, $styleStartPos, $styleEndPos - $styleStartPos);
            $styleContent = html_entity_decode( $styleContent, ENT_QUOTES, 'UTF-8' );

            // Check if background-image is already present
            $backgroundImagePos = strpos($styleContent, 'background-image:');
            if ($backgroundImagePos !== false) {
                // Update the existing background-image
                $semicolonPos = strpos($styleContent, ';', $backgroundImagePos);
                if ($semicolonPos === false) {
                    $semicolonPos = strlen($styleContent);
                }
                $styleContent = substr_replace(
                    $styleContent,
                    "background-image: url('$backgroundImageUrl') !important;",
                    $backgroundImagePos,
                    $semicolonPos - $backgroundImagePos + 1
                );
            } else {
                // Append the new background-image
                $styleContent .= " background-image: url('$backgroundImageUrl') !important;";
            }

            // Replace the old style attribute with the updated one
            $openingTag = substr_replace($openingTag, $styleContent, $styleStartPos, $styleEndPos - $styleStartPos);
        } else {
            // Style attribute is not present, insert a new one
            $openingTag = substr_replace($openingTag, " style=\"background-image: url('$backgroundImageUrl') !important;\"", $tagEndPos, 0);
        }

        // Replace the old opening tag with the new one in the element
        $element = substr_replace($element, $openingTag, 0, $tagEndPos + 1);

        return $element;
    }

    public static function improve_images_performance( $cache_file_content, $url_hash ) {
        try {
            /**
             * Get all images tags.
             */
            $image_items = self::extractImgTagsFromString( $cache_file_content );
    
            /**
             * Extract the images attributes.
             */
            $image_items = self::parseDataJptgbAttributesAndSrc( $image_items );
    
            /**
             * Extract bg images.
             */
            $bg_items       = self::extractTagsWithDataAttributeFromString( $cache_file_content );
            $bg_items       = self::parseDataJptgbAttributesAndSrc( $bg_items );
            $image_items    = array_merge( $bg_items, $image_items );
    
            /**
             * Resize images.
             */
            $image_items = self::reszie_and_optimize_images( $image_items, $url_hash );

            /**
             * Elements with before and or after background
             */
            $elements_with_boa = [];
    
            /**
             * Update modified images in the cache contens.
             */
            foreach( $image_items as $image_item ){
                if( !isset( $image_item['modified_tag'] ) && !isset( $image_item['modified_bg_url'] ) ){
                    continue;
                }
    
                $original_tag = $image_item['tag'] ?? '';
                $modified_tag = $image_item['modified_tag'] ?? '';  
                


                if( strpos( $original_tag, 'arkets-inside' ) !== false ){
                    $stop = 1;
                }
                
                /**
                 * Lets create a preload link if the setpreload attribute is set or the image is the max paint area.
                 */
                if( ( strpos( $original_tag, 'data-jptgbsetpreload="1"' ) !== false ) || ( $image_item['paintArea'] ?? 0 ) === self::$largest_paint_area ){
                    $cache_file_content = self::set_preload_link( $image_item, $cache_file_content );
                }
                // if( ( isset( $image_item['setpreload'] ) && $image_item['setpreload'] ) || ( isset( $image_item['ismaxpaintarea'] ) && $image_item['ismaxpaintarea'] ) ){
                    // $cache_file_content = self::set_preload_link( $image_item, $cache_file_content );
                // }
    
                /**
                 * Check if we have to replace only on the css critical style tag.
                 */
                $image_type     = $image_item['type'] ?? '';
                $bg_image_types = [ 'bg', 'before', 'after' ];

                if( 'before' === $image_type || 'after' === $image_type ){
                    
                    // continue;

                    $unique_class_selector = $image_item['unique_class_selector'] ?? null;

                    if( !$unique_class_selector ) {
                        // continue;
                        $unique_class_selector = uniqid();
                    }

                    /**
                     * Upadte the bacground image.
                     */
                    $element_tag        = $image_item['tag'];
                    $modified_bg_url    = $image_item['modified_bg_url'];

                    /**
                     * Check if the bg element was defined before
                     */
                    if( isset( $elements_with_boa[ $unique_class_selector ] ) ){
                        $element_tag = $elements_with_boa[ $unique_class_selector ];
                        $class_selector = 'jptgb_bg_'. $image_type .'_' . $unique_class_selector;
                        $new_element = self::attach_pseudo_bg( $element_tag , $modified_bg_url, $class_selector, $image_type );
                    } else {
                        $class_selector = 'jptgb_bg_'. $image_type .'_' . $unique_class_selector;
                        $new_element = self::attach_pseudo_bg( $element_tag , $modified_bg_url, $class_selector, $image_type );

                        $before_unique_class = 'jptgb_bg_before_' . $unique_class_selector;
                        $after_unique_class = 'jptgb_bg_after_' . $unique_class_selector;
                        $new_element = self::add_class_to_opening_tag( $new_element, $before_unique_class );
                        $new_element = self::add_class_to_opening_tag( $new_element, $after_unique_class );

                        $elements_with_boa[ $unique_class_selector ] = $new_element;

                        $element_tag =  !empty( self::$original_tags[$unique_class_selector] ) ? self::$original_tags[$unique_class_selector] : $element_tag;
                    }

                    /**
                     * Update changes on the cache content.
                     */
                    $cache_file_content = self::replace_in_entire_document( $element_tag, $new_element, $cache_file_content );



                } else if ( 'bg' === $image_type ){
                    /**
                     * Upadte the bacground image.
                     */
                    $element_tag        = $image_item['tag'];
                    $modified_bg_url    = $image_item['modified_bg_url'];

                    /**
                     * Update the background image in the element tag.
                     */
                    $new_element = self::updateElementBackgroundImage( $element_tag, $modified_bg_url );

                    if( !empty( $image_item['unique_class_selector'] ) ){
                        $unique_class_selector = $image_item['unique_class_selector'];
                        $before_unique_class = 'jptgb_bg_before_' . $image_item['unique_class_selector'];
                        $after_unique_class = 'jptgb_bg_after_' . $image_item['unique_class_selector'];
                        $new_element = self::add_class_to_opening_tag( $new_element, $before_unique_class );
                        $new_element = self::add_class_to_opening_tag( $new_element, $after_unique_class );

                        $elements_with_boa[$image_item['unique_class_selector']] = $new_element;

                        $element_tag =  !empty( self::$original_tags[$unique_class_selector] ) ? self::$original_tags[$unique_class_selector] : $element_tag;
                    }
    
                    /**
                     * Update changes on the cache content.
                     */
                    $cache_file_content = self::replace_in_entire_document( $element_tag, $new_element, $cache_file_content );
    
                /**
                 * If it is not bg then it is a img tag
                 */
                } else {
                    /**
                     * Check if we have to set explicit width and height.
                     */
                    if( isset( self::$settings['jptgb_setting_set_explicit_w_and_h'] ) && self::$settings['jptgb_setting_set_explicit_w_and_h'] ){
                        $modified_tag = self::set_explicit_width_and_height( $modified_tag, $image_item );
                    }
        
                    /**
                     * Update changes on the cache content.
                     */
                    $cache_file_content = self::replace_in_entire_document( $original_tag, $modified_tag, $cache_file_content );
                }
                
            }
    
            return $cache_file_content;
        } catch (\Throwable $th) {
            $error_message = $th->getMessage();
            BaseController::$logs->register( 'Something failed while trying to improve the images | Error message: ' . $error_message, 'error' );
            
            throw $th;
        }
    }

    /**
     * Add a CSS class to an opening HTML tag string.
     *
     * @param string $opening_tag The opening HTML tag (e.g. '<div id="foo">').
     * @param string $class_name  The class name to add.
     * @return string             The modified opening tag with the class added.
     */
    public static function add_class_to_opening_tag( string $opening_tag, string $class_name ): string {
        /** Sanitize the class name to valid HTML class format. */
        $class = sanitize_html_class( $class_name );

        /** If a class attribute already exists, append the new class. */
        if ( preg_match( '/\sclass=[\'"]([^\'"]*)[\'"]/', $opening_tag, $matches ) ) {
            $new_classes = $matches[1] . ' ' . $class;
            $opening_tag = preg_replace(
                '/\sclass=[\'"][^\'"]*[\'"]/',
                ' class="' . esc_attr( $new_classes ) . '"',
                $opening_tag,
                1
            );
        } else {
            /** Otherwise, insert a class attribute right after the tag name. */
            $opening_tag = preg_replace(
                '/^<([a-z0-9]+)/i',
                '<$1 class="' . esc_attr( $class ) . '"',
                $opening_tag,
                1
            );
        }

        return $opening_tag;
    }

    /**
     * Attach a pseudo-element background image to an HTML opening tag.
     *
     * Accepts an opening tag string, a background-image URL, and a flag
     * indicating whether to use ::before or ::after. Prepend a scoped
     * <style> block and injects a unique class (prefixed with
     * "jptgb_before_bg_" or "jptgb_after_bg_") into the tag.
     *
     * @param string $opening_tag The opening HTML tag (e.g., '<div id="foo">').
     * @param string $image_url   URL of the background image.
     * @param string $pseudo      Which pseudo-element: 'before' or 'after'.
     * @return string Modified output: <style> + opening tag with injected class.
     */
    public static function attach_pseudo_bg( string $opening_tag, string $image_url, $unique_class, string $pseudo = 'before', bool $inject_class_name = false ): string {
        // Sanitize the image URL.
        $url = esc_url_raw( $image_url );

        // Determine pseudo-element and class prefix.
        $pseudo_el = 'before';
        if ( 'after' === strtolower( $pseudo ) ) {
            $pseudo_el = 'after';
        }

        // Build CSS: only content and background-image.
        $css  = ".{$unique_class}::{$pseudo_el} {\n";
        $css .= "    background-image: url('{$url}');\n";
        $css .= "}";

        // Wrap CSS in a <style> tag.
        $style_tag = '<style type="text/css">' . wp_strip_all_tags( $css ) . '</style>';

        if( $inject_class_name ){
            // Inject the unique class into the opening tag.
            if ( preg_match( '/class=["\']([^"\']*)["\']/', $opening_tag, $matches ) ) {
                // Append to existing class attribute.
                $new_classes  = $matches[1] . ' ' . $unique_class;
                $escaped     = esc_attr( trim( $new_classes ) );
                $opening_tag = preg_replace(
                    '/class=["\'][^"\']*["\']/',
                    "class=\"{$escaped}\"",
                    $opening_tag,
                    1
                );
            } else {
                // No class attribute: insert one after tag name.
                $opening_tag = preg_replace(
                    '/^<([a-z0-9]+)/i',
                    "<\\1 class=\"" . esc_attr( $unique_class ) . "\"",
                    $opening_tag,
                    1
                );
            }
        }

        // Return the style block plus the modified opening tag.
        return $style_tag . $opening_tag;
    }


    private static function set_preload_link( $image_item, $cache_file_content ) {
        /**
         * Get modfied path.
         */
        $src_path = $image_item['src_path'] ?? '';

        /**
         * Return if no src path is found.
         */
        if( !$src_path ){
            return $cache_file_content;
        }

        /**
         * Convert the src path to an url.
         */
        $modified_url = Utils::convert_absolute_path_to_url( $src_path );

        /**
         * Define the preload tag.
         */
        $preload_tag = '<link rel="preload" as="image" href="'. $modified_url .'" fetchpriority="high">';

        /**
         * Insert the preload tag into at the end of the head.
         */
        $cache_file_content = preg_replace( '/(<head(\s+[^>]*>|>))/', '$1' . $preload_tag, $cache_file_content, 1 );

        /**
         * Return the modified cache contents.
         */
        return $cache_file_content;
    }

    private static function set_explicit_width_and_height( $modified_tag, $image_item ) {
        $width  = $image_item['width'];
        $height = $image_item['height'];

        /**
         * Skip if the width is set to full widht.
         */
        if( '1920' == $width ){
            return $modified_tag;
        }

        /**
         * If the image do not have the style attribute, then lets add 
         * the explicit width and height in it.
         */
        if( !self::img_tag_has_style_attribute( $modified_tag ) ){
            $modified_tag = self::add_inline_css_to_img_tag( $modified_tag, $width, $height );
        }

        return $modified_tag;
    }

    /**
     * Adds explicit width and height as inline CSS and a custom attribute to an <img> tag.
     *
     * @param string $img_tag The <img> tag as a string.
     * @param int $width The width to set for the image, in pixels.
     * @param int $height The height to set for the image, in pixels.
     * @return string The modified <img> tag with inline CSS for width and height, and a custom data attribute.
     */
    private static function add_inline_css_to_img_tag( $img_tag, $width, $height ) {
        /** Define the inline CSS for width and height */
        $inline_css = ' style="width: ' . $width . 'px; height: ' . $height . 'px;"';
        
        /** Define the custom data attribute */
        $data_attribute = ' data-jptgb-inlinedcss="1"';
        
        /** Insert the inline CSS and data attribute before the closing ">" of the <img> tag */
        $modified_img_tag = rtrim($img_tag, '>') . $inline_css . $data_attribute . '>';
        
        return $modified_img_tag;
    }

    /**
     * Checks if an <img> tag string contains a style attribute.
     *
     * @param string $img_tag The <img> tag as a string.
     * @return bool True if the style attribute is present, false otherwise.
     */
    private static function img_tag_has_style_attribute( $img_tag ) {
        /** Check if 'style=' is present in the <img> tag string */
        if ( false !== strpos( $img_tag, 'style=' ) ) {
            /** The style attribute is present */
            return true;
        } else {
            /** The style attribute is not present */
            return false;
        }
    }

    ///wp-content/uploads/2024/02/mann-und-frau-einem-Tisch-auf-Workation-im-freien-600x400.webp
    ///wp-content/uploads/jptgb/images/mann-und-frau-einem-Tisch-auf-Workation-im-freien-600x400-webp-698797ff434ba57e.webp

    /**
     * Replaces a target string with a replacement string within a <style> tag that has a specific ID, while accounting for additional attributes in the tag.
     *
     * @param string $target The string to be replaced.
     * @param string $replacement The string to replace with.
     * @param string $html The HTML document as a string.
     * @return string The modified HTML document.
     */
    public static function replace_in_critical_css_section( $target, $replacement, $html ) {
        /** Identify the beginning of the <style> tag pattern with the specified ID */
        $style_tag_start_pattern = '<style id="jptgb-critical-css"';
        /** Identify the end of the opening <style> tag, considering additional attributes */
        $style_tag_close = '>';

        /** Locate the start of the <style> tag with the specified ID */
        $start_pos = strpos( $html, $style_tag_start_pattern );
        if ( false === $start_pos ) {
            /** If the <style> tag is not found, return the original HTML */
            return $html;
        }

        /** Find the position of the closing bracket of the opening <style> tag */
        $style_tag_open_end_pos = strpos( $html, $style_tag_close, $start_pos );
        if ( false === $style_tag_open_end_pos ) {
            /** If the closing bracket is not found, return the original HTML */
            return $html;
        }

        /** Adjust to get content start position after the closing bracket of the opening tag */
        $content_start_pos = $style_tag_open_end_pos + strlen( $style_tag_close );

        /** Locate the end of the <style> tag */
        $end_pos = strpos( $html, '</style>', $content_start_pos );
        if ( false === $end_pos ) {
            /** If the closing tag is not found after the opening tag, return the original HTML */
            return $html;
        }

        /** Extract the content inside the <style> tag */
        $style_content = substr( $html, $content_start_pos, $end_pos - $content_start_pos );

        /** Replace the target string with the replacement string in the extracted content */
        $modified_style_content = str_replace( $target, $replacement, $style_content );

        /** Reassemble the HTML document with the modified <style> tag content */
        $before_style = substr( $html, 0, $content_start_pos );
        $after_style = substr( $html, $end_pos );
        $modified_html = $before_style . $modified_style_content . $after_style;

        return $modified_html;
    }

    public static function replace_in_entire_document( $target, $replacement, $content ) {
        if( strpos( $content, $target ) !== false ) {
            return str_replace( $target, $replacement, $content );
        }

        return $content;
    }
}
