<?php

/**
 * The Scripts Controller Class.
 *
 * @author Yaidier Perez
 * */

namespace Jptgb\Controllers;

use Jptgb\Resources\Utils;

class ImmediateDomLoadController extends BaseController {

    public static function init() {
        /**
         * Clear the cache if we just 
         * turn on/off the LDL controller.
         */
        if( in_array( 'jptgb_setting_immediate_dom_load', self::$changed_settings ) || in_array( 'jptgb_setting_idl_for_hidden_elements', self::$changed_settings ) ) {
            CacheController::flush_all_cache_files();
        }

        /**
         * If Later Dom Load is deactiavted, then just return.
         * 
         * @see Wp Dashboard -> Theme Settings -> Performance Settings
         */
        if( !isset( self::$settings['jptgb_setting_immediate_dom_load'] ) || !self::$settings['jptgb_setting_immediate_dom_load'] ) {
            return;
        }

        add_action( 'wp', function() {
            /**
             * Do not apply Optimizations for the backend or logged-in users.
             */
            if( is_admin() || is_user_logged_in() ) {
                return;
            }

            /*
             * If this is an autosave, our form has not been submitted,
             * so we don't want to do anything.
             */
            if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) || isset( $_REQUEST['bulk_edit'] ) ) {
                return;
            } 

            /**
             * Return if the jptgbgeneratecache parameter is not set on the URL.
             */
            if( !self::$is_api_request ) {
                return;
            }

            // add_action( 'wp_footer', [self::class, 'later_dom_load__callback'], PHP_INT_MAX - 1 );
        } );
    }

    

    /**
     * Processes HTML content to replace specific comment tags with placeholders and handles the content within those tags.
     *
     * @param string $html The HTML content to be processed.
     * @return string Processed HTML content with replaced tags and placeholders.
     */
    public static function create_immediate_dom_load( $hashed_url, $html, $device_name ) {
        /**
         * Deifne the templates string
         */
        $idl_templates = '';

        $pattern = '/<!--JPTGBLDLSTART\[(.*?)\]-->/';
        while ( preg_match( $pattern, $html, $matches, PREG_OFFSET_CAPTURE ) ) {
            $full_match = $matches[0][0];
            $tag_name   = $matches[1][0];
            $start_pos  = $matches[0][1];

            if ( $tag_name ) {
                /**
                 * Generate a unique ID.
                 */
                $unique_id = uniqid('jptgb_idl_');

                /**
                 * Save the starting position for later extraction.
                 */
                $adjusted_start_pos = $start_pos + strlen($full_match);

                /**
                 * Find the end marker based on the tag name.
                 */
                $end_marker = "<!--JPTGBLDLEND[$tag_name]-->";
                $end_pos = strpos($html, $end_marker, $adjusted_start_pos);

                if ( $end_pos !== false ) {
                    /**
                     * Extract the content and store it in the static array.
                     */
                    $content_length     = $end_pos - $adjusted_start_pos;
                    $wrapped_content    = substr( $html, $adjusted_start_pos, $content_length );
                    $idl_templates      .= '<template id="'. $unique_id .'">' . $wrapped_content . '</template>';

                    /**
                     * Check if replacement has to be hidden.
                     */
                    $hidden_style = ( strpos( $wrapped_content, 'data-jptgbishidden="1"' ) !== false ) ? 'style="display: none;"' : '';

                    /**
                     * Create a placeholder element with the unique ID.
                     */
                    $replacement = "<$tag_name class=\"jptgb_ldl_placeholder\"  $hidden_style data-templateid=\"$unique_id\"></$tag_name>";

                    /**
                     * Replace the full matched portion with the placeholder.
                     */
                    $html = substr_replace($html, $replacement, $start_pos, $end_pos + strlen($end_marker) - $start_pos);
                }
            }

            if ( empty( $matches[0] ) ) {
                break;
            }
        }

        return [ 'html' => $html, 'templates' => $idl_templates ];
    }

    /**
     * Replace `<script` with `<script data-jptgb-ldlscript="1"` within sections marked by
     * `<!--JPTGBLDLREMOVE` and `JPTGBLDLREMOVE-->`.
     *
     * @param string $content The original content string.
     * @return string The modified content with replacements made within specified sections.
     */
    private static function replace_script_within_markers( string $content ): string {
        $start_marker = '<!--JPTGBLDLREMOVE';
        $end_marker = 'JPTGBLDLREMOVE-->';
        $start_pos = 0;

        while (($start_pos = strpos($content, $start_marker, $start_pos)) !== false) {
            $end_pos = strpos($content, $end_marker, $start_pos);
            if ($end_pos === false) {
                break; // Exit the loop if there's no closing marker.
            }
        
            // Calculate the length of the section including the end marker.
            $section_length = $end_pos - $start_pos + strlen($end_marker);
            $section_to_modify = substr($content, $start_pos, $section_length);
        
            // Perform the replacement within this section.
            $modified_section = str_replace('<script', '<script data-jptgb-ldlscript="1"', $section_to_modify);
        
            // Replace the original section with the modified section in the content.
            $content = substr_replace($content, $modified_section, $start_pos, $section_length);
        
            // Update the start position to move past the modified section.
            // This takes into account the increased length of the content.
            $start_pos += strlen($modified_section);
        }

        return $content;
    }

    /**
     * Removes all HTML comments from a given string.
     *
     * @param string $string The input string from which HTML comments will be removed.
     * @return string The string with HTML comments removed.
     */
    private static function remove_html_comments( string $string ): string {
        // Regular expression to match HTML comments
        $pattern = '/<!--.*?-->/s';
        
        // Remove HTML comments from the string
        $cleaned_string = preg_replace( $pattern, '', $string );
        
        return $cleaned_string;
    }  

    /**
     * Removes all HTML comments within sections marked by specific start and end strings.
     *
     * This function manually parses the HTML string to find sections between the specified start and end markers.
     * Within these sections, it removes all instances of HTML comment delimiters.
     *
     * Note: This approach is simplistic and designed for specific use cases. It might not handle complex HTML structures well.
     *
     * @param string $html The HTML document as a string.
     * @return string The modified HTML document with comments removed within marked sections.
     */
    public static function remove_comments_from_marked_sections( $html ) {
        /** Define start and end markers */
        $start_marker   = '<!--JPTGBLDLSTART-->';
        $end_marker     = '<!--JPTGBLDLEND-->';
        $start_pos = 0;
        
        while ( false !== ( $start_pos = strpos( $html, $start_marker, $start_pos ) ) ) {
            $end_pos = strpos( $html, $end_marker, $start_pos );
            if ( false === $end_pos ) {
                /** End marker not found, break the loop */
                break;
            }

            $section_start  = $start_pos + strlen( $start_marker );
            $section_length = $end_pos - $section_start;
            $section        = substr( $html, $section_start, $section_length );

            /** Manually remove comment delimiters within the section */
            $clean_section = str_replace( '<!--', '', $section );
            $clean_section = str_replace( '-->', '', $clean_section );

            /** Replace the original section with the cleaned section in the HTML */
            $html = substr_replace( $html, $clean_section, $section_start, $section_length );

            /** Move past this section for the next iteration */
            $start_pos = $end_pos + strlen( $end_marker );
        }

        return $html;
    }
}