<?php

namespace Jptgb\Controllers;

use Jptgb\Controllers\CacheController;
use Jptgb\Controllers\CriticalCssController;
use Jptgb\Controllers\ScriptsController;
use Jptgb\Controllers\ImmediateDomLoadController;
use Jptgb\Controllers\LazyDomLoadController;
use Jptgb\Controllers\AdminToolsController;
use Jptgb\Controllers\AjaxController;
use Jptgb\Controllers\CompabilityController;
use Jptgb\Resources\CloudflareHelper;
use Jptgb\Resources\SitegroundHelper;
use Jptgb\Resources\Utils;
use WP_Post;

use function GuzzleHttp\Psr7\try_fopen;

/**
 * The Main Controller Class
 *
 * @author Yaidier Perez
 * */

class MainController extends BaseController {

    public static $urls_purge = [];

    public static function init() {
        CacheController::init();
        CriticalCssController::init();
        ScriptsController::init();
        FontsController::init();
        ImagesController::init();
        DebuggerController::init();
        ImmediateDomLoadController::init();
        LazyDomLoadController::init();
        AdminToolsController::init();
        CompabilityController::init();

        self::setup_wp_endpoint();
    }

    public static function setup_wp_endpoint() {
        add_action( 'rest_api_init', [ self::class, 'register_my_custom_endpoint' ] );
    }

    /**
     * Register a REST API endpoint for receiving notifications.
     */
    public static function register_my_custom_endpoint() {
        // jptgb/v1/notify
        register_rest_route( self::$endpoint_namespace, self::$notifications_endpoint, array(
            'methods'             => \WP_REST_Server::CREATABLE,
            'callback'            => [self::class, 'handle_notification'],
            'permission_callback' => [self::class, 'check_api_key_permission'],
        ));

        register_rest_route( 'jptgb/v1', '/handshake', [
                'methods'             => \WP_REST_Server::CREATABLE,
                'callback'            => [ self::class, 'handle_handshake' ],
                'permission_callback' => '__return_true',
            ]
        );
    }

    /**
     * Callback for domain verification.
     *
     * @param WP_REST_Request $request Incoming REST request (expects body param 'token').
     * @return WP_REST_Response      Returns the token on success, or a 403 error.
     */
    public static function handle_handshake( \WP_REST_Request $request ): \WP_REST_Response {
        $token        = sanitize_text_field( $request->get_param( 'token' ) );
        $stored_token = (string) self::get_setting( 'verify_token' );

        if ( hash_equals( $stored_token, $token ) ) {
            return new \WP_REST_Response( $token, 200 );
        }

        return new \WP_REST_Response( [ 'error' => 'Invalid token.' ], 403 );
    }

    /**
     * Checks if the API key provided in the request is valid.
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return bool True if the API key is valid, otherwise false.
     */
    public static function check_api_key_permission( $request ) {
        /**
         * Retrieve the API key from headers.
         */
        $auth_header    = $request->get_header('authorization');
        $received_key   = str_replace('Bearer ', '', $auth_header);  

         /**
         * Get the api key.
         */
        $api_key = self::$settings['jptgb_setting_api_key'] ?? '';

        return $received_key === $api_key;
    }

    public static function handle_error_message( $post_id, $message ) {
        /**
         * Get wp object from url.
         */
        // $wp_object = AjaxController::get_object_by_url( $requested_url );
        AjaxController::update_cache_status( $post_id, 'error', $message, 0 );
    }

    /**
     * Extracts and removes a consumer item from the array by its post ID.
     *
     * @param array<int,array<string,mixed>> $consumers Array of consumer items, passed by reference.
     * @param int                             $post_id   The post ID to match.
     * @return array<string,mixed>|null      The extracted consumer item, or null if not found.
     */
    public static function extract_consumer_item_by_post_id( array &$consumers, int $post_id ): ?array {
        foreach ( $consumers as $index => $item ) {
            if ( isset( $item['id'] ) && (int) $item['id'] === $post_id ) {
                // ** Found the item; remove it and return it */
                $extracted = $item;
                array_splice( $consumers, $index, 1 );
                return $extracted;
            }
        }

        // ** No matching item found */
        return null;
    }

    /**
     * Handle incoming notifications for the endpoint.
     *
     * @param WP_REST_Request $request Request object containing JSON payload from external service.
     * @return WP_REST_Response Response object with status code and message.
     */
    public static function handle_notification( $request ) {
        /**
         * Get the JSON payload from the request body.
         */
        $parameters = $request->get_json_params();

        /**
         * Check if we are ping pong the wp rest api.
         */
        if( isset( $parameters['endPointTest'] ) && 'ping' === $parameters['endPointTest'] ){
            return new \WP_REST_Response(['message' => 'pong'], 200);
        }

        $action = $parameters['action'] ?? '';
        $post_id = $parameters['post_id'] ?? '';

        if( 'processing_task' === $action && !empty( $parameters['post_id'] ) ) {
            $post_id = $parameters['post_id'];

            

            AjaxController::update_cache_status( $post_id, 'working-on', 'Creating the Cache Files', 30 );

            return;
        }

        /**
         * Get the payload data.
         */
        $contents_to_fetch  = isset( $parameters['contents_to_fetch']) ? $parameters['contents_to_fetch'] : null;
        $requested_url      = isset( $parameters['url']) ? $parameters['url'] : null;
        $post_id            = $parameters['post_id'] ?? null;
        $api_user_id        = $parameters['api_user_id'] ?? null;

        if( !$requested_url || !$contents_to_fetch ){
            self::handle_error_message( $post_id, $parameters[ 'message' ] );
            /**
             * If the data is not what was expected, you can return an error message
             */
            return new \WP_REST_Response(['message' => 'Notification received and processed successfully'], 200);
        }

        /**
         * Update the current requested url.
         */
        self::$requested_url = $requested_url;

        $response_to_api = false;

        /**
         * Determine if the call is coming from generate cache or critical css.
         */
        if( isset( $contents_to_fetch[ 'critical_css' ] ) ){
            $device = $contents_to_fetch['device'] ?? '';
            $critical_endpoint_handler_response = self::handle_critical_css_endpoint_call( $contents_to_fetch );

        } else {

            /**
             * Register the status.
             */
            BaseController::$logs->register( 'Notification Endpoint called for: ' . esc_url( self::$requested_url ), 'info' );
    
            /**
             * Handle error if exist.
             */
            if( isset( $parameters[ 'status' ] ) && 'error' === $parameters[ 'error' ] ){
                self::handle_error_message( $post_id, $parameters[ 'message' ] );
                return new \WP_REST_Response(['message' => 'Notification received and processed successfully'], 200);
            }

            /**
             * Get all the contents.
             */
            foreach( $contents_to_fetch as $content_to_fetch ){
                /**
                 * Get data
                 */
                $realtive_url       = $content_to_fetch['url'];
                $url_hash           = $content_to_fetch['url_hash'];
                $type               = $content_to_fetch['type'];
                $device             = $content_to_fetch['device'] ?? '';
                $aboulute_url       = trailingslashit( self::$webspeed_api_base ) . $realtive_url;
    
                /**
                 * Define is_mobile_cache_page.
                 */
                self::$is_mobile_cache_page = ( 'mobile' === $device ) ? true : false;
    
                /**
                 * Get the files contents.
                 */
                $response = wp_remote_get( $aboulute_url, [
                    'sslverify' => false,
                ] );
        
                if ( is_wp_error( $response ) ) {
                    return new \WP_REST_Response(['message' => $response->get_error_message()], 400);
                }
    
                /**
                 * Check the response code.
                 */
                $response_code = wp_remote_retrieve_response_code( $response );
    
                if( 200 !== $response_code ){
                    /**
                     * Get wp object from url.
                     */
                    $wp_object = AjaxController::get_object_by_url( $requested_url );
    
                    /**
                     * Get the post id if available.
                     */
                    $post_id = $wp_object instanceof WP_Post ? $wp_object->ID : false;
    
                    /**
                     * Build the error message.
                     */
                    $error_message = '['. $device . '] We could\'nt get the bucket content at ' . esc_url( $aboulute_url ) . ' for this url: ' . esc_url( $requested_url );
    
                    /** 
                     * Log the status. 
                     */
                    BaseController::$logs->register( $error_message, 'error' );
    
                    /**
                     * Update cache status.
                     */
                    AjaxController::update_cache_status( $wp_object, 'error', $error_message, 0 );
    
                    /**
                     * Remove this from the consumer pool if exist.
                     */
                    Utils::add_a_task_ready_to_tasks_overview( $post_id );
    
                    return new \WP_REST_Response(['message' => $error_message], 400);
                }
    
                /**
                 * The request was successful, process the data.
                 */
                $content = wp_remote_retrieve_body( $response );
    
                try {
                    if( 'html' === $type ){
                        $insert_response = self::insert_html_content( $requested_url, $url_hash, $content, $device, $post_id, $api_user_id );
                    } else if( 'image' === $type ){
                        
                    } else {
                        // ....
                    } 
                } catch (\Throwable $th) {
                    $error_message  = $th->getMessage();
                    $error_code     = $th->getCode();
    
                    /**
                     * Register the status.
                     */
                    BaseController::$logs->register( '['. $device .'] Something failed when inserting the content for the url: ' . esc_url( self::$requested_url ) . '| Error message: ' . $error_message . ' | Error code: ' . $error_code, 'error' );
                }
            }
        }

        return new \WP_REST_Response(['message' => 'Notification received and processed successfully'], 200);
    }

     /**
     * Converts an image to WebP format via API.
     *
     * Sends a request to the API endpoint to convert an image to WebP format.
     * 
     * @param string $image_content Base64 encoded string of the image.
     * @param string $image_format  The format of the image being converted.
     *
     * @return string|false The Base64 encoded string of the WebP image or false on failure.
     */
    private static function remove_bucket_in_api( $url_hash, $requested_url, $api_user_id ) {
        /** 
         * API endpoint for generating critical CSS. 
         */
        $api_endpoint = self::$webspeed_api_base . 'remove-bucket';

        /**
         * Get the api key value and status.
         */
        $api_key        = self::$settings['jptgb_setting_api_key'] ?? '';
        $api_key_status = self::$settings['jptgb_setting_is_api_key_valid'] ?? '';

        /**
         * Return if the api key is either not set or invalid.
         */
        if( !$api_key || 'success' != $api_key_status ){
            return false;
        }

        /** 
         * Payload to be sent to the API.
         * Adjusted to send image content and format.
         */
        $body = wp_json_encode( [
            'url'               => $requested_url,
            'urlHash'           => $url_hash,
            'apiUserId'         => $api_user_id,
            'email'             => self::get_setting( 'user_email' ),
            'isDebug'           => self::get_setting( 'debug_activate' ),
        ]);

        /** 
         * Set up the request arguments.
         */
        $args = [
            'body'        => $body,
            'headers'     => [ 
                'Content-Type'  => 'application/json',
                'X-API-Key'     => $api_key,
            ],
            'timeout'       => 30,  // Increase timeout for processing
            'redirection'   => 5,
            'blocking'      => true,
            'httpversion'   => '1.0',
            'sslverify'     => self::$is_production,
            'data_format'   => 'body',
        ];

        /** 
         * Perform the API request.
         */
        $response = wp_remote_post( $api_endpoint, $args );

        /** 
         * Handle the response.
         */
        if( is_wp_error( $response ) ) {
            /**
             * Error handle.
             */
            self::error_log( 'Error requesting the SVG Compresssion: ' . $response->get_error_message() );
            return false;
        }

        /**
         * Hanlde the remote response
         */
        $status_code = wp_remote_retrieve_response_code( $response );

        /**
         * Handle non-successful response.
         */
        if( $status_code < 200 || $status_code >= 300 ) {
            self::error_log('API request returned status code ' . $status_code);
            return false;
        }
        
        /**
         * Return the received data.
         */
        return $response;
    }

    public static function handle_critical_css_endpoint_call( $contents_to_fetch ) {
        /**
         * Get processing transient data.
         */
        $critical_css_content   = $contents_to_fetch['critical_css'] ?? '';
        $url_hash               = $contents_to_fetch['url_hash'] ?? '';
        $device                 = $contents_to_fetch['device'] ?? '';
        $transient_name         = 'jptgb_processing_' .  $url_hash . '_' . $device;

        /**
         * Get the processing data in a transient so we can get it back after the API calls our endpoint.
         */
        $processing_data = get_transient( $transient_name );

        delete_transient( $transient_name );

        if( !$processing_data || !$critical_css_content ){
            return true;
        }

        $requested_url      = $processing_data['requested_url'];
        $url_hash           = $processing_data['url_hash'];
        $device             = $processing_data['device'];
        $cache_file_content = $processing_data['cache_file_content'];
        $html_content       = $processing_data['html_content'];
        $css_data           = $processing_data['css_data'];

        /**
         * Handle the critical css.
         */
        $critical_controller_response   = CriticalCssController::critical_css_page_content( $cache_file_content, $html_content, $requested_url, $url_hash, $css_data, $device, $critical_css_content );
        $cache_file_content             = $critical_controller_response[ 'cache_file_content' ];

        /**
         * Update the cache file content.
         */
        $processing_data['cache_file_content'] = $cache_file_content;
        
        return self::continue_after_critical_css( $processing_data );
    }

    public static function insert_html_content( $requested_url, $url_hash, $html_content, $device, $post_id = '', $api_user_id = '' ){
        try {
            /**
             * Register the status.
             */
            BaseController::$logs->register( '['. $device .'] Insert html content for url: ' . esc_url( self::$requested_url ), 'info' );
    
            /**
             * Reset the script loader list.
             */
            self::$scripts_loader_list = [];
    
            /**
             * Get wp object from url.
             */
            if( $post_id ){
                $wp_object = get_post( $post_id );

            } else {
                $wp_object = AjaxController::get_object_by_url( $requested_url );

            }
    
            /**
             * Get device info.
             */
            $device_slug    = "/$device";
            $is_mobile      = 'mobile' == $device;
    
            /**
             * Update cache status.
             */
            if( $is_mobile ) {
            } else {
                // AjaxController::update_cache_status( $wp_object, 'working-on', 'Creating the Mobile version', 85 );
            }
    
            /**
             * Get the current post permalink.
             */
            $parsed_url     = wp_parse_url( $requested_url );
            $url_path       = $parsed_url['path'];        
    
            /**
             * Get file path and post url.
             */
    
            $request_path       = CacheController::get_request_path( $requested_url, true );
            $cache_url_path     = CacheController::get_cache_dir_path() . $device_slug . $request_path;
            $cached_file_path   = $cache_url_path . 'index.html';
    
            /**
             * Extract all the scripts data that we stored in the api.
             */
            $scripts_data   = Utils::extract_wrapped_content( $html_content, 'JPTGB-SCRIPTS-DATA' );
            $css_data       = Utils::extract_wrapped_content( $html_content, 'JPTGB-CSS-DATA' );
    
            /**
             * Remove the wrapped content from the html.
             */
            $html_content = Utils::remove_wrapped_content( $html_content, 'JPTGB-SCRIPTS-DATA' );
            $html_content = Utils::remove_wrapped_content( $html_content, 'JPTGB-CSS-DATA' );
    
            /**
             * Generate the cache file.
             */
            $cache_file_content = CacheController::create_cache_file( $html_content, $requested_url, $cache_url_path, true );
    
            /**
             * Register the status.
             */
            BaseController::$logs->register( '['. $device .'] Cache file created for url: ' . esc_url( self::$requested_url ), 'info' );

            /**
             * Check if we have to generate critical css.
             */
            $cc_api_call = false;
            if( isset( self::$settings['jptgb_setting_critical_css_activate'] ) && self::$settings['jptgb_setting_critical_css_activate'] && !empty( $css_data ) ) {
                /**
                 * Update cache status.
                 */
                if( $is_mobile ) {
                    AjaxController::update_cache_status( $wp_object, 'working-on', 'Generating Critical CSS', 40 );
                }
                
                /**
                 * Handle the critical css.
                 */
                $critical_controller_response   = CriticalCssController::critical_css_page_content( $cache_file_content, $html_content, $requested_url, $url_hash, $css_data, $device );
                $cc_api_call                    = $critical_controller_response[ 'cc_api_call' ];
                $cache_file_content             = $critical_controller_response[ 'cache_file_content' ];
    
                /**
                 * Register the status.
                 */
                BaseController::$logs->register( '['. $device .'] Critical css ready for url: ' . esc_url( self::$requested_url ), 'info' );

            }

            /**
             * Get the current processing data.
             */
            $processing_data =  compact( 'requested_url', 'url_hash', 'device', 'cache_file_content', 'scripts_data', 'wp_object', 'is_mobile', 'cached_file_path', 'html_content', 'css_data', 'api_user_id' );

            /**
             * If we called the critical css then lets stop this execution and wait for the api to call back our endpoint.
             */
            if( $cc_api_call ){
                /**
                 * Define transient data.
                 */
                $transient_name = 'jptgb_processing_' .  $url_hash . '_' . $device;

                /**
                 * Store the processing data in a transient so we can get it back after the API calls our endpoint.
                 */
                set_transient( $transient_name, $processing_data, MINUTE_IN_SECONDS * 3 );

                /**
                 * Let return so we continue only after the API call our endpoint.
                 */
                return false;
            }

            self::continue_after_critical_css( $processing_data );

        } catch (\Throwable $th) {
            $error_message = $th->getMessage();

            BaseController::$logs->register( '['. $device .'] We had an error while trying to inset the html content for the url: ' . esc_url( self::$requested_url ) . ' | Error message: ' . $error_message, 'error' );

            /**
             * Free up the API call.
             */
            return true;
        } 

        /**
         * Free up the API call.
         */
        return true;
    }

    public static function continue_after_critical_css( $processing_data ) {
        try {
            /**
             * Extract the current processing data.
             */
            $requested_url      = $processing_data['requested_url'];
            $url_hash           = $processing_data['url_hash'];
            $scripts_data       = $processing_data['scripts_data'];
            $device             = $processing_data['device'];
            $cache_file_content = $processing_data['cache_file_content'];
            $cached_file_path   = $processing_data['cached_file_path'];
            $wp_object          = $processing_data['wp_object'];
            $is_mobile          = $processing_data['is_mobile'];
            $api_user_id        = $processing_data['api_user_id'];

            /**
             * Remove all unnesary scripts in the document.
             */
            $cache_file_content = AjaxController::remove_enclosed_content( $cache_file_content );
    
            /**
             * Check if we have to modify the cache file to load the scripts on FUI.
             */
            if( isset( self::$settings['jptgb_setting_load_scripts_on_fui'] ) && self::$settings['jptgb_setting_load_scripts_on_fui'] ) {
                $cache_file_content = ScriptsController::scripts_page_content( $cache_file_content, $scripts_data, $device );
    
                /**
                 * Register the status.
                 */
                BaseController::$logs->register( '['. $device .'] Load scripts on FUI ready for url: ' . esc_url( self::$requested_url ), 'info' );
            } else {
                /**
                 * Make the scripts available again.
                 */
                $cache_file_content = ScriptsController::enable_scripts_again( $cache_file_content, $scripts_data, $device );
            }
    
            /**
             * Check if we have to modify the images to increase performance.
             */
            if( self::get_setting( 'images_activate' ) ){
                /**
                 * Update cache status.
                 */
                if( !$is_mobile ) {
                    AjaxController::update_cache_status( $wp_object, 'working-on', 'Improving Images', 75 );
                } else {
                    AjaxController::update_cache_status( $wp_object, 'working-on', 'Generating the Mobile Version', 90 );
                }
    
                $cache_file_content = ImagesController::improve_images_performance( $cache_file_content, $url_hash );
    
                /**
                 * Register the status.
                 */
                BaseController::$logs->register( '['. $device .'] Improve images ready for url: ' . esc_url( self::$requested_url ), 'info' );
            }
    
            /**
             * Check if we have to modify the content for Inmidiate Dom Load (IDL).
             */
            if( self::get_setting( 'later_dom_load_activate' ) ){
                if( self::get_setting( 'immediate_dom_load' ) ) {
                    $inmediate_dom_load_data    = ImmediateDomLoadController::create_immediate_dom_load( $url_hash, $cache_file_content, $device );
                    $cache_file_content         = $inmediate_dom_load_data['html'];
                    $idl_templates              = $inmediate_dom_load_data['templates'];
        
                    /**
                     * Update the templates in the html document.
                     */
                    $cache_file_content = str_replace('</body>', $idl_templates . '</body>', $cache_file_content);
        
                    /**
                     * Register the status.
                     */
                    BaseController::$logs->register( '['. $device .'] IDL ready ready for url: ' . esc_url( self::$requested_url ), 'info' );
    
                } else if ( self::get_setting( 'lazy_dom_load' ) ) {
                    $cache_file_content = LazyDomLoadController::create_lazy_load_content( $url_hash, $cache_file_content, $device );
        
                    /**
                     * Register the status.
                     */
                    BaseController::$logs->register( '['. $device .'] LDL ready ready for url: ' . esc_url( self::$requested_url ), 'info' );
                }
            }
    
            /**
             * Define the helper properties.
             */
            // $jptgb_properties = [
            //     'isDebug' => self::get_setting( 'debug_activate' ),
            // ];
    
            /**
             * Insert the helper properties.
             */
            // $jptgb_properties_injector = 'window.jptgbProp = ' . json_encode( $jptgb_properties ) . ';';
    
            /**
             * Insert the helper script right in the opening of the head.
             */
            $helper_script = '<script type="text/javascript">'
                // . $jptgb_properties_injector 
                . Utils::get_js_content( 'assets/js/controllers/helper.js' ) 
                . '</script>'; 
    
    
            /**
             * Insert the helper script in the cache file content.
             */
            $cache_file_content = preg_replace( '/(<head(\s+[^>]*>|>))/', '$1' . $helper_script, $cache_file_content, 1 );
    
            /**
             * Insert the in-cache script.
             */
            $cache_file_content = AjaxController::insert_in_cache_script( $cache_file_content, $is_mobile, $requested_url );
    
            /**
             * Filter out the scripts loader list before sending it to js.
             */
            // self::$scripts_loader_list = apply_filters( 'jptgb_scripts_loader_list', self::$scripts_loader_list );
            
            /** 
             * Insert the scripts loader trigger function.
             */
            ob_start();
            ?>
                <script class="jptgb-script" type="text/javascript">
                    new window.JptgbScriptLoaderV3(<?php echo wp_json_encode( self::$scripts_loader_list ) ?>);
                </script>
            <?php
            
            /** 
             * Capture the script output and store it.
             */
            $scripts_loader_script = ob_get_clean();
            
            /**
             * Insert the laoding modal
             */
            $loading_animation = '
                <style>
                    body > div.jptgb_loading_animation {
                        position: fixed;
                        top: 0;
                        left: 0;
                        background-color: #00000082;
                        width: 100%;
                        height: 100%;
                        pointer-events: none;
                        z-index: 9999999;
                    }
                    body > div.jptgb_loading_animation > img {
                        position: absolute;
                        left: 50%;
                        top: 50%;
                        transform: translate(-50%, -50%);
                        width: 100px;
                        height: 100px;
                        object-fit: contain;
                    }
                </style>
                <div class="jptgb_loading_animation" style="display: none;"><img src="'. esc_url( JPTGB_URL . 'assets/svg/loading-white.svg' ) .'" loading="lazy"></div>
            ';
            $cache_file_content = str_replace( '</body>', $loading_animation . '</body>', $cache_file_content );
    
            /**
             * Insert the styles loader trigger into the cache files content.
             * This replaces the closing body tag with the script followed by the closing body tag.
             */
            $cache_file_content = str_replace('</body>', $scripts_loader_script . '</body>', $cache_file_content);
    
            /**
             * Clean up some unwated data from the html document.
             */
            $cache_file_content = Utils::remove_html_attribute( $cache_file_content, 'data-jptgbfonts' );
            $cache_file_content = Utils::remove_html_attribute( $cache_file_content, 'data-jptgb' );
    
            /**
             * Write the cache file.
             */
            // file_put_contents( $cached_file_path, $cache_file_content );
            Utils::write_file( $cached_file_path, $cache_file_content );

            /**
             * If the mobile cache is not set, then lets write the same content for desktop and mobile.
             */
            if( !self::get_setting( 'mobile_cache' ) ){
                $request_path       = CacheController::get_request_path( $requested_url, true );
                $mobile_cache_url_path     = CacheController::get_cache_dir_path() . '/mobile' . $request_path;
                $mobile_cache_file_path   = $mobile_cache_url_path . 'index.html';

                Utils::write_file( $mobile_cache_file_path, $cache_file_content );
            }
    
            /**
             * Register the status.
             */
            BaseController::$logs->register( '['. $device .'] Write cache file ready for url: ' . esc_url( self::$requested_url ), 'info' );

            /**
             * Check if check if both files are ready
             */
            $request_path               = CacheController::get_request_path( $requested_url, true );
            $cache_desktop_url_path     = CacheController::get_cache_dir_path() . '/desktop' . $request_path;
            $cache_mobile_url_path      = CacheController::get_cache_dir_path() . '/mobile' . $request_path;
            $cached_desktop_file_path   = $cache_desktop_url_path . 'index.html';
            $cached_mobile_file_path    = $cache_mobile_url_path . 'index.html';

            /**
             * If this is the last step of the cache generation...
             */
            if( Utils::does_file_exists( $cached_desktop_file_path ) && Utils::does_file_exists( $cached_mobile_file_path ) ) {
                /**
                 * Notify the API that we are set.
                 */
                $bucket_response = self::remove_bucket_in_api( $url_hash, $requested_url, $api_user_id );
                $bucket_status_code = wp_remote_retrieve_response_code( $bucket_response );

                if( false !== $bucket_status_code ) {
                    /**
                     * Sync api data to our plugin.
                     */
                    AjaxController::sync_api_data_to_plugin( $bucket_status_code, $bucket_response );
                }

                /**
                 * Update cache status if the request is not self hosted.
                 */
                if( !self::get_setting( 'activate_self_hosted' ) ){
                    AjaxController::update_cache_status( $wp_object, 'cache-ready', 'Cache Ready', 100 );
                }
    
                /**
                 * Remove this from the consumer pool if exist.
                 */
                $post_id = $wp_object instanceof WP_Post ? $wp_object->ID : false;
                Utils::add_a_task_ready_to_tasks_overview( $post_id );
    
                /**
                 * Register the cache generation.
                 */
                Utils::update_cache_entry( $post_id, 'cache-ready', [] );

                /**
                 * Remove the cosumer if it exist.
                 */
                Utils::delete_consumer( $post_id );
    
                /**
                 * Clean up the transient so we can request regeneration of cache without waiting.
                 */
                delete_transient( "jptgb_cache_build_$url_hash" );
    
                /**
                 * Flush hosting's cache.
                 */
                self::flush_hostings_cache( $wp_object, $requested_url );
    
                /**
                 * Register the status.
                 */
                BaseController::$logs->register( '['. $device .'] We just generated all the cache files for the url: ' . esc_url( self::$requested_url ), 'info' );
            } else {
                // AjaxController::update_cache_status( $wp_object, 'working-on', 'Generating the mobile version', 85 );
            }
    
            /**
             * If we are on mobile, the this is the last page version generation,
             * lets send a message to the API to release the task was assigned to this URL.
             */
            // if( $is_mobile ) {
                
            // }
        } catch (\Throwable $th) {
            //throw $th;
        }

        return true;
    }

    public static function flush_wp_engine_cache( $wp_post = null ) {
        /**
         * Get the post id if available.
         */
        $post_id = ( $wp_post instanceof WP_Post ) ? $wp_post->ID : null;

        /**
         * Get the post title.
         */
        $post_title = $post_id ? ' post ' . get_the_title( $post_id ) : ' all posts';

        try {
            BaseController::$logs->register('Trying to flush WpEngine cache for ' . $post_title, 'info');
            if( 
                class_exists('\wpengine\cache_plugin\ClearAllCachesController') && 
                class_exists('\wpengine\cache_plugin\WpeCommonAdapter') && 
                class_exists('\wpengine\cache_plugin\CacheDbSettings') &&  
                class_exists('\wpengine\cache_plugin\DateTimeHelper') &&  
                class_exists('\wpengine\cache_plugin\PageSpeedBoost') &&
                method_exists('\wpengine\cache_plugin\ClearAllCachesController', 'clear_all_caches')
                ) {
                    BaseController::$logs->register('We found the needed WpEngine classes to flush the WpEngine cache', 'info');
                    ob_start();
                    $wpe_controller = new \wpengine\cache_plugin\ClearAllCachesController( 
                        \wpengine\cache_plugin\WpeCommonAdapter::get_instance(), 
                        \wpengine\cache_plugin\CacheDbSettings::get_instance(), 
                        \wpengine\cache_plugin\DateTimeHelper::get_instance(), 
                        \wpengine\cache_plugin\PageSpeedBoost::get_instance() 
                    );

                    $wpe_controller->clear_all_caches();

                    $capture_any_output = ob_get_clean();

                    unset( $capture_any_output );
            }
        } catch (\Throwable $th) {
            BaseController::$logs->register('Something went wrong while trying to flush the WpEngine cache. | Error message: ' . $th->getMessage() , 'error');
        }

        if( method_exists('WpeCommon', 'purge_varnish_cache') ){
            try {
                /**
                 * Attempt to purge the cache for post id,
                 * if $post_id is null then all cache will be purged.
                 */
                $output = \WpeCommon::purge_varnish_cache( $post_id, true );
                $output = \WpeCommon::purge_memcached();
    
                /**
                 * Check the result of the cache purge attempt.
                 */
                if (!$output) {
                    /**
                     * Log failure on cache purge.
                     */
                    BaseController::$logs->register('Not possible to flush varnish cache for' . $post_title, 'warning');
                    
                } else {
                    BaseController::$logs->register('Varnish cache flushed for' .  $post_title, 'info');
    
                }
            } catch (\Exception $e) {
                /**
                 * Log any exceptions that occur during the cache purge.
                 */
                BaseController::$logs->register('Error occurred while trying to flush cache: ' . $e->getMessage(), 'error');
            }
        }
    }

    /**
     * Push URLs related to a given post into the purge list.
     *
     * This method collects several URLs associated with the post—including the main permalink,
     * REST API endpoints, AMP URLs, category and tag URLs, author URLs, archive URLs, feed URLs,
     * and the homepage—and merges them into the existing cache purge list.
     *
     * @param int $post_id The post ID from which URLs should be generated.
     * @return void
     */
    public static function push_url( int $post_id ): void {
        /**
         * Get essential post information.
         */
        $permalink   = Utils::get_permalink( $post_id );
        $post_status = get_post_status( $post_id );
        $post_type   = get_post_type( $post_id );

        /**
         * Define valid post statuses and excluded post types.
         */
        $valid_post_statuses  = array( 'publish', 'private', 'trash' );
        $invalid_post_types   = array( 'nav_menu_item', 'revision' );
        $noarchive_post_types = array( 'post', 'page' );

        /**
         * Guard clause: Return early if we do not have a valid permalink,
         * if the post status is not valid, or if the post type should be excluded.
         */
        if ( ! $permalink || ! in_array( $post_status, $valid_post_statuses, true ) || in_array( $post_type, $invalid_post_types, true ) ) {
            return;
        }

        /**
         * Initialize an array to collect URLs.
         */
        $urls = array();

        /**
         * Add the primary post URL.
         */
        $urls[] = $permalink;

        /**
         * Set up the REST API details.
         */
        $rest_api_route = 'wp/v2';
        $rest_url       = rtrim( get_rest_url(), '/' );
        $rest_url_prefix = $rest_url . '/' . $rest_api_route;

        /**
         * Generate REST API permalink based on the post type.
         */
        $post_type_object = get_post_type_object( $post_type );
        if ( isset( $post_type_object->rest_base ) && ! empty( $post_type_object->rest_base ) ) {
            $rest_permalink = $rest_url_prefix . '/' . $post_type_object->rest_base . '/' . $post_id . '/';
        } elseif ( 'post' === $post_type ) {
            $rest_permalink = $rest_url_prefix . '/posts/' . $post_id . '/';
        } elseif ( 'page' === $post_type ) {
            $rest_permalink = $rest_url_prefix . '/views/' . $post_id . '/';
        }
        if ( isset( $rest_permalink ) ) {
            $urls[] = $rest_permalink;
        }

        /**
         * Add AMP URL if the AMP function exists.
         */
        if ( function_exists( 'amp_get_permalink' ) ) {
            $urls[] = \amp_get_permalink( $post_id );
        }

        // Add the regular AMP URL.
        $urls[] = $permalink . 'amp/';

        /**
         * For trashed posts, clean the URL by removing any '__trashed' fragment.
         */
        if ( 'trash' === $post_status ) {
            $trash_post = str_replace( '__trashed', '', $permalink );
            $urls[] = $trash_post;
            $urls[] = $trash_post . 'feed/';
        }

        /**
         * Add category-related URLs.
         */
        $categories = get_the_category( $post_id );
        if ( $categories ) {
            foreach ( $categories as $cat ) {
                $category_link         = get_category_link( $cat->term_id );
                $category_link_no_cat  = str_replace( 'category/', '', $category_link );
                if ( ! empty( $category_link ) && $category_link !== $category_link_no_cat ) {
                    $urls[] = $category_link_no_cat;
                }
                $urls[] = $category_link;
                $urls[] = $rest_url_prefix . '/categories/' . $cat->term_id . '/';
            }
        }

        /**
         * Add tag-related URLs.
         */
        $tags = get_the_tags( $post_id );
        if ( $tags ) {
            foreach ( $tags as $tag ) {
                $urls[] = get_tag_link( $tag->term_id );
                $urls[] = $rest_url_prefix . '/tags/' . $tag->term_id . '/';
            }
        }

        /**
         * Add author-related URLs.
         */
        $author_id = get_post_field( 'post_author', $post_id );
        $urls[] = get_author_posts_url( $author_id );
        $urls[] = get_author_feed_link( $author_id );
        $urls[] = $rest_url_prefix . '/users/' . $author_id . '/';

        /**
         * Add archive URLs for custom post types.
         */
        if ( $post_type && ! in_array( $post_type, $noarchive_post_types, true ) ) {
            $urls[] = get_post_type_archive_link( $post_type );
            $urls[] = get_post_type_archive_feed_link( $post_type );
        }

        /**
         * Add various feed URLs.
         */
        $urls[] = get_bloginfo_rss( 'rdf_url' );
        $urls[] = get_bloginfo_rss( 'rss_url' );
        $urls[] = get_bloginfo_rss( 'rss2_url' );
        $urls[] = get_bloginfo_rss( 'atom_url' );
        $urls[] = get_bloginfo_rss( 'comments_rss2_url' );
        $urls[] = get_post_comments_feed_link( $post_id );

        /**
         * Trim spaces from all URLs.
         */
        $urls = array_map( 'trim', $urls );

        /**
         * Ensure the homepage URL is included.
         */
        $homepage = trailingslashit( home_url() );
        if ( ! in_array( $homepage, $urls, true ) ) {
            $urls[] = $homepage;
        }

        /**
         * Merge the new URLs into the existing purge list, ensuring uniqueness.
         */
        if ( ! empty( $urls ) ) {
            self::$urls_purge = array_filter(
                array_unique(
                    array_merge( self::$urls_purge, $urls ),
                    SORT_REGULAR
                )
            );
        }
    }

    /**
     * Purge entire Varnish cache.
     *
     * Sends a PURGE request to Varnish for the home URL (full-site purge).
     *
     * @return bool True on success, false on failure.
     */
    public static function purge_varnish_cache(): bool {
        try {
            // We always do a full-site purge.
            $purge_method = 'PURGE';
            $pregex       = '.*';

            // Log the purge action.
            BaseController::$logs->register( 'Purging entire Varnish cache for site', 'info' );

            // Parse the home URL to get scheme, host, and path.
            $home_url = home_url();
            $parts    = wp_parse_url( $home_url );

            // $parts['path'] may be empty if home_url() is just the domain.
            $path = $parts['path'] ?? '';
            if ( '' === $path ) {
                $path = '/'; // Ensure the slash is present before adding ".*"
            }

            $scheme = isset( $parts['scheme'] ) ? $parts['scheme'] . '://' : 'http://';
            $host   = $parts['host']   ?? '';

            /*
            * Define Varnish host (and optional port).
            *
            * - Defaults to 127.0.0.1:6081.
            * - Override via filters:
            *     add_filter( 'jptgb_varnish_host', fn() => 'localhost' );
            *     add_filter( 'jptgb_varnish_port', fn() => 80 );
            *
            * If PHP and Varnish live in different Docker containers, set
            * 'jptgb_varnish_host' to the Docker service name (e.g. 'varnish').
            */
            $varnish_host = apply_filters( 'jptgb_varnish_host', '127.0.0.1' );
            $varnish_port = apply_filters( 'jptgb_varnish_port', 6081 );

            // Build the purge endpoint: host:port + path + regex
            $endpoint = $varnish_host . ':' . $varnish_port . $path . $pregex;

            // SSL verification: apply filter, disable in WP_DEBUG, ensure bool.
            $ssl_verify = apply_filters( 'jptgb_ssl_check_certificate', true );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                $ssl_verify = false;
            } elseif ( ! is_bool( $ssl_verify ) ) {
                // If a developer filter returns something odd, force it to true/false
                $ssl_verify = true;
            }

            // Prepare args for wp_remote_request.
            $args = [
                'method'    => $purge_method,
                'headers'   => [
                    'Host'       => $host,
                    'User-Agent' => 'WordPress/' . get_bloginfo( 'version' ),
                ],
                'sslverify' => $ssl_verify,
            ];

            // First attempt with the original scheme (http or https).
            $response = wp_remote_request( $scheme . $endpoint, $args );

            // If it fails (WP_Error) or returns a non-200 code, flip http↔https and retry.
            if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
                $scheme   = ( 'https://' === $scheme ) ? 'http://' : 'https://';
                $response = wp_remote_request( $scheme . $endpoint, $args );
            }

            // If still an error or non-200, log and return false.
            if ( is_wp_error( $response ) || 200 !== (int) wp_remote_retrieve_response_code( $response ) ) {
                $error_message = is_wp_error( $response )
                    ? $response->get_error_message()
                    : 'Unexpected response code.';

                BaseController::$logs->register(
                    'Failed to flush Varnish cache | Error: ' . $error_message,
                    'error'
                );

                return false;
            }

            // Success: log and return true.
            BaseController::$logs->register( 'Varnish cache flushed successfully', 'info' );
        }
        catch ( \Throwable $th ) {
            // Catch any unexpected PHP exception and log it.
            $error_message = 'Something went wrong while trying to flush the Varnish cache | Error: ' 
                            . $th->getMessage();
            BaseController::$logs->register( $error_message, 'error' );

            return false;
        }

        return true;
    }

    /**
     * Flushes the WP Engine cache based on a given WP_Post object.
     *
     * This method checks for the presence of the WP Engine-specific class and methods
     * before attempting to purge the cache. It handles possible exceptions and logs
     * them appropriately.
     *
     * @param mixed $wp_object The WP_Post object whose cache needs to be purged.
     */
    public static function flush_hostings_cache( $wp_object = null, $requested_url = '' ) {
        BaseController::$logs->register( 'Flushing Hostings Cache', 'info' );

        $post_id = $wp_object instanceof WP_Post ? $wp_object->ID : false;

        /**
         * Get the post object.
         */
        if( is_numeric( $wp_object ) ){
            $wp_object = get_post( $wp_object );
        }

        /**
         * Update the post in order to flush hostings cache by default.
         */
        if( $post_id ){
            /**
             * Check if hosting is Siteground.
             */
            if( SitegroundHelper::is_siteground()){
                /**
                 * Register the status.
                 */
                BaseController::$logs->register( 'Looks like this site is hosted in Siteground, lets try to flush the Dynamic Cache', 'info' );

                /**
                 * Flush Siteground dynamic cache
                 */
                SitegroundHelper::flulsh_cache( $post_id, $requested_url );

            /**
             * Check if the WP Engine specific class exists.
             */
            } elseif( class_exists('WpeCommon') ) {
                /**
                 * Register the status.
                 */
                BaseController::$logs->register( 'Looks like this site is hosted in WpEngine, lets try to the hosting cache', 'info' );

                /**
                 * Flush WpEngine Cache
                 */
                self::flush_wp_engine_cache( $wp_object );

            /**
             * If it is none of above hostings, then lets simulate that the post has been updated.
             */
            } else {
                BaseController::$logs->register( 'We haven\'t found any recognised hosting, lets simulate a post update...', 'info' );

                CacheController::$suppress_transition_post_status = true;
                do_action( 'edit_post',         $post_id, $wp_object );
                do_action( 'post_updated',      $post_id, $wp_object, $wp_object );
                do_action( 'save_post',         $post_id, $wp_object, true );
                do_action( 'wp_insert_post',    $post_id, $wp_object, true );
                do_action( 'wp_after_insert_post', $post_id, $wp_object, true, $wp_object );
                CacheController::$suppress_transition_post_status = false;
            }
        }

        /**
         * Lets flush varnish cache if is enabled.
         */
        if(  Utils::is_varnish_cache_started() ){
            BaseController::$logs->register( 'It looks like the site uses Varnish cache, lets try to clean it up.', 'info' );
            self::purge_varnish_cache();
        }

        /**
         * Reset all cloudflare cache.
         */
        if( CloudflareHelper::is_cloudflare_enabled() ) {
            BaseController::$logs->register( 'It looks like the site uses Cloudflare and is hosted in Cloudways, lets try to clean up the cache.', 'info' );
            CloudflareHelper::reset_all_cache();
		}
    }
}
