<?php

/**
 * The Scripts Controller Class.
 *
 * @author Yaidier Perez
 * */

namespace Jptgb\Controllers;

use Jptgb\Resources\Utils;

class ScriptsController extends BaseController {
    public static function init() {
        /**
         * If we just turn on/off the scripts controller,
         * and the cache was active then lets
         * clear up the cache then.
         */
        if( in_array( 'jptgb_setting_load_scripts_on_fui', self::$changed_settings ) ) {
            CacheController::flush_all_cache_files();
        }

        /**
         * Return if Load Scripts on FUI is deactivated.
         * 
         * @see Wp Dashboard -> Jptgb -> Scripts
         */
        if( !isset( self::$settings['jptgb_setting_load_scripts_on_fui'] ) || !self::$settings['jptgb_setting_load_scripts_on_fui'] ) {
            return;
        }

        /**
         * Init scripts handler.
         */
        add_action( 'wp', [self::class, 'init_scripts_handler'] );
    }

    /**
     * Creates a 'critical-css' directory inside the WordPress 'uploads' directory and returns its path and URL.
     * Adheres to WordPress coding standards.
     *
     * @return array|bool An associative array with 'path' and 'url' of the critical-css directory, or false on failure.
     */
    public static function get_scripts_directory() {
        /**
         * Get the path and URL to the WordPress 'uploads' directory.
         */
        $upload_dir_info   = wp_upload_dir();
        $upload_path       = $upload_dir_info['basedir'];

        /** 
         * Define the path and URL for the 'critical-css' directory.
         */
        $scripts_base_dir  = $upload_path . '/jptgb/scripts';

        /**
         * Check if the directory already exists.
         */
        if( !Utils::does_file_exists( $scripts_base_dir ) ) {
            /**
             * Attempt to create the directory.
             */
            if ( !Utils::create_directory_recursively( $scripts_base_dir, 0755 ) ) {
                /**
                 * Handle the error, e.g., log or notify.
                 */
                self::error_log( 'Failed to create the scripts directory.' );
                return false;
            }
        }

        return $scripts_base_dir;
    }

    public static function init_scripts_handler() {
        /**
         * Do not apply Optimizations for the backend or logged-in users.
         */
		if ( is_admin() || is_user_logged_in() ) {
            return;
		}

        /** 
         * Return if we are on login page.
         */
        if( $GLOBALS['pagenow'] == 'ps-access.php' ){
            return;
        }

        /**
         * Generate critical css.
         */
        // add_action( 'wp_footer', [self::class, 'call_scripts_on_fui__callback'], PHP_INT_MAX - 8 );
    }

    

    public static function scripts_page_content( $cache_file_contents, $scripts_data, $device ) {
        try {
            //code...
            /**
             * Iterate over each script data.
             */
            foreach( $scripts_data as $script_data ){
                /**
                 * Get the script attributes.
                 */
                $attributes = $script_data['attributes'] ?? [];
    
                /**
                 * Define the handle.
                 */
                $handle = $attributes['id'] ?? '';
                
                /**
                 * Generate an unique id.
                 */
                $script_id = $script_data['script_id'] ?? self::get_unique_id();
                
                /**
                 * Add our id as attribute.
                 */
                $attributes['data-jptgbscriptid'] = $script_id;
    
                /**
                 * If no id is set, then lets create a temporal id.
                 */
                if( !$handle ){
                    $handle = $script_id;
                }
    
                /**
                 * Get the action to perform.
                 */
                $action = $script_data['action'] ?? 'load_on_fui';
    
                /**
                 * Perform the different actions.
                 */
                switch( $action ) {
                    case 'execute_on_puppetter': { 
                        $cache_file_contents = self::execute_on_puppetter_action( $cache_file_contents, $script_id );
                        continue 2;
                    }
                    break;
    
                    case 'do_not_execute_on_fui': { 
                        $cache_file_contents = self::do_not_execute_on_fui_action( $script_data, $cache_file_contents, $script_id );
                        continue 2;
                    }
                    break;
                    
                    /**
                     * Defualt is assumed to be 'load_on_fui'.
                     */
                    default:
                        self::create_script_for_fui_event( $handle, $attributes, $script_data );
                        break;
                }
            }
    
            /**
             * Remove the scripts tags from the cache file content.
             */
            // $cache_file_contents = self::remove_all_wrapped_content_general( $cache_file_contents );
    
            /**
             * Return the page content.
             */
            return $cache_file_contents;
        } catch (\Throwable $th) {
            $error_message = $th->getMessage();
            BaseController::$logs->register( 'Something failed while trying to generate the scripts page content | Error message: ' . $error_message, 'error' );
            
            throw $th;
        }
    }

    public static function enable_scripts_again( $cache_file_contents, $scripts_data, $device ) {
        try {
            //code...
            /**
             * Iterate over each script data.
             */
            foreach( $scripts_data as $script_data ){            
                /**
                 * Generate an unique id.
                 */
                $script_id = $script_data['script_id'] ?? null;
    
                /**
                 * Continue if no script id has beend defined.
                 */
                if( !$script_id ){
                    continue;
                }
                
                /**
                 * Get the script content.
                 */
                $original_script_content = self::get_wrapped_content_to_remove_by_id( $cache_file_contents, $script_id );
    
                if( $original_script_content ) {
                    /**
                     * Enable previously disabled type.
                     */ 
                    if( isset( $script_data['original_type'] ) && $script_data['original_type'] ){
                        $modified_script_content = str_replace( 'javascript/jptgbblocked', $script_data['original_type'], $original_script_content );
        
                    } else {
                        $modified_script_content = str_replace( 'type="javascript/jptgbblocked"', '', $original_script_content );
                    }
    
                    /**
                     * Replace the the origin script content by the modified one.
                     */
                    $cache_file_contents = str_replace( $original_script_content, $modified_script_content, $cache_file_contents );
                }
    
    
                /**
                 * Get the script content.
                 */
                $original_script_content_to_keep = self::get_wrapped_content_to_keep_by_id( $cache_file_contents, $script_id );
    
                if( $original_script_content_to_keep ) {
                    /**
                     * Enable previously disabled type.
                     */ 
                    if( isset( $script_data['original_type'] ) && $script_data['original_type'] ){
                        $modified_script_content = str_replace( 'javascript/jptgbblocked', $script_data['original_type'], $original_script_content_to_keep );
        
                    } else {
                        $modified_script_content = str_replace( 'type="javascript/jptgbblocked"', '', $original_script_content_to_keep );
                    }
    
                    /**
                     * Replace the the origin script content by the modified one.
                     */
                    $cache_file_contents = str_replace( $original_script_content_to_keep, $modified_script_content, $cache_file_contents );
                }
            }
    
            /**
             * Clean up the comments
             */
            $cache_file_contents = preg_replace( '/<!--JPTGBREMOVESCRIPTSTART-[\w]+-->/', '', $cache_file_contents );
            $cache_file_contents = preg_replace( '/<!--JPTGBREMOVESCRIPTEND-->/', '', $cache_file_contents );
            $cache_file_contents = preg_replace( '/<!--JPTGBKEEPSCRIPTSTART-[\w]+-->/', '', $cache_file_contents );
            $cache_file_contents = preg_replace( '/<!--JPTGBKEEPSCRIPTEND-->/', '', $cache_file_contents );
    
            /**
             * Return the page content.
             */
            return $cache_file_contents;
        } catch (\Throwable $th) {
            $error_message = $th->getMessage();
            BaseController::$logs->register( 'Something failed while trying to enable the scripts again | Error message: ' . $error_message, 'error' );
            
            throw $th;
        }
    }

    private static function do_not_execute_on_fui_action( $script_data, $cache_file_contents, $script_id ): string {
        /**
         * Get the script content.
         */
        $original_script_content = self::get_wrapped_content_to_keep_by_id( $cache_file_contents, $script_id );

        /**
         * Enable previously disabled type.
         */ 
        if( isset( $script_data['original_type'] ) and $script_data['original_type'] ){
            $modified_script_content = str_replace( 'javascript/jptgbblocked', $script_data['original_type'], $original_script_content );

        } else {
            $modified_script_content = str_replace( 'type="javascript/jptgbblocked"', '', $original_script_content );
        }

        /**
         * Replace the the origin script content by the modified one.
         */
        $cache_file_contents = str_replace( $original_script_content, $modified_script_content, $cache_file_contents );

        return $cache_file_contents;
    }

    private static function execute_on_puppetter_action( $cache_file_contents, $script_id ): string {
        /**
         * Get the script content.
         */
        $original_script_content = self::get_wrapped_content_to_keep_by_id( $cache_file_contents, $script_id );

        /**
         * Replace the the origin script content by the modified one.
         */
        $cache_file_contents = str_replace( $original_script_content, '', $cache_file_contents );

        return $cache_file_contents;
    }

    /**
     * Removes all instances of content wrapped between custom start and end comment markers,
     * regardless of the presence of an ID.
     *
     * @param string $html The full HTML document as a string.
     * @return string The HTML content after all specified wrapped contents have been removed.
     */
    private static function remove_all_wrapped_content_general( string $html ): string {
        // Pattern to match and remove any content between the specified start and end comments
        $pattern = "/<!--JPTGBREMOVESCRIPTSTART-[\s\S]*?-JPTGBREMOVESCRIPTEND-->/";

        // Perform the regex replacement to remove matched content
        $updatedHtml = preg_replace($pattern, '', $html);

        // Return the updated HTML content
        return $updatedHtml;
    }

    /**
     * Extracts the content wrapped within specified start and end comments by ID.
     *
     * @param string $html The full HTML document as a string.
     * @param string $id The unique identifier used within the comment markers.
     * @return string|null The extracted content or null if not found.
     */
    private static function get_wrapped_content_by_id( string $html, string $id ): ?string {
        // Pattern to match the wrapped content based on the provided ID
        $pattern = "/<!--JPTGBREMOVESCRIPTSTART-$id-->(.*?)<!--JPTGBREMOVESCRIPTEND-->/s";

        // Perform the regex search
        if ( preg_match( $pattern, $html, $matches ) ) {
            // If a match is found, return the content (without the comment markers)
            return $matches[1];
        }

        // Return null if no content is found
        return null;
    }

    /**
     * Extracts the content wrapped within specified start and end comments by ID.
     *
     * @param string $html The full HTML document as a string.
     * @param string $id The unique identifier used within the comment markers.
     * @return string|null The extracted content or null if not found.
     */
    private static function get_wrapped_content_to_keep_by_id( string $html, string $id ): ?string {
        // Pattern to match the wrapped content based on the provided ID
        $pattern = "/<!--JPTGBKEEPSCRIPTSTART-$id-->(.*?)<!--JPTGBKEEPSCRIPTEND-->/s";

        // Perform the regex search
        if ( preg_match( $pattern, $html, $matches ) ) {
            // If a match is found, return the content (without the comment markers)
            return $matches[1];
        }

        // Return null if no content is found
        return null;
    }

    private static function get_wrapped_content_to_remove_by_id( string $html, string $id ): ?string {
        // Pattern to match the wrapped content based on the provided ID
        $pattern = "/<!--JPTGBREMOVESCRIPTSTART-$id-->(.*?)<!--JPTGBREMOVESCRIPTEND-->/s";

        // Perform the regex search
        if ( preg_match( $pattern, $html, $matches ) ) {
            // If a match is found, return the content (without the comment markers)
            return $matches[1];
        }

        // Return null if no content is found
        return null;
    }

    /**
     * Convert a string of key-value pairs into an associative array.
     *
     * @param string $input The input string containing key-value pairs.
     * @return array The associative array representation of the input string.
     */
    public static function convert_to_array( $input ) {
        // Use a regular expression to extract the key-value pairs
        preg_match_all( '/\{([^}]+)\}/', $input, $matches );

        // Initialize the result array
        $result = array();

        // Loop through each match
        foreach ( $matches[1] as $match ) {
            // Split the key-value pairs
            $pairs = explode( ', ', $match );

            // Initialize an associative array for each set of key-value pairs
            $array = array();

            // Loop through each pair and add it to the associative array
            foreach ( $pairs as $pair ) {
                list( $key, $value ) = explode( ': ', $pair );
                $key = trim( $key, "'" );
                $value = trim( $value, "'" );
                $array[ $key ] = $value;
            }

            // Add the associative array to the result array
            $result[] = $array;
        }

        return $result;
    }

    private static function check_if_script_is_set_to_not_modify( $attributes ){
        /**
         * Return true if we do not allow modification for external scripts.
         */
        if( !self::get_setting( 'modify_external_scripts' ) ){
            return true;
        }

        /**
         * Get the exclusion rules.
         */
        $exclusion_rules = self::get_setting( 'exclude_external_scripts_from_modify' );

        return self::does_it_meet_any_exclusion_rule( $attributes, $exclusion_rules );
    }

    private static function does_it_meet_any_exclusion_rule( $attributes, $exclusion_rules ): bool {
        /**
         * Return true if exclusion rules is empty.
         */
        if( !$exclusion_rules ){
            return true;
        }

        /**
         * Extract the rules.
         */
        $exclusion_rules_list = self::convert_to_array( $exclusion_rules );

        /**
         * Initially set the no modify to false.
         */
        $not_modify = false;

        /**
         * Check if the attribute name is set to exclude
         */
        foreach( $exclusion_rules_list as $exclusion_rule ){
            /**
             * Iterate over each exclusion value.
             */
            foreach( $exclusion_rule as $exclusion_att_name => $exclusion_att_value ){
                /**
                 * Check if the exclusion attribute name exist and its value is included in the attribute of the element.
                 */
                if( isset( $attributes[$exclusion_att_name] ) && strpos( $attributes[$exclusion_att_name], $exclusion_att_value ) !== false ){
                    $not_modify = true;

                } else {
                    /**
                     * If at least one of the attributes of the exclusion object fail, then 
                     * lets break the foreach to avoid further proccessing.
                     */
                    $not_modify = false;
                    break;

                }
            }

            /**
             * Break if at least one exclusion rule match.
             */
            if( $not_modify ){
                break;
            }
        }

        return $not_modify;
    }

    /**
     * Function to check if the src attribute belongs to jQuery or jQuery Migrate library
     */
    private static function is_jquery( $src ) {
        $jquery_patterns = [
            /**
             * Common CDN patterns for jQuery.
             */
            '/jquery(-\d+\.\d+\.\d+)?(\.min)?\.js$/',

            /**
             * WordPress patterns for jQuery and jQuery Migrate.
             */
            '/\/wp-includes\/js\/jquery\/jquery(\.min)?\.js$/',
        ];

        foreach ($jquery_patterns as $pattern) {
            if (preg_match($pattern, $src)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Function to check if the provided script content belongs to the jQuery library.
     */
    private static function is_jquery_lib_content($content) {
        /**
         * Check for jQuery version comment or specific jQuery patterns.
         */
        if (preg_match('/\/\*\! jQuery v\d+\.\d+\.\d+ \| \(c\) OpenJS Foundation and other contributors \| jquery\.org\/license \*\//', $content) ||
            preg_match('/jQuery v\d+\.\d+\.\d+/', $content) && strpos($content, 'jQuery.fn') !== false) {
            return true;
        }

        return false;
    }

    private static function update_custom_events_in_script_content( $script_content, $script_id = '' ){
        $script_content = self::replace_events_with_custom( $script_content );
        $script_content = self::replace_jquery_events_with_custom( $script_content );
        $script_content = self::other_replacements( $script_content, $script_id );

        return $script_content;
    }

    /**
     * Replace standard events with custom events in JavaScript content.
     *
     * @param string $script_content The original script content.
     * @return string The modified script content with custom events.
     */
    private static function replace_events_with_custom( $script_content ) {
        /**
         * The execution order for the scripts is as following.
         * 
         *  1 - document.addEventListener('DOMContentLoaded', () => {...}) in the same order as they appear on the document.
         *  2 - jQuery(window).load(() => {...}) | jQuery(window).on('load', () => {...}) | jQuery(window).bind('load', () => {...}) in the same order as they appear on the document.
         *  3 - window.addEventListener('load', function() {...}) | window.onload = () => {...} in the same order as they appear on the document.
         *  4 - jQuery(document).on('ready', () => {...}) in the same order as they appear on the document.
         *  5 - jQuery(document).ready(() => {...}) in the same order as they appear on the document.
         */

        $patterns = [
            '/(document\.addEventListener\(\s*[\'"]DOMContentLoaded[\'"])/',    // document.addEventListener('DOMContentLoaded'
            '/(window\.addEventListener\(\s*[\'"]load[\'"])/',                  // window.addEventListener('load'
            '/(document\.addEventListener\(\s*[\'"]readystatechange[\'"])/',    // document.addEventListener('readystatechange'
            '/(window\.onload\s*=\s*function\s*\()/i',                          // window.onload = function(
            '/(window\.onload\s*=\s*\()/i',                                     // window.onload = (
            '/(document\.readyState\s+)/',
            '/document\.addEventListener\(\s*([\'"]([^\'"]+)[\'"]|\w+)\s*,/',   // document.addEventListener( 'event' or eventName 
            '/window\.addEventListener\(\s*([\'"]([^\'"]+)[\'"]|\w+)\s*,/',     // document|window.addEventListener( 'event' or eventName 
        ];
        
        $replacements = [
            'document.addEventListener(\'jptgbEventDOMContentLoaded\'',
            'window.addEventListener(\'jptgbEventLoad\'',
            'document.addEventListener(\'jptgbEventReadyStateChange\'',
            'window.onJptgbLoad = function(',                               // Replace with custom event
            'window.onJptgbLoad = (',                                       // Replace with custom event for arrow function
            'document.jptgbReadyState ',      
            'document.jptgbOnDocument($1, ',                                // window.addEventListener('load'
            'document.jptgbOnWindow($1, ',                                  // window.addEventListener('load'
        ];

        /**
         * Make the replacements.
         */
        $script_content = preg_replace( $patterns, $replacements, $script_content );
        
        return $script_content;
    }

    /**
     * Replace standard jQuery events with custom events in JavaScript content.
     *
     * @param string $script_content The original script content.
     * @return string The modified script content with custom events.
     */
    private static function replace_jquery_events_with_custom( $script_content ) {
        /**
         * The execution order for the scripts is as following.
         * 
         *  1 - document.addEventListener('DOMContentLoaded', () => {...}) in the same order as they appear on the document.
         *  2 - jQuery(window).load(() => {...}) | jQuery(window).on('load', () => {...}) | jQuery(window).bind('load', () => {...}) in the same order as they appear on the document.
         *  3 - window.addEventListener('load', function() {...}) | window.onload = () => {...} in the same order as they appear on the document.
         *  4 - jQuery(document).on('ready', () => {...}) in the same order as they appear on the document.
         *  5 - jQuery(document).ready(() => {...}) in the same order as they appear on the document.
         */

        $patterns = [
            // No conflict matchs
            '/(\$|\w+)\(\s*document\s*\)\.ready\s*\(\s*(\(*)\s*function\s*\(\s*([^\),\s]+)\s*\)\s*{/',  // $(document).ready( function($) {
            '/jQuery\(\s*function\s*\(\s*([^\),\s]+)\s*\)\s*{/',                                        // $( function ($) {
            '/jQuery\(\s*\(\s*([^\),\s]+)\s*\)\s*=>\s*{/',                                              // $( ($) {

            // Regular matchs
            '/(\$|\w+)\(\s*document\s*\)\.ready\s*\(/',                         // $(document).ready(
            '/(\$|\w+)\(\s*window\s*\)\.on\(\s*[\'"]load[\'"]\s*,/',            // $(window).on('load',
            '/(\$|\w+)\(\s*document\s*\)\.on\(\s*[\'"]ready[\'"]\s*,/',         // $(document).on('ready',
            '/(\$|\w+)\(\s*window\s*\).load\s*\(/',                             // $(window).load(
            '/(\$|jQuery)\(\s*function\s*\(/',                                  // $(function( -> This is shorthand for $(document).ready(
            '/(\$|jQuery)\(\s*\(/',                                             // $(( -> This is shorthand for $(document).ready(

            // Multiple events
            '/(\$|\w+)\(\s*document\s*\)\.on\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(["\'][^"\']*["\']|\w+)\s*,\s*(\{[^{}]*\}|\w+)\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*document\s*\)\.on\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(["\'][^"\']*["\']|\w+|\{[^{}]*\})\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*document\s*\)\.on\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*window\s*\)\.on\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(["\'][^"\']*["\']|\w+)\s*,\s*(\{[^{}]*\}|\w+)\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*window\s*\)\.on\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(["\'][^"\']*["\']|\w+|\{[^{}]*\})\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*window\s*\)\.on\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',

            // Deprecated patterns
            '/(\$|\w+)\(\s*document\s*\)\.bind\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(["\'][^"\']*["\']|\w+|\{[^{}]*\})\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*document\s*\)\.bind\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*window\s*\)\.bind\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(["\'][^"\']*["\']|\w+|\{[^{}]*\})\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
            '/(\$|\w+)\(\s*window\s*\)\.bind\(\s*((["\'][^"\']*["\']|\w+)\s*,\s*(function\s*\(|\(|\b\w+\b(?!,)))/',
        ];
        
        $replacements = [
            // No conflict replacements
            'document.jptgbJqueryOnDocument(\'jptgbJqueryReady\', $2 function(data) { let $3 = data.detail.jQuery;', 
            'document.jptgbJqueryOnDocument(\'jptgbJqueryReady\', function(data) { let $1 = data.detail.jQuery;', 
            'document.jptgbJqueryOnDocument(\'jptgbJqueryReady\', (data) => { let $1 = data.detail.jQuery;', 

            // Regular replacements
            'document.jptgbJqueryOnDocument(\'jptgbJqueryReady\',',           // Replacement for $(document).ready(
            'document.jptgbJqueryOnDocument(\'jptgbJQueryEventLoad\',',         // Replacement for $(window).on('load',
            'document.jptgbJqueryOnDocument(\'jptgbJqueryOnReady\',',         // Replacement for $(document).on('ready',
            'document.jptgbJqueryOnWindow(\'jptgbJQueryEventLoad\',',         // Replacement for $(window).load(
            'document.jptgbJqueryOnDocument(\'jptgbJqueryReady\', function(', // Replacement for $(function(
            'document.jptgbJqueryOnDocument(\'jptgbJqueryReady\', (',         // Replacement for $((

            // Multiple events replacements
            'document.jptgbJqueryOnDocument($2',                            // Replacement for $(document).on('load resize etc',
            'document.jptgbJqueryOnDocument($2',                            // Replacement for $(document).on('load resize etc',
            'document.jptgbJqueryOnDocument($2',                            // Replacement for $(document).on('load resize etc',
            'document.jptgbJqueryOnWindow($2',                              // Replacement for $(window).on('load resize etc',
            'document.jptgbJqueryOnWindow($2',                              // Replacement for $(window).on('load resize etc',
            'document.jptgbJqueryOnWindow($2',                              // Replacement for $(window).on('load resize etc',

            // Deprecated patterns replacements
            'document.jptgbJqueryOnDocument($2',                              // Replacement for $(document).bind('load resize etc',
            'document.jptgbJqueryOnDocument($2',                              // Replacement for $(document).bind('load resize etc',
            'document.jptgbJqueryOnWindow($2',                                // Replacement for $(window).bind('load resize etc',
            'document.jptgbJqueryOnWindow($2',                                // Replacement for $(window).bind('load resize etc',
        ];
        
        /**
         * Make the replacements.
         */
        $script_content = preg_replace($patterns, $replacements, $script_content);

        return $script_content;
    }

    /**
     * 
     */
    private static function other_replacements( $script_content, $script_id ) {
        $patterns = [
            '/\s*document.write\s*\(/',
        ];
        
        $replacements = [
            'document.jptgbWrite("' . $script_id . '", ', 
        ];
        
        /**
         * Make the replacements.
         */
        $script_content = preg_replace( $patterns, $replacements, $script_content );

        return $script_content;
    }

    /**
     * Check if we have to exclude the script.
     */
    private static function is_script_to_exclude( $attributes ){
        /**
         * Get the attributes names.
         */
        $attribute_names = array_keys( $attributes );

        /**
         * Return false if neither the id not the class attributes are set.
         */
        if( !in_array( 'id', $attribute_names ) && !in_array( 'class', $attribute_names ) ){
            return false;
        }

        /**
         * Get selectors to exlcude.
         */
        $selectors_to_deactivate_and_exclude_from_fui = BaseController::get_setting( 'jptgb_setting_exclude_deactivation_scripts' );
        $selectors_to_deactivate_and_exclude_from_fui = Utils::convert_selectors_to_array( $selectors_to_deactivate_and_exclude_from_fui );

        /**
         * Convert the selectors to an array.
         */
        $selectors_to_exclude = BaseController::get_setting( 'jptgb_setting_exclude_scripts' );
        $selectors_to_exclude = Utils::convert_selectors_to_array( $selectors_to_exclude );

        /**
         * Merge both exclusions.
         */
        $selectors_to_exclude = array_merge( $selectors_to_deactivate_and_exclude_from_fui, $selectors_to_exclude );

        /**
         * Return if there are not selectors to exlucde.
         */
        if( !$selectors_to_exclude || empty( $selectors_to_exclude ) ){
            return false;
        }

        /**
         * Initially assume we don't have to exclude the array.
         */
        $is_to_exclude = false;

        /**
         * Check if any of the attributes names are set to exlcude.
         */
        foreach( $selectors_to_exclude as $selector_to_exclude ){
            /**
             * Check for id.
             */
            if( ( strpos( $selector_to_exclude, '#' ) !== false ) && in_array( 'id', $attribute_names ) ){
                /**
                 * Get the selector name.
                 */
                $selector_name = ltrim( $selector_to_exclude, '#' );

                /**
                 * Check if the selector match the id of the script.
                 */
                if( $attributes['id'] == $selector_name ){
                    $is_to_exclude = true;
                    break; 
                }
            } 

            /**
             * Check for classes.
             */
            if( ( strpos( $selector_to_exclude, '.' ) !== false ) && in_array( 'class', $attribute_names ) ){
                /**
                 * Get the selector name.
                 */
                $selector_name = ltrim( $selector_to_exclude, '.' );

                /**
                 * Get all class names of the selector.
                 */
                $all_classes = explode( ' ', $attributes['class'] );

                /**
                 * Iterate over each class
                 */
                foreach( $all_classes as $class_name ){
                    if( $class_name == $selector_name ){
                        $is_to_exclude = true;
                        break 2; 
                    }
                }
            } 
        }

        return $is_to_exclude;
    }

    /**
     * Updates each item in the array, setting 'is_jquery_core' to true if the
     * 'src' attribute within the 'tag' index contains 'jquery' (excluding cases where it contains 'jquery-migrate'),
     * and 'is_jquery_migrate' to true if it contains 'jquery-migrate'.
     *
     * @param array $scripts_data The array containing script elements.
     * @return void The array is modified directly since it's passed by reference.
     */
    private static function flag_jquery_scripts( array &$scripts_data ): void {
        foreach ($scripts_data as &$script) {
            // Reset flags.
            $script['is_jquery_core'] = false;
            $script['is_jquery_migrate'] = false;

            /** Check if the 'tag' index exists and is a string. */
            if (isset($script['tag']) && is_string($script['tag'])) {
                // Extract the 'src' attribute from the 'tag' index.
                $src_matches = [];
                preg_match('/src=["\']([^"\']+)["\']/i', $script['tag'], $src_matches);
                $src = $src_matches[1] ?? '';

                // Case-insensitive check for 'jquery' and 'jquery-migrate' in the src attribute.
                if (stripos($src, 'jquery-migrate') !== false) {
                    $script['is_jquery_migrate'] = true;
                } elseif (stripos($src, 'jquery') !== false) {
                    $script['is_jquery_core'] = true;
                }
            }
        }
        // No need to return anything since we're modifying the array by reference.
    }

    /**
     * Reorders the array to place items with 'is_jquery_core' set to true first,
     * then items with 'is_jquery_migrate' set to true, followed by the rest.
     *
     * @param array $scripts_data The array containing script elements.
     * @return array The reordered array with 'is_jquery_core' and 'is_jquery_migrate' prioritized.
     */
    private static function reorder_jquery_scripts(array $scripts_data): array {
        usort($scripts_data, function($a, $b) {
            // Check the flags for each script.
            $a_has_jquery_core   = $a['is_jquery_core'] ?? false;
            $b_has_jquery_core   = $b['is_jquery_core'] ?? false;
            $a_has_jquery_migrate = $a['is_jquery_migrate'] ?? false;
            $b_has_jquery_migrate = $b['is_jquery_migrate'] ?? false;

            // Prioritize 'is_jquery_core' first.
            if ($a_has_jquery_core && !$b_has_jquery_core) {
                return -1;
            } elseif (!$a_has_jquery_core && $b_has_jquery_core) {
                return 1;
            }

            // If neither or both are 'is_jquery_core', then check for 'is_jquery_migrate'.
            if (!$a_has_jquery_core && !$b_has_jquery_core) {
                if ($a_has_jquery_migrate && !$b_has_jquery_migrate) {
                    return -1;
                } elseif (!$a_has_jquery_migrate && $b_has_jquery_migrate) {
                    return 1;
                }
            }

            // If both are the same regarding jQuery core and migrate, maintain the original order.
            return 0;
        });

        return $scripts_data;
    }

    private static function get_unique_id() {
        /**
         * Using uniqid() function to generate a unique identifier.
         */
        return 'jptgb_tmpid_' . uniqid();
    }

    private static function create_script_for_fui_event( $handle, $attributes, $script_data ) {
        $order = isset( $attributes['data-jptgb-order'] ) ? (int) $attributes['data-jptgb-order'] : 99999;

        self::$scripts_loader_list[] = [
            'handle'            => $handle,
            'dependencies'      => [],
            'attributes'        => $attributes,
            'type'              => $script_data['tag_name'] ?? 'script',
            'in_footer'         => false,
            'order'             => $order,
            'tag_name'          => $script_data['tag_name'] ?? 'script',
        ];
    }

    /**
     * Extracts scripts from the given HTML, excluding scripts that contain 'jptgb-script' in the opening tag.
     * It includes both external and inline scripts and captures all attributes found in the script tags.
     *
     * @param string $html The HTML content to extract scripts from.
     * @return array An array of scripts with their detailed attributes and content, excluding specific scripts.
     */
    public static function extract_scripts($html) {
        $scripts            = [];
        $start              = 0;
        $scripts_to_skip    = [ 'jptgb-script', 'application/json', 'text/template', 'text/plain', 'application/ld+json' ];

        while (($start = strpos($html, '<script', $start)) !== false) {
            $endOfOpeningTag    = strpos($html, '>', $start) + 1;
            $scriptTagOpening   = substr($html, $start, $endOfOpeningTag - $start);
            $skip_script_found  = false;

            /**
             * Check if any any of the values in the $scripts_to_skip 
             * is found in the opening tag of the script.
             */
            foreach( $scripts_to_skip as $script_to_skipt ){
                if (strpos($scriptTagOpening, $script_to_skipt) !== false) {
                    $skip_script_found = true;
                    break;
                }
            }

            /**
             * Skip the script if any of the values in the $scripts_to_skip 
             * is found in the opening tag of the script.
             */
            if( $skip_script_found ){
                $start = $endOfOpeningTag;
                continue;
            }

            $nextScriptStart = strpos($html, '<script', $endOfOpeningTag);
            $endTagStart = strpos($html, '</script>', $start);
            
            // Determine the end of the current script tag
            if ($endTagStart !== false && ($nextScriptStart === false || $endTagStart < $nextScriptStart)) {
                $endTagEnd = $endTagStart + strlen('</script>');
            } else {
                $endTagEnd = $endOfOpeningTag; // Shorthand syntax, no closing tag
            }

            // Extract the full script tag
            $scriptTagFull = substr($html, $start, $endTagEnd - $start);
            $start = $endTagEnd;

            // Initialize attributes array
            $attributes = [
                'is_inline' => false,
                'attributes' => [],
            ];

            // Extract attributes and possibly the script content
            $pattern = '/([\w-]+)\s*=\s*[\'"]([^\'"]*)[\'"]/';
        
            // Execute regex to match all attributes and their values
            if (preg_match_all($pattern, $scriptTagOpening, $matches, PREG_SET_ORDER)) {
                foreach ($matches as $match) {
                    // Store the attribute and its value in the attributes array
                    $attributes['attributes'][$match[1]] = $match[2];
                }
            }
        
            if (strpos($scriptTagOpening, 'src=') === false) {
                $attributes['is_inline'] = true;
                $scriptContent = substr($html, $endOfOpeningTag, $endTagStart - $endOfOpeningTag);
                $attributes['content'] = trim($scriptContent);
            }

            if( isset( $attributes['attributes']['data-jptgb-order'] ) ){
                $attributes['order'] = (int) $attributes['attributes']['data-jptgb-order'];
            }

            $attributes['tag'] = $scriptTagFull;

            if (!empty($attributes)) {
                $scripts[] = $attributes;
            }
        }

        return $scripts;
    }

    /**
     * Get script dependencies with "-js" suffix.
     *
     * @param string $handle The script handle.
     * @return array Returns an array of dependencies with "-js" suffix.
     */
    public static function get_script_dependencies( $handle ) {
        /** Get global scripts variable */
        global $wp_scripts;

        /** Define initial dependencies */
        $dependencies = [];

        /** Check if this script handle is registered in $wp_scripts */
        if ( isset( $wp_scripts->registered[$handle] ) ) {
            /** Get the dependencies for this script */
            $dependencies = $wp_scripts->registered[$handle]->deps;

            /** Add "-js" suffix to each dependency */
            $dependencies = array_map(function($dependency) {
                return $dependency . '-js';
            }, $dependencies);
        }

        return $dependencies;
    }
}