<?php

namespace Jptgb\Includes;

use Jptgb\Controllers\BaseController;
use Jptgb\Controllers\AjaxController;
use Jptgb\Controllers\MainController;
use Jptgb\Includes\Settings;
use Dotenv\Dotenv;
use Jptgb\Views\Partials\AdminHeader;
use Jptgb\Views\Partials\AdminFooter;
use Jptgb\Resources\Utils;

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://yaidier.com
 * @since      1.0.0
 *
 * @package    Just_Press_The_Green_Button
 * @subpackage Just_Press_The_Green_Button/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Jptgb
 * @subpackage Jptgb/Includes
 * @author     Yaidier Perez <yaidier.perez@gmail.com>
 */
class Jptgb extends BaseController {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Just_Press_The_Green_Button_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
        // /**
        //  * Check if the .env file exists.
        //  */
        // if( file_exists( JPTGB_PATH . '/.env' ) ) {
        //     $dotenv = Dotenv::createImmutable( JPTGB_PATH );
        //     $dotenv->load();
        // }

        // Path to your custom env file
        $env_file = JPTGB_PATH . 'env';

        /** Check if our custom env file exists, then load it */
        if ( file_exists( $env_file ) ) {
            /** 
             * Create the Dotenv loader, pointing at JPTGB_PATH and instructing it
             * to read “env” (not “.env”).
             */
            $dotenv = Dotenv::createImmutable( JPTGB_PATH, 'env' );
            $dotenv->load();
        }

        /**
         * Define the version
         */
		if ( defined( 'JPTGB_VERSION' ) ) {
			$this->version = JPTGB_VERSION;
		} else {
			$this->version = '1.0.0';
		}

        /**
         * Define the plugin name.
         */
		$this->plugin_name = 'just-press-the-green-button';

        /**
         * Initialize the base controller.
         */
        BaseController::init();

        /**
         * Initiazlize the ajax controller.
         */
        AjaxController::init();

        /**
         * Init the optimizations controller.
         */
        MainController::init();

        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->admin_messages();
        $this->handle_free_account();
	}

    public function admin_messages(){
        /**
         * 
         */
        $is_free_account_created = self::get_setting( 'free_account_created' );
        if( $is_free_account_created ){
            self::add_notification( 'Free Account created Successfully.' );
            self::update_setting( 'free_account_created', false );
        }

        if ( self::is_settings_update() ) {
            self::add_notification( 'Settings saved.' );
        }

        /**
         * Get the api key status.
         */
        $api_key_status = self::get_setting('jptgb_setting_is_api_key_valid');

        /**
         * Display this message if the API key is incorrect.
         */
        if( 'error' === $api_key_status ) {
            add_action( 'init', function() {
                self::add_notification( 'Your <b>API Key</b> is not correct. Please enter the right API Key in <a href="' . esc_url( admin_url( 'admin.php?page=jptgb-settings&tab=1' ) ) .'">Settings</a>', 'error' );

                $free_account_nonce = wp_create_nonce( 'webspeed-get-free-account' );
                self::add_notification( 'You can create a <b>Free Account</b> and obtain an API Key simply by clicking <a href="' . esc_url( admin_url( 'admin.php?page=jptgb-settings&tab=1&request-free-account=' . $free_account_nonce ) ) .'">here</a>', 'info' );
            });
        }

        /**
         * Display this message if the API key is incorrect.
         */
        if( !$api_key_status ) {
            add_action( 'init', function() {
                self::add_notification( 'Your <b>API Key</b> is missing. Please enter your API Key in <a href="' . esc_url( admin_url( 'admin.php?page=jptgb-settings&tab=1' ) ) .'">Settings</a>', 'error' );

                $free_account_nonce = wp_create_nonce( 'webspeed-get-free-account' );
                self::add_notification( 'You can create a <b>Free Account</b> and obtain an API Key simply by clicking <a href="' . esc_url( admin_url( 'admin.php?page=jptgb-settings&tab=1&request-free-account=' . $free_account_nonce ) ) .'">here</a>', 'info' );
            } );
        }
    }

    public static function handle_free_account() {
        if ( empty( $_GET['request-free-account'] ) ) {
            return;
        }
       
        
        add_action( 'init', function(){
            // set_transient( 'webspeed_free_account_requested', true, 30 );
            BaseController::$logs->register( 'Getting free API key for the site', 'info' );

            if( !wp_verify_nonce( $_GET['request-free-account'], 'webspeed-get-free-account' ) ){
                return;
            }

            // Generate & store a one-time token.
            $token = \wp_generate_password( 20, false, false );


            self::update_setting( 'verify_token', $token );

            // Build the request.
            $domain = Utils::get_site_doamin();

            $endpoint = BaseController::get_env_variable( 'JPTGB_SITE_DOMAIN' ) . '/wp-json/my-api/v1/generate-key';

            $response = wp_remote_post( $endpoint, array(
                'body'    => array(
                    'domain' => sanitize_text_field( $domain ),
                    'token'  => $token,
                ),
                'timeout' => 20,
                'sslverify' => false,
            ) );

            if ( is_wp_error( $response ) ) {
                /**
                 * Log the cache flush.
                 */
                BaseController::$logs->register( 'API key request failed: ' . $response->get_error_message(), 'error' );
                return;
            }

            $data = json_decode( wp_remote_retrieve_body( $response ), true );
            if ( ! empty( $data['api_key'] ) ) {
                self::update_setting( 'api_key', $data['api_key'] );

            } else {
                BaseController::$logs->register( 'API key not returned: ' . wp_remote_retrieve_body( $response ), 'error' );

            }

            $response = BaseController::sync_with_api();

            /**
             * Check api key.
             */
            self::set_api_key_status();
            self::update_setting( 'free_account_created', true );

            wp_safe_redirect( esc_url( admin_url( 'admin.php?page=jptgb-settings&tab=1' ) ) );
        } );
    }

    public function admin_notifications() {
        foreach( self::$notifications as $notification ) {
            $status     = $notification['status'];
            $message    = $notification['message'];

            ?>
                <div class="notice notice-<?php echo esc_attr( $status ) ?> is-dismissible">
                    <p><?php echo wp_kses_post( $message ) ?></p>
                </div>
            <?php
        }
    }

    public function enqueue_admin_scripts() {
        /**
         * Get the file versions.
         */
        $version_style  = $this->get_mix_file_version( 'jptgb-admin.css' );
        $version_script = $this->get_mix_file_version( 'jptgb-admin.js' );

        /**
         * Enqueue the style on the admin.
         */
        wp_enqueue_style( 'jptgb-admin-styles', esc_url( JPTGB_URL . 'assets/css/jptgb-admin.css' ), [], $version_style );

        /**
         * Enqueue the script on the admin.
         */
        wp_enqueue_script( 'jptgb-admin-app', esc_url( JPTGB_URL . 'assets/js/jptgb-admin.js' ), [], $version_script, true );

        /**
         * Define data to send to js.
         */
        $data = [
            'ajaxUrl'                               => esc_url( admin_url( 'admin-ajax.php' ) ),
            'requestNewApiNonce'                    => wp_create_nonce( 'jptgb_newapi_nonce' ),
            'testApiEndpointNonce'                  => wp_create_nonce( 'jptgb_test_api_endpoint_nonce' ),
            'deleteUserNonce'                       => wp_create_nonce( 'jptgb_deleteuser_nonce' ),
            'flushAllCacheNonce'                    => wp_create_nonce( 'jptgb_flush_all_cache_nonce' ),
            'generateAllPagesCacheNonce'            => wp_create_nonce( 'jptgb_generate_all_pages_cache_nonce' ),
            'deletePluginsDataNonce'                => wp_create_nonce( 'jptgb_delete_plugins_data_nonce' ),
            'flushCcNonce'                          => wp_create_nonce( 'jptgb_flush_all_cc_nonce' ),
            'forceFlushCcNonce'                     => wp_create_nonce( 'jptgb_force_flush_all_cc_nonce' ),
            'flushAllImagesNonce'                   => wp_create_nonce( 'jptgb_flush_all_images_nonce' ),
            'toolsColumnGenerateCacheNonce'         => wp_create_nonce( 'jptgb_tools_column_generate_cache_nonce' ),
            'toolsColumnGetWorkingOnStatusNonce'    => wp_create_nonce( 'jptgb_tools_column_get_working_on_status_nonce' ),
            'syncApiNonce'                          => wp_create_nonce( 'jptgb_sync_api_nonce' ),
        ];

        /**
         * Send the data to the js app.
         */
        wp_add_inline_script( 'jptgb-admin-app', 'const jpgbData = ' . json_encode( $data ) .';' );
    }

    public function load_dependencies(){
        /**
         * Enqueue admin scripts.
         */
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );

        /**
         * Enqueue scripts.
         */
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
    }

    public function enqueue_scripts() {
        /**
         * Do not enqueue admin bar scripts if the user is not logged-in.
         */
        if ( !is_user_logged_in() || !current_user_can( 'manage_options' ) ) {
            return;
        }

        /**
         * Get the asset's versions.
         */
        $version_script = $this->get_mix_file_version( 'jptgb-adminbar.js' );
        $version_style  = $this->get_mix_file_version( 'jptgb-adminbar.css' );

        /**
         * Enqueue the style for the admin bar.
         */
        wp_enqueue_style( 'jptgb-admin-styles', esc_url( JPTGB_URL . 'assets/css/jptgb-adminbar.css' ), [], $version_style );

        /**
         * Enqueue the script for the admin bar.
         */
        wp_enqueue_script( 'jptgb-adminbar-app', esc_url( JPTGB_URL . 'assets/js/jptgb-adminbar.js' ), [], $version_script, false );
        wp_localize_script('jptgb-adminbar-app', 'jptgbCacheData', [
            'cacheClearedMessage' => get_transient('jptgb_cache_cleared_message')
        ]);
    
        /**
         * Delete the transient so it's only shown once.
         */
        delete_transient('jptgb_cache_cleared_message');
    }

    /**
     * Retrieves the version hash of a specified file from the Laravel Mix manifest.
     *
     * @param string $filename The filename to retrieve the version for.
     * @return string|null The version hash, or null if not found.
     */
    function get_mix_file_version( $filename ) {
        $manifest_path = JPTGB_PATH . 'mix-manifest.json';

        // Check if the manifest file exists.
        if ( ! file_exists( $manifest_path ) ) {
            return null;
        }

        // Read and decode the manifest file.
        $manifest = json_decode( file_get_contents( $manifest_path ), true );

        // Loop through the manifest to find a match.
        foreach ( $manifest as $key => $value ) {
            if ( strpos( $key, $filename ) !== false ) {
                // Parse the URL to get the query string, then extract the version.
                $query = wp_parse_url( $value, PHP_URL_QUERY );
                parse_str( $query, $params );
                return $params['id'] ?? null;
            }
        }

        return null;
    }

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {
        /** 
         * Register the admin pages.
         */
        add_action( 'admin_menu', [ $this, 'register_settings_page' ], PHP_INT_MAX );

        /**
         * Handle the admin notifiactions in our page.
         */
        add_action( 'admin_head', function() {
            /**
             * Get the current screen object.
             */
            $screen = get_current_screen();

            /**
             * Check if we're on the plugin's admin page.
             */
            if( !isset( $screen->id ) || ( $screen->id !== 'webspeed-ai_page_jptgb-settings' && $screen->id !== 'toplevel_page_jptgb' ) ) {
                return;
            }

            /**
             * Suppress all notices for this page.
             */
            remove_all_actions( 'admin_notices' );

            /**
             * Check if the 'admin_message' and 'admin_status' query parameters are present.
             */
            add_action('admin_notices', function () {
                if( !isset( $_GET[ 'admin_message' ] ) || !isset( $_GET[ 'admin_status' ] ) ) {
                    return;
                }

                /**
                 * Get message info.
                 */
                $message    = sanitize_text_field( urldecode( $_GET['admin_message'] ) );
                $status     = sanitize_text_field( urldecode( $_GET['admin_status'] ) );
                
                /**
                 * Display the admin notice.
                 */
                echo sprintf( '<div class="notice %s"><p>%s</p></div>', esc_attr( $status ), esc_html( $message ) );
            });

            /**
             * Notifications.
             */
            add_action( 'admin_notices', [ $this, 'admin_notifications' ] );
        }, PHP_INT_MAX );
	}

    public function register_settings_page() {
        global $menu;

        // Default position (fallback if "Settings" menu isn't found).
        $menu_position = 81;
    
        // Dynamically find the position of the "Settings" menu.
        foreach ( $menu as $key => $menu_item ) {
            if ( isset( $menu_item[2] ) && $menu_item[2] === 'options-general.php' ) {
                $menu_position = $key + 1; // Insert right after "Settings".
                break;
            }
        }
    
        // Register the custom menu.
        add_menu_page(
            'WebSpeed AI',                   // Page title
            'WebSpeed AI',                   // Menu title
            'manage_options',                // Capability
            'jptgb',                         // Menu slug
            [ $this, 'dashboard_page' ], // Callback function for the main menu page
            'dashicons-performance',         // Icon URL or Dashicons class
            $menu_position                   // Menu position
        );
    
        // Add a subpage for plugin settings under the custom menu.
        add_submenu_page(
            'jptgb',                         // Parent slug
            'WebSpeed AI Settings',          // Page title
            'Settings',                      // Submenu title
            'manage_options',                // Capability
            'jptgb-settings',                // Submenu slug
            [ $this, 'settings_page' ]       // Callback function for the settings page
        );

        // Safely grab & sanitize the 'my_param' GET var
        if( isset( $_GET['jptgb-remove-all-logs'] ) ) {
            self::$logs->delete_all_logs();
        }
    }

    /**
     * Display callback for the top-level menu
     */
    public function settings_page() {
        AdminHeader::render();
        Settings::render();
        AdminFooter::render();
    }

    /**
     * Display callback for the top-level menu
     */
    public function dashboard_page() {
        AdminHeader::render();
        Dashboard::render();
        AdminFooter::render();
    }

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
}
