<?php

namespace Jptgb\Includes;

use Jptgb\Controllers\BaseController;

/**
 * Handles logging various actions and errors into the database.
 */
class Logs extends BaseController {
    /** @var string Name of the database table for logs. */
    private $table_name;

    /**
     * Constructs the log object and sets the table name.
     */
    public function __construct() {
        global $wpdb;

        $this->table_name = $wpdb->prefix . 'jptgb_logs';
        $this->init_log_table();
    }

    /**
     * Logs a message to the database.
     * Only 'success', 'info', 'warning', and 'error' types are allowed.
     *
     * @param string $message The message to log.
     * @param string $type    The type of log entry. Valid values: 'success', 'info', 'warning', 'error'.
     * @return void
     * @throws \InvalidArgumentException If an invalid log type is provided.
     */
    public function register( string $message, string $type = 'info' ): void {
        // Return early if logging is disabled via settings.
        if ( ! self::get_setting( 'activate_logs' ) ) {
            return;
        }

        // Validate log type (strict check).
        if ( ! in_array( $type, [ 'success', 'info', 'warning', 'error' ], true ) ) {
            throw new \InvalidArgumentException( 'Invalid log type specified.' );
        }

        global $wpdb;

        // Insert the new log entry.
        $wpdb->insert(
            $this->table_name,
            [
                'message'    => $message,
                'type'       => $type,
                'created_at' => current_time( 'mysql' ), // Current timestamp (WP‐formatted).
            ],
            [
                '%s', // message
                '%s', // type
                '%s', // created_at
            ]
        );

        // After insertion, check total count and delete oldest if more than 1,000 entries.
        $total_logs = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$this->table_name}`" ) );

        if ( $total_logs > 1000 ) {
            // Calculate how many excess rows we have.
            $excess = $total_logs - 1000;

            // Delete the oldest $excess rows, ordered by the primary key (id).
            $wpdb->query( $wpdb->prepare( "DELETE FROM `{$this->table_name}` ORDER BY `id` ASC LIMIT %d", $excess ) );
        }
    }

    /**
     * Retrieves all logs from the database (no pagination).
     *
     * @return array Returns an array of log entry objects.
     */
    public function get_all_logs(): array {
        global $wpdb;

        return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `{$this->table_name}` ORDER BY `id` DESC" ), OBJECT );
    }

    /**
     * Retrieves the total number of log entries in the database.
     *
     * @return int Total count of logs.
     */
    public function get_total_log_count(): int {
        global $wpdb;
        return (int) $wpdb->get_var( "SELECT COUNT(*) FROM `{$this->table_name}`" );
    }

    /**
     * Retrieves a paginated set of log entries.
     *
     * @param int $page     Page number (1-based). Defaults to 1.
     * @param int $per_page Number of logs per page. Defaults to 50.
     * @return array Array of log entry objects for the requested page.
     */
    public function get_logs_page( int $page = 1, int $per_page = 50 ): array {
        global $wpdb;

        // Ensure $page and $per_page are positive integers.
        $page     = max( 1, $page );
        $per_page = max( 1, $per_page );

        $offset = ( $page - 1 ) * $per_page;

        return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `{$this->table_name}` ORDER BY `id` DESC LIMIT %d OFFSET %d", $per_page, $offset ), OBJECT );
    }

    /**
     * Creates the log table if it does not exist.
     *
     * Important: dbDelta() expects a plain “CREATE TABLE `table_name` (…)”
     * (no “IF NOT EXISTS”). It will handle existence checks internally.
     *
     * @return void
     */
    private function init_log_table(): void {
        global $wpdb;

        // Get the correct charset/collate for this installation.
        $charset_collate = $wpdb->get_charset_collate();

        // Build the SQL WITHOUT “IF NOT EXISTS”
        $sql = "
            CREATE TABLE `{$this->table_name}` (
                `id` bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
                `message` text NOT NULL,
                `type` varchar(100) NOT NULL DEFAULT 'info',
                `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) {$charset_collate};
        ";

        // Load the upgrade functions (dbDelta is defined here).
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        // dbDelta() will create the table if it doesn’t exist, or update it if it does.
        dbDelta( $sql );
    }

    /**
     * Deletes all logs from the database.
     *
     * @return bool True on success; false on failure.
     */
    public function delete_all_logs(): bool {
        global $wpdb;

        $deleted = $wpdb->query( $wpdb->prepare( "DELETE FROM `{$this->table_name}`" ) );

        if ( false === $deleted ) {
            // There was an error in the SQL execution.
            return false;
        } elseif ( $deleted >= 0 ) {
            // Delete was successful; $deleted contains number of rows affected.
            return true;
        }

        // Fallback in case something unexpected occurs.
        return false;
    }
}