<?php

namespace Jptgb\Includes;

use Jptgb\Resources\Utils;

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class PostListTable extends \WP_List_Table {

    private $post_type;

    public function __construct( $post_type ) {
        parent::__construct( [
            'singular' => 'post',
            'plural'   => 'posts',
            'ajax'     => false, // Disable AJAX for simplicity.
        ] );

        $this->post_type = $post_type;
    }

    /**
     * Define the columns for the table.
     */
    public function get_columns() {
        return [
            'title'  => __( 'Title', 'webspeed' ),
            'jptgb-optimizations_column' => __( 'WebSpeed', 'webspeed' ),
        ];
    }

    /**
     * Define sortable columns.
     */
    public function get_sortable_columns() {
        return [
            'title' => [ 'title', true ], // true = default sorting by title (ascending).
        ];
    }

    /**
     * Prepare the items for display.
     */
    public function prepare_items() {
        $columns = $this->get_columns();
        $hidden = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [ $columns, $hidden, $sortable ];

        // Pagination parameters.
        $per_page = $this->get_items_per_page( 'posts_per_page', 14 );
        $current_page = $this->get_pagenum();

        // Determine the sorting criteria.
        $orderby = ! empty( $_GET['orderby'] ) ? sanitize_text_field( $_GET['orderby'] ) : 'title';
        $order = ! empty( $_GET['order'] ) ? sanitize_text_field( $_GET['order'] ) : 'asc';

        // Handle search query.
        $search = ! empty( $_GET['s'] ) ? sanitize_text_field( $_GET['s'] ) : '';

        // Query the posts.
        $args = [
            'post_type'      => $this->post_type,
            'post_status'    => 'publish',
            'posts_per_page' => $per_page,
            'paged'          => $current_page,
            'orderby'        => $orderby,
            'order'          => $order,
        ];

        if ( ! empty( $search ) ) {
            $args['s'] = $search; // Add search query.
        }

        $query = new \WP_Query( $args );

        $this->items = $query->posts;

        // Set pagination.
        $this->set_pagination_args( [
            'total_items' => $query->found_posts,
            'per_page'    => $per_page,
            'total_pages' => $query->max_num_pages,
        ] );
    }

    /**
     * Add the search box to the list table.
     */
    public function display_search_box( $text ) {
        $this->search_box( $text, 'search_id' );
    }

    /**
     * Render a column when no specific column method is defined.
     */
    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'title':
                $title = sprintf(
                    '<a class="row-title" href="%s">%s</a>',
                    get_edit_post_link( $item->ID ),
                    esc_html( $item->post_title )
                );

                $actions = [
                    'edit' => sprintf(
                        '<a href="%s">%s</a>',
                        get_edit_post_link( $item->ID ),
                        __( 'Edit', 'webspeed' )
                    ),
                    'trash' => sprintf(
                        '<a href="%s" class="submitdelete">%s</a>',
                        get_delete_post_link( $item->ID ),
                        __( 'Trash', 'webspeed' )
                    ),
                    'view' => sprintf(
                        '<a href="%s">%s</a>',
                        Utils::get_permalink( $item->ID ),
                        __( 'View', 'webspeed' )
                    ),
                ];

                return sprintf(
                    '%s <div class="row-actions">%s</div>',
                    $title,
                    $this->row_actions( $actions )
                );

            case 'jptgb-optimizations_column': ColumnPanel::render( $item->ID );

            default:
                return ''; // No other columns defined.
        }
    }
}