<?php

namespace Jptgb\Resources;

use Jptgb\Controllers\BaseController;

/**
 * The Cache Controller Class
 *
 * @author Yaidier Perez
 * */

if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * This class handles Cloudways - CloudFlare microservice action.
 *
 * @since 2.0.15
 * @final
 */
final class CloudflareHelper {

	private $cw_platform = '';

	function __construct() {
		// add_action( 'switch_theme', array( &$this, 'clear_cf_on_changing_theme' ), 11, 3 );
	}

	/**
	 * Define Microservice url.
	 *
	 * @return false|string
	 * @since 2.0.15
	 * @private
	 */
	private function get_microservice_url() {
		if ( false === self::is_cloudflare_enabled() ) {
			return false;
		}
		$fpc_microservice_url = ''; // default
		/**
		 * Contains the dynamic microservice URL.
		 */
		if ( true === self::is_fp_server() ) {
			$this->cw_platform    = 'fp';
			$fpc_microservice_url = getenv( 'FPC_ENV' ); // TODO Add the hardcoded link for Flexible (stating|production).

			if ( true === self::is_log_enabled() ) {
				$server_type_text = '';
				if ( true === self::is_staging_server() ) {
					$server_type_text = 'Staging';
				}

				if ( true === self::is_production_server() ) {
					$server_type_text = 'Production';
				}

				BaseController::error_log( 'Cloudways FP (Flexible) is ON: ' . $server_type_text );
			}
		} elseif ( ! empty( getenv( 'FPC_ENV' ) ) ) {
			$this->cw_platform    = 'fmp';
			$fpc_microservice_url = getenv( 'FPC_ENV' );// FMP
			if ( true === self::is_log_enabled() ) {
				BaseController::error_log( 'Cloudways FMP (Autoscale) is ON ' );
			}
		}

		if ( true === self::is_log_enabled() ) {
			BaseController::error_log( 'Microservice URL: ' . var_export( $fpc_microservice_url, true ) );
		}

		if ( empty( $fpc_microservice_url ) ) {
			return false;
		}

		return trailingslashit( $fpc_microservice_url );
	}

	/**
	 * Purge Cloudflare cache on theme switch.
	 *
	 * @param string $new_name Name of the new theme.
	 * @param string $new_theme WP_Theme instance of the new theme.
	 * @param string $old_theme WP_Theme instance of the old theme.
	 *
	 * @return void
	 * @since 2.0.15
	 * @public
	 */
	public function clear_cf_on_changing_theme( string $new_name, string $new_theme, string $old_theme ) {
		$list_of_urls[] = get_home_url();
		CloudflareHelper::reset_all_cache( $list_of_urls );
	}

	/**
	 * Clear the cache for the given url list.
	 * Needs at least one element.
	 *
	 * @param array $specific_urls Array with the list of URLs.
	 *
	 * @return bool|string|null
	 * @since 2.0.15
	 * @access public
	 * @static
	 */
	static public function purge_cloudflare_cache_urls( array $specific_urls = array() ) {
		// If we do not have everything that we need, stop the process.
		if ( true === self::is_log_enabled() ) {
			BaseController::error_log( '######### PURGE CLOUDFLARE ###: ' . var_export( 'Single URL START', true ) );
		}
		if ( false === self::is_cloudflare_enabled() ) {
			return false;
		}
		// Remove any non URL items.
		$specific_urls = ( new CloudflareHelper )->remove_not_url_elements( $specific_urls );
		if ( true === self::is_log_enabled() ) {
			BaseController::error_log( 'single url : ' . var_export( $specific_urls, true ) );
		}

		// Call cache reset.
		return ( new CloudflareHelper )->request_cache_reset( $specific_urls, 'purge-fpc-url' );
	}

	/**
	 * Purge all cache in CloudFlare.
	 * In multisite clears for all sub-sites.
	 *
	 * @param array $home_url Used by WP-CLI
	 *
	 * @return bool|string|null
	 * @since 2.0.15
	 * @access public
	 * @static
	 */
	static public function reset_all_cache( array $home_url = array() ) {
		// If we do not have everything that we need, stop the process.

		if ( ! is_array( $home_url ) ) {
			$home_url = array();
		}

		/**
		 * Execute code if this function is not called by WP-CLI.
		 */
		if ( 'cli' !== php_sapi_name() ) {

			// For multisite network, clear cache for all sub-sites.
			if ( ( is_multisite() && is_network_admin() ) ) {
				$blogs = get_sites();
				foreach ( $blogs as $blog_data ) {
					$url        = get_home_url( $blog_data->blog_id );
					$home_url[] = trailingslashit( $url );
				}
			} else {
				$home_url[] = trailingslashit( home_url() );
			}
		}

		$purge_request_endpoint = 'purge-fpc-domain';

		if ( is_multisite() ) {
			if ( is_subdomain_install() ) {
				$home_url = ( new CloudflareHelper )->clear_domain_purge_urls( $home_url );
				if ( true === self::is_log_enabled() ) {
					BaseController::error_log( '######### CF SubDomains: ' . var_export( $home_url, true ) );
				}
			} else {
				$purge_request_endpoint = 'purge-fpc-sub-dir';
				if ( ! empty( $home_url ) ) {
					foreach ( $home_url as &$url ) {
						$url = untrailingslashit( $url );
					}
					if ( true === self::is_log_enabled() ) {
						BaseController::error_log( '######### CF SubDirectory: ' . var_export( $home_url, true ) );
					}
				}
			}
		} else {
			$home_url = ( new CloudflareHelper )->clear_domain_purge_urls( $home_url );
			if ( true === self::is_log_enabled() ) {
				BaseController::error_log( '######### CF Domain: ' . var_export( $home_url, true ) );
			}
		}

		return ( new CloudflareHelper )->request_cache_reset( $home_url, $purge_request_endpoint );
	}

	/**
	 * Clear the list of URLs of HTTP schema and remove the slash at the end.
	 * This is needed for domain CF purge.
	 *
	 * @param array $url_list List of URLs.
	 *
	 * @return array
	 */
	private function clear_domain_purge_urls( array $url_list = array() ): array {
		if ( empty( $url_list ) ) {
			return $url_list;
		}

		foreach ( $url_list as &$url ) {
			$url = trim( $url );
			$url = ltrim( $url, 'https:' );
			$url = ltrim( $url, '//' );
			$url = untrailingslashit( $url );
		}

		return $url_list;
	}

	/**
	 * Remove all array elements which are not a valid URL.
	 *
	 * @param array $url_list Given url list.
	 *
	 * @return array
	 *
	 * @access private
	 * @since 2.0.15
	 */
	private function remove_not_url_elements( array $url_list = array() ): array {
		// Remove any white spaces from URL list.
		$url_list = array_map( 'trim', $url_list );
		// Making sure there are no duplicates.
		$url_list = array_unique( $url_list );

		return array_filter(
			$url_list,
			function ( $value, $index ) {
				return false !== filter_var( $value, FILTER_VALIDATE_URL );
			},
			ARRAY_FILTER_USE_BOTH
		);

	}

	/**
	 * Will return true if defined constants are found.
	 *
	 * @return bool
	 *
	 * @since 2.0.15
	 * @access public
	 * @static
	 */
	static public function is_cloudflare_enabled(): bool {
		$return_value = true;

		if( ! defined( 'CDN_SITE_ID' ) || ! defined( 'CDN_SITE_TOKEN' ) ) {
			$return_value = false;
		}

		if ( false === self::is_cloudways_server() ) {
			$return_value = false;
		}

		return $return_value;
	}

	/**
	 * Detect if it's Cloudways server.
	 *
	 * @return bool
	 * @access public
	 * @since 2.0.19
	 */
	static public function is_cloudways_server(): bool {

		if (
			false !== strpos( $_SERVER['DOCUMENT_ROOT'], 'cloudwaysapps' ) ||
			false !== strpos( $_SERVER['DOCUMENT_ROOT'], 'cloudwaysstagingapps' ) ||
			! empty( getenv( 'FPC_ENV' ) )
		) {
			return true;
		}

		return false;
	}

	static public function is_fmp_server(): bool {

		if (
			! empty( getenv( 'FPC_ENV' ) ) &&
			isset( $_SERVER['HTTP_CF_WORKER'] )
		) {

			if ( true === self::is_log_enabled() ) {
				if ( false !== strpos( getenv( 'FPC_ENV' ), 'uat-' ) ) {
					BaseController::error_log( '# Microservice Server URL UAT: ON ' );
				}

				if ( false !== strpos( getenv( 'FPC_ENV' ), 'stg-' ) ) {
					BaseController::error_log( '# Microservice Server URL STG: ON ' );
				}

				if ( false !== strpos( getenv( 'FPC_ENV' ), 'dev-' ) ) {
					BaseController::error_log( '# Microservice Server URL DEV: ON ' );
				}

				if ( false !== strpos( getenv( 'FPC_ENV' ), 'prod-' ) ) {
					BaseController::error_log( '# Microservice Server URL PROD: ON ' );
				}
			}

			return true;
		}

		return false;
	}

	/**
	 * Detect if it's Cloudways staging server.
	 *
	 * @return bool
	 * @access public
	 * @since 2.0.19
	 */
	static public function is_staging_server(): bool {

		if (
			false !== strpos( $_SERVER['DOCUMENT_ROOT'], 'cloudwaysstagingapps.com' )
		) {
			if ( true === self::is_log_enabled() ) {
				BaseController::error_log( 'Cloudways Staging ON ' );
			}

			return true;
		}

		return false;
	}

	/**
	 * Detect if it's Cloudways staging server.
	 *
	 * @return bool
	 * @access public
	 * @since 2.0.19
	 */
	static public function is_production_server(): bool {

		if (
			false !== strpos( $_SERVER['DOCUMENT_ROOT'], 'cloudwaysapps.com' )
		) {
			if ( true === self::is_log_enabled() ) {
				BaseController::error_log( 'Cloudways Production ON ' );
			}

			return true;
		}

			return false;
		}

	/**
	 * Check if the server type is FP ( Flexible ).
	 *
	 * @return bool
	 * @since 2.0.15
	 */
	static public function is_fp_server(): bool {
		if ( true === self::is_staging_server() || true === self::is_production_server() ) {
		return true;
	}

		return false;
	}

    /**
     * Handle a cache-purge request via the WP HTTP API.
     *
     * @param array  $purge_url_list List of URLs for which to purge cache.
     * @param string $endpoint_path  Endpoint path to clear URL cache or whole domain cache.
     * @return string|false|null     The raw response body on success, false on error, or null if skipped.
     */
    private function request_cache_reset( array $purge_url_list = [], string $endpoint_path = 'purge-fpc-url' ) {
        // Bail early if Cloudflare integration is off or no URLs passed.
        if ( false === self::is_cloudflare_enabled() || empty( $purge_url_list ) ) {
            return null;
        }

        // Ensure we’re on an FMP server if that matters.
        self::is_fmp_server();

        // Clean up URL list: trim, filter out empties, dedupe, reindex.
        $purge_url_list = array_values( array_unique( array_filter( array_map( 'trim', $purge_url_list ) ) ) );
        if ( empty( $purge_url_list ) ) {
            return null;
        }

        // SSL verification settings.
        $ssl_verification = apply_filters( 'jptgb_ssl_check_certificate', true );
        $ssl_verification = is_bool( $ssl_verification ) ? $ssl_verification : true;

        // If debugging, turn it off so dev environments don’t choke on self-signed certs.
        if ( defined( 'WP_DEBUG' ) && true === WP_DEBUG ) {
            $ssl_verification = false;
        }

        // Build endpoint URL.
        $microservice_url = $this->get_microservice_url();
        if ( false === $microservice_url ) {
            if ( true === self::is_log_enabled() ) {
                BaseController::error_log( 'Error: Microservice URL is not defined' );
            }
            return false;
        }
        $url = rtrim( $microservice_url, '/' ) . '/' . ltrim( $endpoint_path, '/' );

        if ( true === self::is_log_enabled() ) {
            BaseController::error_log( '>> Purge endpoint: ' . $url );
        }

        // Prepare JSON payload.
        $payload = wp_json_encode( [
            'urls'     => $purge_url_list,
            'appToken' => CDN_SITE_TOKEN,
            'appId'    => CDN_SITE_ID,
            'platform' => $this->cw_platform,
        ] );

        if ( true === self::is_log_enabled() ) {
            BaseController::error_log( '>> Payload URLs: ' . var_export( $purge_url_list, true ) );
            BaseController::error_log( '>> Platform: ' . var_export( strtoupper( $this->cw_platform ), true ) );
        }

        // WP HTTP API args
        $args = [
            'method'      => 'POST',
            'timeout'     => 4,
            'redirection' => 2,
            'blocking'    => true,
            'headers'     => [
                'Accept'       => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'body'        => $payload,
            'user-agent'  => 'jptgb-plugin-cache-reset',
            'sslverify'   => $ssl_verification,
        ];

        // Send the request.
        $response = wp_remote_post( $url, $args );

        // Handle errors.
        if ( is_wp_error( $response ) ) {
            if ( true === self::is_log_enabled() ) {
                BaseController::error_log( 'WP HTTP Error: ' . $response->get_error_message() );
            }
            return false;
        }

        // Grab the raw body.
        $body = wp_remote_retrieve_body( $response );

        if ( true === self::is_log_enabled() ) {
            BaseController::error_log( '<< Purge response: ' . var_export( $body, true ) );
        }

        return $body;
    }

	/**
	 * Check if WP_DEBUG is set to true.
	 * if true then enable logs for this library.
	 *
	 * @return bool
	 *
	 * @since 2.0.15
	 * @access public
	 * @static
	 */
	static public function is_log_enabled(): bool {
        return true;
	}
}

new CloudflareHelper();