<?php 

namespace Jptgb\Resources;

use Jptgb\Controllers\BaseController;

// use 

class SitegroundHelper extends BaseController {
    /**
     * SiteTools Client Unix Socket.
     *
     * @since 7.2.7
     *
     * @var string Path to the SiteTools UNIX socket file.
     */
    const SITE_TOOLS_SOCK_FILE = '/chroot/tmp/site-tools.sock';

    public static function flulsh_cache( $post_id, $requested_url ){
        try {       
            // 1. Verify SG Optimizer’s Supercacher_Posts class is available.
            if( class_exists( 'SiteGround_Optimizer\Supercacher\Supercacher_Posts' ) ) {
                BaseController::$logs->register( 'It looks like the SG Optimizer is available, we will use it to try to flush the cache for: ' . esc_url( $requested_url ), 'info' );
                // SG Optimizer (SuperCacher) is not active, or this class cannot be found.
                //    (The plugin itself does not expose a get_instance(), so we create a new one.)
                $sg_posts = new \SiteGround_Optimizer\Supercacher\Supercacher_Posts();
    
                // 3. Call the cache‐purge method for this post.
                $sg_posts->purge_all_post_cache( $post_id );
    
            } else {
                BaseController::$logs->register( 'SG Optimizer is not available, we will try manually to flush the cache for: ' . esc_url( $requested_url ), 'info' );
    
                $result = self::attempt_to_flush_siteground_cache_manually( $post_id );
    
                if( $result ){
                    BaseController::$logs->register( 'Siteground cache flushed successfully manually for: ' . esc_url( $requested_url ), 'info' );
                }
            }
        } catch (\Throwable $th) {
            $error_message = 'We had an issue while flushing Siteground Dynamic Cache for: '. esc_url( $requested_url ) .' | Error Message: ' . $th->getMessage();
            BaseController::$logs->register( $error_message, 'error' );
        }
    }

    /**
     * Checks if the plugin run on the new SiteGround interface.
     *
     * @since  1.0.0
     *
     * @return boolean True/False.
     */
    public static function is_siteground() {
        // Bail if open_basedir restrictions are set, and we are not able to check certain directories.
        if ( ! empty( ini_get( 'open_basedir' ) ) ) {
            return 0;
        }

        return (int) ( @file_exists( '/etc/yum.repos.d/baseos.repo' ) && @file_exists( '/Z' ) );
    }

    public static function attempt_to_flush_siteground_cache_manually( $post_id ){
		// Bail if the current hook is save_post and the post is scheduled.
		if ( 'save_post' === current_action() && 'future' === get_post_status( $post_id ) ) {
			return;
		}

		// Bail if the current hook is publish_post and the post isn't scheduled.
		if ( 'publish_post' === current_action() && 'future' !== get_post_status( $post_id ) ) {
			return;
		}

        /**
         * Purge everything;
         */
        return self::purge_cache_request( get_home_url( null, '/' ) );
    }

    public static function purge_cache_request( $url, $include_child_paths = true ) {
		// Bail if the url is empty.
		if ( empty( $url ) ) {
			return;
		}

		$hostname   = str_replace( 'www.', '', wp_parse_url( home_url(), PHP_URL_HOST ) );
		$parsed_url = wp_parse_url( $url );
		$main_path  = wp_parse_url( $url, PHP_URL_PATH );

		if ( empty( $main_path ) ) {
			$main_path = '/';
		}

		// Bail if the url has get params, but it matches the home url.
		// We don't want to purge the entire cache.
		if (
			isset( $parsed_url['query'] ) &&
			wp_parse_url( home_url( '/' ), PHP_URL_PATH ) === $main_path
		) {
			return;
		}

		// Change the regex if we have to delete the child paths.
		if ( true === $include_child_paths ) {
			$main_path .= '(.*)';
		}

		// Flush the cache for the URL.
		return self::flush_dynamic_cache( $hostname, $main_path, $url );
    }

    	/**
	 * Call site tools client to purge the cache.
	 *
	 * @param      string $hostname  The domain which cache will be flushed.
	 * @param      string $main_path The path to be flushed.
	 * @param      string $main_path The URL to be flushed.
	 *
	 * @return     bool              True if the cache is purged successfully, false otherwise.
	 */
	public static function flush_dynamic_cache( $hostname, $main_path, $url ) {
		// Build the request params.
		$args = array(
			'api'      => 'domain-all',
			'cmd'      => 'update',
			'settings' => array( 'json' => 1 ),
			'params'   => array(
				'flush_cache' => '1',
				'id'          => $hostname,
				'path'        => $main_path,
			),
		);

		$site_tools_result = self::call_site_tools_client( $args, true );

		// do_action( 'siteground_optimizer_flush_cache', $url );

		if ( false === $site_tools_result ) {
			return false;
		}

		if ( isset( $site_tools_result['err_code'] ) ) {
			BaseController::error_log( 'There was an issue purging the cache for this URL: ' . $url . '. Error code: ' . $site_tools_result['err_code'] . '. Message: ' . $site_tools_result['message'] . '.' );
			return false;
		}

		return true;
	}

    /**
     * Open Unix socket and run a specific command using the WP_Filesystem API.
     *
     * @since 7.2.7
     *
     * @param array $args        The command arguments (api, cmd, params, settings).
     * @param bool  $json_object Whether to force JSON objects in encoding.
     * @return array|false       Array of results on success, false on failure.
     */
    public static function call_site_tools_client( array $args, bool $json_object = false ): array|bool {
        // Load the WP_Filesystem API if not already loaded.
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        // Initialize the global filesystem object.
        WP_Filesystem();
        global $wp_filesystem;

        // Bail if socket file is missing or no arguments provided.
        if ( empty( $args ) || ! $wp_filesystem->exists( self::SITE_TOOLS_SOCK_FILE ) ) {
            return false;
        }

        // Assemble the request payload.
        $request = [
            'api'      => $args['api'] ?? '',
            'cmd'      => $args['cmd'] ?? '',
            'params'   => $args['params'] ?? [],
            'settings' => $args['settings'] ?? [],
        ];

        // Determine JSON encoding flags.
        $flags   = $json_object ? JSON_FORCE_OBJECT : 0;
        $payload = wp_json_encode( $request, $flags ) . "\n";

        // Write payload to the Unix socket file.
        $written = $wp_filesystem->put_contents(
            self::SITE_TOOLS_SOCK_FILE,
            $payload,
            FS_CHMOD_FILE
        );

        if ( false === $written ) {
            return false;
        }

        // Read the response back from the socket.
        $response = $wp_filesystem->get_contents( self::SITE_TOOLS_SOCK_FILE );

        if ( empty( $response ) ) {
            return false;
        }

        // Decode JSON response.
        $decoded = json_decode( trim( $response ), true );

        if ( null === $decoded || isset( $decoded['err_code'] ) ) {
            return false;
        }

        return $decoded;
    }
}
