<?php

namespace Jptgb\Resources;

/**
 * The Cache Controller Class
 *
 * @author Yaidier Perez
 * */

use Jptgb\Controllers\BaseController;
use Jptgb\Controllers\AjaxController;
use Jptgb\Controllers\CacheController;

class Utils extends BaseController {
    /**
     * Check if a given URL points to an image based on its file extension.
     *
     * @param string $url The URL to check.
     * @return bool Returns true if the URL points to an image, otherwise false.
     */
    static function wp_is_image_url(string $url): bool {
        $parsed_url = wp_parse_url($url);
        $path       = $parsed_url['path'] ?? '';
        $extension  = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        

        /** 
         * List of image extensions you want to allow.
         */
        $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg', 'ico'];

        return in_array($extension, $image_extensions);
    }

    /**
     * Recursively delete a directory and all of its contents.
     *
     * Tries to use the WP_Filesystem API first; if that fails or isn’t
     * available, falls back to a native PHP glob() + unlink()/rmdir() loop.
     *
     * @param string $path Absolute filesystem path to the directory.
     * @return bool True on success (or if directory didn’t exist), false on any failure.
     */
    public static function remove_directory_recursively( string $path ): bool {
        BaseController::$logs->register( 'Delete action - Removing directory: ' . $path , 'info' );

        // Bail early if path doesn’t exist.
        if ( ! is_dir( $path ) ) {
            return true;
        }

        // Attempt WP_Filesystem deletion.
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        global $wp_filesystem;

        if ( isset( $wp_filesystem ) && is_object( $wp_filesystem ) ) {
            /** 
             * delete( $path, $recursive ) 
             * @param string $path Path to file or directory.
             * @param bool   $recursive True to delete directories recursively.
             * @return bool True on success, false on error.
             */
            if ( $wp_filesystem->delete( $path, true ) ) {
                return true;
            }
        }
    }

    /**
     * Retrieve the current full URL, optionally decoding before sanitization.
     *
     * @since 1.0.0
     * @see   esc_url_raw()
     *
     * @param bool $decode Whether to rawurldecode the URL before escaping.
     * @return string      The sanitized current URL.
     */
    public static function get_current_url( bool $decode = true ): string {
        // Scheme: sanitize/whitelist or fallback to is_ssl().
        $raw_scheme = strtolower( sanitize_text_field( wp_unslash( $_SERVER['REQUEST_SCHEME'] ?? '' ) ) );

        if ( ! in_array( $raw_scheme, [ 'http', 'https' ], true ) ) {
            $raw_scheme = is_ssl() ? 'https' : 'http';
        }

        // Host and URI: unslash + sanitize.
        $raw_host = sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST']  ?? '' ) );
        $raw_uri  = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ?? '' ) );

        // Reconstruct.
        $current = "{$raw_scheme}://{$raw_host}{$raw_uri}";

        // 1) Optionally decode percent-encoded chars
        if ( $decode ) {
            $current = rawurldecode( $current );
        }

        // 2) Finally escape for a safe URL
        return (string) esc_url_raw( $current );
    }

    public static function get_js_content( $relative_path ) {
        $js_path = JPTGB_PATH . $relative_path;

        if( !file_exists( $js_path ) ) {
            return '';
        }
        
        $js_content = file_get_contents( $js_path );

        return $js_content;
    }

    /**
     * Removes a parameter from an URL.
     *
     * @param string $param The query parameter name to remove.
     * @param string $url   The URL from which to remove the parameter.
     * @return string       The URL with the specified parameter removed.
     */
    public static function remove_parameter_from_url( $param, $url ) {
        // Ensure $param is not an empty string and $url is a valid URL
        if ( ! empty( $param ) && filter_var( $url, FILTER_VALIDATE_URL ) ) {
            // Remove the parameter and return the modified URL
            return remove_query_arg( $param, $url );
        }

        // Return the original URL if the input is not valid
        return $url;
    }

    public static function file_get_contents( $path_or_url ) {
        /** 
         * Fetch the image content from the URL.
         */
        if( self::$is_production ){
            $file_content = file_get_contents( $path_or_url );

        /**
         * Else allow to get contents even 
         * if the ssl certificate is not valid for 
         * developing environments.
         */
        } else {
            $context_options = [
                'ssl' => [
                    'verify_peer'       => false,
                    'verify_peer_name'  => false,
                ],
            ];

            $context        = stream_context_create( $context_options );
            $file_content   = file_get_contents( $path_or_url, false, $context );
        }

        return $file_content;
    }

    /**
     * Retrieves the HTTP status code of a URL and the target URL if redirected.
     *
     * @param string $url The URL to check.
     * @return array An associative array containing the HTTP status code and the redirect target URL (if any).
     */
    public static function get_url_http_code( $url ) {
        if( strpos( $url, 'kick-top-streamers-ranking' ) !== false ) {
            $stop = 1;
        }
        // Add a query parameter to identify the request.
        // $url = esc_url_raw( $url . ( strpos( $url, '?' ) === false ? '?' : '&' ) . 'jptgb-redirect-check=1' );

        // Make a HEAD request to minimize data transfer.
        $response = wp_remote_head( $url, [
            'timeout'       => 10,
            'redirection'   => 0, // Prevent automatic following of redirects.
            'user-agent'    => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.3',
            'sslverify'     => self::$is_production, // Verify SSL only in production.
            'headers'       => [
                'X-Redirect-Check' => '1', // Custom header to identify the redirect check.
            ],
        ]);

        // Check if the request was successful.
        if ( is_wp_error( $response ) ) {
            return [
                'http_code' => 0,
                'redirect_url' => null,
            ];
        }

        // Retrieve the HTTP status code.
        $http_code = wp_remote_retrieve_response_code( $response );

        // Retrieve the redirect target URL if present.
        $headers        = wp_remote_retrieve_headers( $response );
        $redirect_url   = isset( $headers['location'] ) ? esc_url_raw( $headers['location'] ) : null;

        /**
         * Remove the identifier parameter.
         */
        // $redirect_url = str_replace( '?jptgb-redirect-check=1', '', $redirect_url );

        return [
            'http_code'     => $http_code,
            'redirect_url'  => $redirect_url,
        ];
    }

    /**
     * Validate URL structure using wp_parse_url().
     *
     * @param string $url The URL to validate.
     * @return bool True if the URL structure is valid, false otherwise.
     */
    public static function is_url_valid( $url ) {
        $parsed_url = wp_parse_url( $url );

        return !empty( $parsed_url['scheme'] ) && !empty( $parsed_url['host'] );
    }

    /**
     * Retrieves the cache status for a given WordPress post or post ID.
     *
     * This function supports both WP_Post objects and post IDs as input. It attempts
     * to fetch cache status information stored in the WordPress options table, specifically
     * looking for an option related to the post type of the given post. The function returns
     * an array containing the cache status information if found; otherwise, it returns an
     * empty array.
     *
     * @param WP_Post|int $wp_object The WP_Post object or post ID for which to retrieve the cache status.
     * @return array The cache status information as an array, or an empty array if not found or unsupported input is provided.
     */
    public static function get_cache_data( $wp_object ) {
        /**
         * Parse integer if $wp_object is numeric.
         */
        $post_id = is_numeric( $wp_object ) ? (int) $wp_object : false;

        /**
         * Check if the provided argument is a WP_Post object. If so, extract the post type
         * as the option suffix and use the post's ID. This allows the function to handle
         * cache status retrieval based on the type of post.
         */
        if ( is_a( $wp_object, 'WP_Post' ) || $post_id ) {
            /**
             * Get the post id.
             */
            if( !$post_id ){
                $post_id = $wp_object->ID;
            }

            /**
             * Get current status.
             */
            $cache_status = get_post_meta( $post_id, 'jptgb_cache_status', true );
        } else {
            /**
             * If the provided argument is neither a WP_Post object nor an integer (post ID),
             * the function currently does not handle such cases and simply returns.
             * This limitation is explicitly noted here for future development considerations.
             */
            return;
        }

        if( !$cache_status ){
            $cache_status = [
                'status'    => 'no-cache',
                'message'   => 'No cache',
                'pecentage' => 0,
            ];
        }

        return $cache_status;
    }

        // $site_domain = Utils::get_site_doamin();

    /**
     * Retrieve the permalink for an object, and if it still points at a local host
     * (127.0.0.1 or localhost), swap in the real site domain — preserving the original scheme.
     *
     * @param int    $object_id   Post (or other object) ID.
     * @return string Fully qualified URL for the object.
     */
    public static function get_permalink( $object_id ): string {
        /**
         * 1) Get whatever WP is generating (may be http(s)://127.0.0.1/…).
         */
        $requested_url = get_permalink( $object_id );

        /**
         * 2) If it contains a local host, we’ll replace only the scheme+host.
         * 
         * Note: Some hosting might serve 127.0.0.1 or localhost if the site is being served
         * behind e.g a load balancer or a proxy and HTTP_X_FORWARDED_HOST is not properly set 
         * in wp-config.php.
         */
        if ( false !== strpos( $requested_url, '127.0.0.1' )
        || false !== strpos( $requested_url, 'localhost' ) ) {

            /**
             * Get the real domain (no scheme, no path).
             */
            $site_domain = Utils::get_site_doamin(); 

            /**
             * Pull the original scheme (http or https) from the URL.
             */
            $scheme = wp_parse_url( $requested_url, PHP_URL_SCHEME );

            /**
             * Build new prefix: e.g. "https://wrtrading.com".
             */
            $new_prefix = $scheme . '://' . untrailingslashit( $site_domain );

            /**
             * Swap out only the original scheme+host portion.
             */
            $requested_url = preg_replace(
                '#^https?://[^/]+#i',
                $new_prefix,
                $requested_url
            ) ?: $requested_url;
        }

        return $requested_url;
    }

    /**
     * Deletes all metadata entries with a specific key for all posts, pages, and custom post types.
     *
     * @global wpdb $wpdb WordPress database abstraction object.
     * @param string $meta_key The meta key to search for and delete.
     * @return void
     */
    public static function delete_all_post_meta_by_key( $meta_key ) {
        global $wpdb;

        /** Ensure that the meta key is not empty to avoid accidental deletion of all post meta. */
        if ( empty( $meta_key ) ) {
            return;
        }

        /** Execute the query. */
        $result = $wpdb->query( $wpdb->prepare( "DELETE FROM $wpdb->postmeta WHERE meta_key = %s", $meta_key ) );

        /** Optionally, you can log the result to check how many rows were affected. */
        if ( $result !== false ) {
            // Log or handle the successful deletion case.
            self::error_log( "Successfully deleted all post meta for key: $meta_key" );
        } else {
            // Log or handle the error case.
            self::error_log( "Error deleting post meta for key: $meta_key" );
        }
    }

    /**
     * Check if a file exists in the WordPress uploads directory with a fallback to native PHP functions.
     *
     * @param string $file_path The path of the file to check.
     * @return bool Returns true if the file exists, false otherwise.
     */
    static function does_file_exists( $file_path, bool $do_decode_path = true ) {
        /**
         * Include the WordPress file API and initialize the WordPress filesystem.
         */
        if ( !function_exists( 'WP_Filesystem' ) ) {
            require_once( ABSPATH . 'wp-admin/includes/file.php' );
        }

        WP_Filesystem();
        global $wp_filesystem;

        /** 
         * Decode the file path to handle special characters
         */
        if( $do_decode_path ){
            $file_path = urldecode( $file_path );
        }

        /**
         * Attempt to use the WordPress Filesystem API to check if the file exists.
         */
        if ( !empty( $wp_filesystem ) ) {
            if ( $wp_filesystem->exists( $file_path ) ) {
                return true;
            }
        }

        /**
         * Return false if both methods fail.
         */
        return false;
    }

    /**
     * Check if a file in the WordPress uploads directory contains a given phrase, with fallback to native PHP functions.
     *
     * @param string $file_path The path of the file to check.
     * @param string $search_phrase The phrase to search for inside the file.
     * @param bool   $do_decode_path Whether to urldecode the path before processing.
     * @return bool Returns true if the phrase is found in the file, false otherwise.
     */
    static function does_file_contain_phrase( $file_path, $search_phrase, $do_decode_path = true ) {
        /**
         * Include the WordPress file API and initialize the WordPress filesystem.
         */
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once( ABSPATH . 'wp-admin/includes/file.php' );
        }

        WP_Filesystem();
        global $wp_filesystem;

        /** 
         * Decode the file path to handle special characters.
         */
        if ( $do_decode_path ) {
            $file_path = urldecode( $file_path );
        }

        /**
         * Attempt to use the WordPress Filesystem API to read the file.
         */
        if ( ! empty( $wp_filesystem ) ) {
            $file_content = $wp_filesystem->get_contents( $file_path );

            if ( false !== $file_content && strpos( $file_content, $search_phrase ) !== false ) {
                return true;
            }
        }

        /**
         * Fallback to using native PHP functions if WP Filesystem is not available.
         */
        if ( is_readable( $file_path ) ) {
            $file_content = file_get_contents( $file_path );

            if ( false !== $file_content && strpos( $file_content, $search_phrase ) !== false ) {
                return true;
            }
        }

        /**
         * Return false if both methods fail or phrase not found.
         */
        return false;
    }

    /**
     * Retrieves the post ID from a provided WP_Post object.
     *
     * This function checks if the given variable is an instance of WP_Post and returns the post ID.
     * If the variable is not a valid WP_Post object, it returns false.
     *
     * @param mixed $wp_object The WP_Post object from which to retrieve the post ID.
     * @return int|false The post ID if valid, false otherwise.
     */
    public static function get_post_id_from_object( $wp_object ) {
        /**
         * Check if the provided variable is an instance of WP_Post.
         */
        if( !is_a( $wp_object, 'WP_Post' ) ) {
            /**
             * Return false if $wp_object is not a WP_Post object
             */
            return false;
        }
        
        /**
         * Return the post ID.
         */
        return $wp_object->ID;
    }

    /**
     * Check if the provided URL contains any query parameters.
     *
     * @param string $url The URL to check for query parameters.
     * @return bool Returns true if parameters are found, false otherwise.
     */
    public static function has_url_parameters(string $url): bool {
        /**
         * Parse the URL and return the query component.
         */
        $parsed_url = wp_parse_url($url);

        /**
         * Check if the 'query' component is set and is not empty.
         */
        return isset($parsed_url['query']) && !empty($parsed_url['query']);
    }

    /**
     * Create a directory (and all parents) using the WP_Filesystem API.
     *
     * @since 1.0.0
     *
     * @param string $path        Absolute path to the directory.
     * @param int    $permissions Octal file permissions (e.g. 0755).
     * @return bool               True on success, false on failure.
     */
    public static function create_directory_recursively( string $path, int $permissions ): bool {
        // Initialize the WP_Filesystem API if needed.
        if ( ! isset( $GLOBALS['wp_filesystem'] ) || ! is_object( $GLOBALS['wp_filesystem'] ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }

        /** @var WP_Filesystem_Base $wp_filesystem */
        global $wp_filesystem;

        // If it already exists, nothing to do.
        if ( $wp_filesystem->is_dir( $path ) ) {
            return true;
        }

        // Recursively ensure parent exists.
        $parent = dirname( $path );
        if ( ! $wp_filesystem->is_dir( $parent ) ) {
            if ( ! self::create_directory_recursively( $parent, $permissions ) ) {
                return false;
            }
        }

        // Finally, create the target directory.
        return (bool) $wp_filesystem->mkdir( $path, $permissions );
    }

    /**
     * Converts a comma-separated string of CSS selectors into an array.
     *
     * @param string $selectors The string containing CSS selectors separated by commas.
     * @return array An array of CSS selectors.
     */
    public static function convert_selectors_to_array( $selectors ) {
        if( !$selectors ){
            return [];
        }

        // Trim the entire string to remove whitespace from the beginning and end,
        // then explode it into an array using comma as the delimiter,
        // and finally apply array_map with trim to each element to remove any extra spaces.
        return array_map('trim', explode(',', $selectors));
    }

    public static function write_file( $file_path, $content ) {
        try {
            /**
             * Attempt to use the WordPress Filesystem API.
             */
            if (!function_exists('WP_Filesystem')) {
                include_once(ABSPATH . 'wp-admin/includes/file.php');
            }
            
            /**
             * Initialize the WordPress filesystem, no more using 'file-put-contents' directly.
             */
            if (function_exists('WP_Filesystem')) {
                WP_Filesystem();
            }
        
            global $wp_filesystem;
    
            /**
             * Extract the directory path from the full file path.
             */
            $directory = dirname( $file_path );
        
            /**
             * Check if the WordPress filesystem API is available and initialized.
             */
            if ( !empty( $wp_filesystem ) ) {
                /**
                 * Check if the directory exists, if not, try to create it.
                 */
                if (!$wp_filesystem->is_dir($directory)) {
                    /**
                     * Recursive directory creation with appropriate permissions.
                     */
                    $output = self::create_directory_recursively( $directory, FS_CHMOD_DIR );
                    
                    if( !$output ) {
                        return false; // Exit if the directory cannot be created
                    }
                }
    
                /**
                 * Successfully written using WP Filesystem.
                 */
                $output = $wp_filesystem->put_contents( $file_path, $content, FS_CHMOD_FILE );
    
                if( $output ){
                    return true;
                }
            }
        
            /**
             * Return false if both methods fail.
             */
            return false;

        } catch (\Throwable $th) {
            error_log( sprintf( 'WebSpeed Error: %s', $th->getMessage() ) );
            return false;
        }
    }

    /**
     * Fetches content from an external URL using the WordPress HTTP API.
     *
     * @param string $url The URL to fetch content from.
     * @return mixed The content on success, or false on failure along with an error message.
     */
    public static function read_file_from_url( $url ) {
        /**
         * Get the files contents.
         */
        $response = wp_remote_get( $url, [
            'sslverify' => self::$is_production,
        ] );
        
        if ( is_wp_error( $response ) ) {
            // Log the error for debugging
            self::error_log( 'Error fetching the URL: ' . sanitize_text_field($url) . ' Error: ' . $response->get_error_message() );
            return false;
        }
        
        $body = wp_remote_retrieve_body( $response );
        return $body ? $body : false;
    }

    /**
     * Reads the contents of a file using the WordPress Filesystem API with a fallback to file_get_contents.
     * 
     * @param string $file_path The path to the file.
     * @return mixed The file contents on success, or false on failure.
     */
    public static function read_file( $file_path ) {
        /**
         * Attempt to use the WordPress Filesystem API.
         */
        if (!function_exists('WP_Filesystem')) {
            include_once(ABSPATH . 'wp-admin/includes/file.php');
        }
        
        /**
         * Initialize the WordPress filesystem, no more using 'file-get-contents' directly.
         */
        if (function_exists('WP_Filesystem')) {
            WP_Filesystem();
        }

        global $wp_filesystem;

        /**
         * Check if the WordPress filesystem API is available and initialized.
         */
        if (!empty($wp_filesystem)) {
            /**
             * Successfully read using WP Filesystem.
             */
            if ($wp_filesystem->exists($file_path)) {
                return $wp_filesystem->get_contents($file_path);
            }
        } else {
            /**
             * Fallback to using file_get_contents if WP Filesystem is not available.
             */
            if (function_exists('file_get_contents') && file_exists($file_path)) {
                BaseController::$logs->register( '$wp_filesystem is not available using file_get_contents() to read the contents of ' . sanitize_text_field( $file_path ) , 'warning' );
                return file_get_contents( $file_path );
            }
        }

        /**
         * Return false if both methods fail.
         */
        return false;
    }

    /**
     * Delete a file, preferring WP_Filesystem but falling back to unlink().
     *
     * @param string $file Absolute path to the file you want to delete.
     * @return bool True if the file was deleted (or didn't exist), false on failure.
     */
    public static function delete_file( string $file ): bool {
        // If it doesn’t exist or isn’t a file, nothing to do.
        if ( ! is_file( $file ) ) {
            return true;
        }

        // 1) Try WP_Filesystem
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        global $wp_filesystem;
        if ( $wp_filesystem instanceof \WP_Filesystem_Base ) {
            // second param $recursive = false, third param $mode is ignored for delete()
            $deleted = $wp_filesystem->delete( $file, false );
            if ( $deleted ) {
                return true;
            }
        }

        return false;
    }

    // helper to glob + delete matching directories
    public static function delete_prefixed_dirs( $base, $prefix ) {
        foreach ( glob( $base . $prefix . '*' ) as $dir ) {
            if ( is_dir( $dir ) ) {
                Utils::remove_directory_recursively( $dir );
            }
        }
    }

    /**
     * Converts the content of a textarea into an array where each line is an item.
     *
     * @param string $textarea_content The content of the textarea.
     * @return array An array where each line is an item.
     */
    public static function textarea_content_to_array__breaklines( $textarea_content ) {
        /**
         * Normalize line endings to handle different OS line endings.
         */
        $normalized_content = str_replace( "\r\n", "\n", $textarea_content );
        $normalized_content = str_replace( "\r", "\n", $normalized_content );

        /**
         * Split the content into an array where each line is an item.
         */
        $lines_array = explode( "\n", $normalized_content );

        /**
         * Optionally, remove empty lines.
         */
        $lines_array = array_filter( $lines_array, function( $line ) {
            return trim( $line ) !== '';
        });

        return $lines_array;
    }

    /**
     * Extracts the filename, filename without extension, and path segment (excluding filename) from a given URL.
     *
     * @param string $url The URL from which to extract the details.
     * @return array Associative array with 'filename', 'filename_no_ext', and 'path' (URL path up to but excluding the filename).
     */
    public static function extract_filename_and_path_from_url( string $url, bool $do_decode = true ): array {
        /**
         * Parse the URL and return its components.
         */
        $parsed_url = wp_parse_url($url);

        /**
         * Extract the path excluding the filename using dirname.
         */
        $path = isset($parsed_url['path']) ? dirname($parsed_url['path']) : '';

        /**
         * Construct the full path URL segment without the filename.
         */
        $path_url_segment = $parsed_url['scheme'] . '://' . $parsed_url['host'] . (isset($parsed_url['port']) ? ':' . $parsed_url['port'] : '') . ($path !== '.' ? $path : '') . '/';

        /**
         * Get the filename without extension using pathinfo.
         */
        $file_info = pathinfo($parsed_url['path']);
        $filename_no_ext = $file_info['filename'];

        /**
         * Get the basename, which is the filename with extension.
         */
        $filename = basename($parsed_url['path']);

        /**
         * Check if we have to decode the data.
         */
        if( $do_decode ){
            $filename           = urldecode( $filename );
            $filename_no_ext    = urldecode( $filename_no_ext );
            $path_url_segment   = urldecode( $path_url_segment );
        }

        /**
         * Return the filename, filename without extension, and path segment without filename as an associative array.
         */
        return [
            'filename'          => $filename,
            'filename_no_ext'   => $filename_no_ext,
            'path'              => $path_url_segment
        ];
    }

    public static function create_url_safe_hash( $input ) {
        /**
         * Generate raw MD5 hash.
         */
        $raw_hash = md5( $input, true );
    
        /**
         * Encode in Base64.
         */
        $base_64_hash = base64_encode( $raw_hash );
    
        /**
         * Make URL-safe by replacing '+' with '-', '/' with '_', and removing '='
         */
        $url_safe_hash = str_replace( ['+', '/', '='], ['-', '_', ''], $base_64_hash );
    
        return $url_safe_hash;
    }

    /**
     * Check if a URL is absolute or relative using wp_parse_url.
     *
     * @param string $url The URL to check.
     * @return bool True if the URL is absolute, false if it is relative.
     */
    public static function is_absolute_url($url) {
        /**
         * Parse the URL.
         */
        $parsed_url = wp_parse_url($url);

        /**
         * A URL is absolute if it has a scheme and a host, or if it starts with.
         */
        return (isset($parsed_url['scheme']) && isset($parsed_url['host'])) || (strpos($url, '//') === 0);
    }

    /**
     * Convert a relative URL to an absolute URL.
     *
     * @param string $relative_url The relative URL to convert.
     * @param string $base_url The base URL of the page where the relative URL was found.
     * @return string The absolute URL.
     */
    public static function convert_relative_url_to_absolute( $relative_url, $base_url ) {
        /**
         * Return the url if it is already absolute.
         */
        if( Utils::is_absolute_url( $relative_url ) ){
            return $relative_url;
        }

        /**
         * Parse base URL and relative URL.
         */
        $base       = wp_parse_url($base_url);
        $relative   = wp_parse_url($relative_url);

        /**
         * Ensure the base URL has a scheme and host.
         */
        if( !isset( $base[ 'scheme' ] ) || !isset( $base[ 'host' ] ) ) {
            return $relative_url;
        }

        /**
         * Build the absolute URL.
         */
        $absolute_url = $base['scheme'] . '://' . $base['host'];

        /**
         * Add port if present.
         */
        if( isset( $base[ 'port' ] ) ) {
            $absolute_url .= ':' . $base[ 'port' ];
        }

        /**
         * Handle different types of relative URLs.
         */
        if( isset( $relative[ 'path' ] ) ) {
            if( $relative[ 'path' ][0] === '/' ) {
                /**
                 * Relative URL is an absolute path.
                 */
                $absolute_url .= $relative['path'];
            } else {
                /**
                 * Relative URL is a relative path.
                 */
                $path           = dirname($base['path']) . '/' . $relative['path'];
                $absolute_url   .= '/' . ltrim($path, '/');
            }
        }

        /**
         * Add query and fragment if present.
         */
        if( isset( $relative[ 'query' ] ) ) {
            $absolute_url .= '?' . $relative[ 'query' ];
        }

        if( isset( $relative[ 'fragment' ] ) ) {
            $absolute_url .= '#' . $relative[ 'fragment' ];
        }

        return $absolute_url;
    }

    /**
     * Function to get the true root directory of the WordPress installation.
     *
     * @return string Absolute path to the true root directory.
     */
    public static function get_true_wp_root_directory() {
        /**
         * Return the path if is already defined.
         */
        if( self::$wp_root_path ){
            return self::$wp_root_path;
        }

        /**
         * Start from the directory of this script.
         */
        $dir = ABSPATH;
        
        /**
         * Loop up through the directories.
         */
        while (true) {
            /**
             * Check if 'wp-config.php' is in the current directory.
             */
            if (file_exists($dir . '/wp-config.php')) {
                self::$wp_root_path = $dir;
                return self::$wp_root_path;
            }
            
            /**
             * Move one level up.
             */
            $parent_dir = dirname($dir);
            
            /**
             * If we have reached the root directory, stop.
             */
            if ($parent_dir === $dir) {
                /**
                 * If we didn't find 'wp-config.php', return an empty string or handle it accordingly.
                 */
                return '"wp-config.php" not found';
            }
            
            $dir = $parent_dir;
        }
    }

    /**
     * Return the site URL (protocol and domain).
     *
     * @param bool $cache Either to return the cache value or not.
     */
    public static function get_site_url( $cache = true ) {
        /**
         * Get cache data.
         */
        if( $cache ) {
            $wp_site_url = self::get_setting( 'site_url', false );

            if( $wp_site_url ) {
                return $wp_site_url;
            }
        }

        /**
         * Get the site url
         */
        $wp_site_url = site_url();

        /**
         * Parse the URL.
         */
        $parsed_url = wp_parse_url( $wp_site_url );

        /**
         * Check if the URL is valid and has a scheme and host component.
         */
        if (!isset($parsed_url['scheme']) || !isset($parsed_url['host'])) {
            return '';
        }

        /**
         * Extract the scheme and host part.
         */
        $wp_site_url = $parsed_url['scheme'] . '://' . $parsed_url['host'];

        /**
         * Add the port if present.
         */
        if (isset($parsed_url['port'])) {
            $wp_site_url .= ':' . $parsed_url['port'];
        }

        /**
         * Update the base variable.
         */
        self::update_setting( 'site_url', $wp_site_url );

        return $wp_site_url;
    }

    public static function get_site_doamin( $cache = true ) {
        /**
         * Check if the site domain is already set. 
         */
        /**
         * Get cache data.
         */
        if( $cache ) {
            $wp_site_domain = self::get_setting( 'site_domain', false );

            if( $wp_site_domain ) {
                return $wp_site_domain;
            }
        }

        $wp_site_domain = self::remove_protocol_segment_from_url( self::get_site_url() );

        self::update_setting( 'site_domain', $wp_site_domain );

        return $wp_site_domain;
    }

    public static function remove_protocol_segment_from_url( $url ) {
        return preg_replace( '/^https?:\/\//', '', $url );
    }

    /**
     * Converts a URL of a file hosted in the WordPress site to an absolute path.
     *
     * @param string $url The URL of the file.
     * @return string|false The absolute path of the file, or false if the URL is invalid.
     */
    public static function convert_url_to_absolute_path( $url, bool $do_decode = true ) {
        /**
         * Remove parameters from the url if they exist.
         */
        $url = Utils::remove_query_parameters( $url );

        /**
         * Get the site url.
         */
        $site_url = Utils::get_site_url();

        /**
         * Remove the protocol section from both.
         */
        $url        = self::remove_protocol_segment_from_url( $url );
        $site_url   = self::remove_protocol_segment_from_url( $site_url );

        /**
         * Remove the current site url part from current url.
         */
        $url = str_replace( $site_url, '', $url );

        /**
         * Get the wp directory path.
         */
        $wp_directory_path = Utils::get_true_wp_root_directory();

        /**
         * Build the absolute path.
         */
        $basolute_path = trailingslashit( $wp_directory_path ) . ltrim ( $url, '/' );

        /**
         * Check if we have to decode the path.
         */
        if( $do_decode ){
            $basolute_path = urldecode( $basolute_path );
        }

        return $basolute_path;
    }

    /**
     * Converts an absolute path of a file in the WordPress site to a URL.
     *
     * @param string $path The absolute path of the file.
     * @return string|false The URL of the file, or false if the path is invalid.
     */
    public static function convert_absolute_path_to_url( $path ) {
        /**
         * Ensure the path is within the WordPress directory.
         */
        $wp_root_path = Utils::get_true_wp_root_directory();

        /**
         * Remove the wordpress root path segment from the path.
         */
        $relative_path = str_replace( $wp_root_path, '', $path );

        /**
         * Get the site url.
         */
        $site_url = Utils::get_site_url();

        /**
         * Build the url.
         */
        $url = Utils::join_url_segments( $site_url, $relative_path );

        return $url;
    }

    /**
     * Join two segements of url.
     */
    public static function join_url_segments( $first_segment, $second_segment ){
        return trailingslashit( $first_segment ) . ltrim( $second_segment, '/' );
    }

    /**
     * Check if a given abslute url is local.
     */
    public static function is_url_local( $url ){
        /**
         * Return true if the url is relative.
         */
        if( !Utils::is_absolute_url( $url ) ){
            return true;
        }

        /**
         * Get the site url.
         */
        $site_url = Utils::get_site_url();

        /**
         * Parse the urls.
         */
        $parsed_url         = wp_parse_url( $url );
        $parsed_site_url    = wp_parse_url( $site_url );

        /**
         * Get both hosts.
         */
        $url_host       = $parsed_url['host'] ?? '';
        $site_url_host  = $parsed_site_url['host'] ?? '';

        if( $url_host ===  $site_url_host ){
            return true;
        }

        return false;
    }

    /**
     * Extract the filename with extension from a given URL or file path.
     *
     * @param string $url_or_path The URL or file path.
     * @return string The filename with its extension, or an empty string if not applicable.
     */
    public static function get_filename_with_extension($url_or_path) {
        // Parse the URL to get the path
        $parsed_url = wp_parse_url($url_or_path);
        
        // If the path component exists, use it; otherwise, use the original input
        $path = isset($parsed_url['path']) ? $parsed_url['path'] : $url_or_path;

        // Use pathinfo to get the details of the path
        $path_info = pathinfo($path);

        // Combine the filename and extension
        $filename = $path_info['filename'];
        $extension = isset($path_info['extension']) ? $path_info['extension'] : '';

        // If there is an extension, add it to the filename
        if ($extension !== '') {
            return $filename . '.' . $extension;
        } else {
            return $filename;
        }
    }

    /**
     * Remove all query parameters from a given URL.
     *
     * @param string $url The original URL.
     * @return string The URL without query parameters.
     */
    public static function remove_query_parameters( $url ) {
        // Parse the URL into its components
        $parsed_url = wp_parse_url($url);

        // Reconstruct the URL without the query part
        $clean_url = $parsed_url['scheme'] . '://' . $parsed_url['host'];

        // Add port if present
        if (isset($parsed_url['port'])) {
            $clean_url .= ':' . $parsed_url['port'];
        }

        // Add the path if present
        if (isset($parsed_url['path'])) {
            $clean_url .= $parsed_url['path'];
        }

        // Add the fragment if present
        if (isset($parsed_url['fragment'])) {
            $clean_url .= '#' . $parsed_url['fragment'];
        }

        return $clean_url;
    }

    /**
     * Extracts content enclosed between specific start and end comment tags for CSS data.
     *
     * @param string $html The full HTML document as a string.
     * @return array The extracted contents wrapped between specified comment tags.
     */
    public static function extract_wrapped_content(string $htmlContent, string $marker, bool $do_json_decode = true): array {
        $startMarker    = "{$marker}-START-";
        $endMarker      = "-{$marker}-END";
        
        $startIndex = strpos($htmlContent, $startMarker);
        $endIndex   = strpos($htmlContent, $endMarker);
    
        if ($startIndex === false || $endIndex === false) {
            // Early exit if either start or end marker is not found
            return [];
        }
    
        // Adjust startIndex to get content just after the start marker
        $cssDataStart   = $startIndex + strlen($startMarker);
        $cssTagsData    = substr($htmlContent, $cssDataStart, $endIndex - $cssDataStart);
        $cssTagsData    = trim($cssTagsData);
    
        if( $do_json_decode ){
            $cssTagsData = $cssTagsData ? json_decode( $cssTagsData, true ) : [];
        }
    
        return $cssTagsData;
    }

    /**
     * Removes content enclosed between specific start and end comment tags from the HTML content.
     *
     * @param string $htmlContent The full HTML document as a string.
     * @param string $marker The marker used to identify the start and end tags.
     * @return string The HTML content with the enclosed section and markers removed.
     */
    public static function remove_wrapped_content(string $htmlContent, string $marker): string {
        $startMarker    = "{$marker}-START-";
        $endMarker      = "-{$marker}-END";

        $startIndex = strpos($htmlContent, $startMarker);
        $endIndex   = strpos($htmlContent, $endMarker);

        if ($startIndex === false || $endIndex === false) {
            // Early exit if either start or end marker is not found
            return $htmlContent;
        }

        // Adjust endIndex to include the length of the end marker
        $endIndex += strlen($endMarker);

        // Remove the content between the markers, including the markers themselves
        $contentBefore = substr($htmlContent, 0, $startIndex);
        $contentAfter  = substr($htmlContent, $endIndex);

        return $contentBefore . $contentAfter;
    }

    public static function get_post_ids_to_exclude( $exlusion_list_id ) {
        if( isset( BaseController::$post_ids_to_exclude[ $exlusion_list_id ] ) ){
            return BaseController::$post_ids_to_exclude[ $exlusion_list_id ];
        }

        /**
         * Get the posts id list to be excluded.
         */
        $post_ids_string = BaseController::get_setting( $exlusion_list_id );

        /**
         * Return empty array if post ids value is empty.
         */
        if ( !$post_ids_string ) {
            BaseController::$post_ids_to_exclude[ $exlusion_list_id ] = [];
            return BaseController::$post_ids_to_exclude[ $exlusion_list_id ];
        }

        /**
         * Normalize the ID list to remove spaces around commas and then explode into an array.
         */
        $normalized_ids         = str_replace( ' ', '', $post_ids_string );
        $post_ids_to_exclude    = explode( ',', $normalized_ids );

        /**
         * Update the static property.
         */
        BaseController::$post_ids_to_exclude[ $exlusion_list_id ] = $post_ids_to_exclude;

        return BaseController::$post_ids_to_exclude[ $exlusion_list_id ];
    }

    public static function is_post_set_to_exclude( $post_id, string $exclusion_list_id ){
        /**
         * Get the post id to exclude values.
         */
        $post_ids_to_exclude = Utils::get_post_ids_to_exclude( $exclusion_list_id );

        /**
         * Check if the current post ID is in the array of excluded IDs, considering it as a string.
         */
        if ( in_array( $post_id, $post_ids_to_exclude ) ) {            
            return true;
        }

        return false;
    }

    public static function get_posts_cache_data() {
        /**
         * Fetch all eligible post types.
         */
        $eligible_post_types = self::get_setting( 'eligible_post_types' );

        /**
         * Reorder post types so that 'page' and 'post' come first if they exist.
         */
        $ordered_post_types = [];

        if ( in_array( 'page', $eligible_post_types, true ) ) {
            $ordered_post_types[] = 'page';
        }

        if ( in_array( 'post', $eligible_post_types, true ) ) {
            $ordered_post_types[] = 'post';
        }

        /**
         * Add remaining post types that are not 'page' or 'post'.
         */
        $remaining_post_types = array_diff( $eligible_post_types, ['page', 'post'] );
        $ordered_post_types = array_merge( $ordered_post_types, $remaining_post_types );

        /**
         * Fetch posts in the desired order: pages, posts, then others.
         */
        $all_posts = [];

        foreach ( $ordered_post_types as $post_type ) {
            /**
             * Fetch all posts by type.
             */
            $posts = get_posts(
                array(
                    'post_type'      => $post_type,
                    'post_status'    => 'publish',
                    'posts_per_page' => -1,
                    'orderby'        => 'title',
                    'order'          => 'ASC',
                )
            );

            $all_posts = array_merge( $all_posts, $posts );
        }
    
        /**
         * Prepare arrays for cache-ready and pending posts.
         */
        $ready_posts            = [];
        $pending_posts          = [];
        $pending_post_ids       = [];
        $total_tasks            = [];
        $ready_tasks            = [];
    
        /**
         * Loop through all posts, check if they're cache-ready,
         * and store them in $ready_posts if so.
         */
        foreach( $all_posts as $post ) {
            $cache_status = AjaxController::get_cache_status( $post );

            $total_tasks[$post->ID] = [];
    
            if ( empty( $cache_status['status'] ) || $cache_status['status'] === 'no-cache' || $cache_status['status'] === 'working-on' ) {
                $pending_posts[]    = $post;
                $pending_post_ids[] = $post->ID;
                
            } else {
                $ready_posts[] = $post;
                $ready_tasks[$post->ID] = [];
            }
        }

        $tasks_overview = [
            'total_tasks'   => $total_tasks,
            'ready_tasks'   => $ready_tasks,
            'percentage'    => number_format( ( count( $ready_tasks ) / count( $total_tasks ) ) * 100, 1 )
        ];

        update_option( 'jptgb_tasks_overview', $tasks_overview, false );

        return [
            'post_type_slugs'   => $ordered_post_types,
            'ready_posts'       => $ready_posts,
            'pending_posts'     => $pending_posts,
            'pending_post_ids'  => $pending_post_ids,
            'all_posts'         => $all_posts,
            'tasks_overview'    => $tasks_overview,
        ];
    }

    /**
     * Updates the task overview by moving a task from pending to ready.
     *
     * @return void
     */
    public static function add_a_task_ready_to_tasks_overview( $object_id ) {
        $tasks_overview = get_option( 'jptgb_tasks_overview', [] );
        $all_tasks      = $tasks_overview['total_tasks'] ?? [];
        $all_tasks_count = count( $all_tasks );

        $ready_tasks = $tasks_overview['ready_tasks'] ?? [];

        $ready_tasks[ $object_id ] = [];
        $ready_tasks_count = count( $ready_tasks );

        // Prevent division by zero
        $percentage = ( $all_tasks_count > 0 ) ? number_format( ( $ready_tasks_count / $all_tasks_count ) * 100, 1 ) : 0;

        $tasks_overview = [
            'total_tasks'   => $all_tasks,
            'ready_tasks'   => $ready_tasks,
            'percentage'    => $percentage
        ];

        update_option( 'jptgb_tasks_overview', $tasks_overview, false );
    }

    public static function get_post_type_labels_and_names_excluding_attachments() : array {
        /**
         * Get all registered post types.
         */
        $args = array(
            'public'   => true,   // Get public post types
            'show_ui'  => true,   // Only show post types that have a UI in the admin
        );
    
        /**
         * Fetch post types as objects using get_post_types function.
         */
        $post_types = get_post_types( $args, 'objects' );
    
        /**
         * Prepare the array with 'label' => post_type_label, 'value' => post_type_name.
         */
        $post_type_data = [];
        foreach( $post_types as $post_type ) {
            if( $post_type->name !== 'attachment' ) { // Exclude 'attachment' post type
                $post_type_data[] = [
                    'label' => $post_type->label,
                    'value' => $post_type->name,
                ];
            }
        }
    
        return $post_type_data;
    }   

    public static function delete_all_plugins_data( $is_uninstall = false ) {
        /**
         * Clear all queued messages in RabbitMQ.
         */
        CacheController::remove_all_queued_messages_in_rabbitmq();

        /**
         * Delete the post status date from meta values.
         */
        Utils::delete_all_post_meta_by_key( 'jptgb_cache_status' );

        /**
         * Delete options.
         */
        delete_option( 'jptgb_consumer_status' );
        delete_option( 'jptgb_all_pages_consumer' );
        delete_option( 'jptgb_pending_tasks_count' );
        delete_option( 'jptgb_all_single_posts_status' );
        delete_option( 'jptgb_settings' );
        delete_option( 'jptgb_consumer_pool' );
        delete_option( 'jptgb_has_redirect_list' );
        delete_option( 'jptgb_critical_data' );
        delete_option( 'jptgb_cc_records' );
        delete_option( 'jptgb_tasks_overview' );

        /**
         * Clean the cache table.
         */
        self::clean_cache_table();

        if( $is_uninstall ){
            /**
             * @deprecated 
             */
            delete_option( 'jptgb_registered_site_url' );
            delete_option( 'jptgb_registered_site_domain' );

            /**
             * Remove the site url and domain.
             */
            self::update_setting( 'site_url', '' );
            self::update_setting( 'site_domain', '' );

            /**
             * Delete the cache data table.
             */
            self::drop_cache_table();
        }
    }

    /**
     * Empty the custom cache table without dropping it.
     *
     * @return void
     */
    public static function clean_cache_table(): void {
        global $wpdb;

        // Fastest way to clear all rows and reset AUTO_INCREMENT        
        $wpdb->query( "TRUNCATE TABLE ". esc_sql( $wpdb->prefix . 'webspeed_cache' ) );
    }

    /**
     * Drop the custom cache table on uninstall.
     *
     * @return void
     */
    private function drop_cache_table(): void {
        global $wpdb;

        $wpdb->query( "DROP TABLE IF EXISTS ". esc_sql( $wpdb->prefix . 'webspeed_cache' ) . ";" );
    }

    /**
     * Insert or replace a cache record for the given post.
     *
     * @param int    $post_id      The post ID for the cache entry.
     * @param string $cache_status A status label.
     * @param array  $data         Data payload as an associative array.
     * @return int                 Inserted/updated row ID; 0 on failure.
     */
    public static function update_cache_entry( int $post_id, string $cache_status, array $data ): int {
        global $wpdb;
        $table_name = $wpdb->prefix . 'webspeed_cache';

        $data_json = wp_json_encode( $data );
        if ( false === $data_json ) {
            return 0;
        }

        $replaced = $wpdb->replace( 
            $table_name, 
            [
                'post_id'      => $post_id,
                'cache_status' => $cache_status,
                'data'         => $data_json,
            ],
            [
                '%d',
                '%s',
                '%s',
            ]
        );

        if ( false === $replaced ) {
            return 0;
        }

        // On success, $wpdb->replace returns number of rows affected (1 for insert, 2 for replace).
        return (int) $wpdb->insert_id;
    }

    /**
     * Remove all cache records for a given post ID.
     *
     * @param int $post_id The post ID whose cache entries should be removed.
     * @return int         Number of rows deleted on success; 0 on failure or none found.
     */
    public static function remove_cache_by_post_id( int $post_id ): int {
        global $wpdb;
        $table_name = $wpdb->prefix . 'webspeed_cache';

        $deleted = $wpdb->delete( $table_name, [ 'post_id' => $post_id ], [ '%d' ] );

        if ( false === $deleted ) {
            return 0;
        }

        return (int) $deleted;
    }

    /**
     * Retrieve the full cache entry for a given post ID.
     *
     * @param int $post_id The post ID whose cache entry should be retrieved.
     * @return array|null  Associative array with keys:
     *                     - id (int)
     *                     - post_id (int)
     *                     - created_at (string, MySQL datetime)
     *                     - cache_status (string)
     *                     - data (array) decoded payload
     *                   Returns null if no entry is found or on error.
     */
    public static function get_cache_entry_by_post_id( int $post_id ): ?array {
        global $wpdb;

        // Fetch the entire row for the given post_id
        $row = $wpdb->get_row( $wpdb->prepare( "SELECT id, post_id, created_at, cache_status, data FROM ". esc_sql( $wpdb->prefix . 'webspeed_cache' ) . " WHERE post_id = %d LIMIT 1", $post_id ), ARRAY_A );

        // If nothing found or query failed, return null
        if ( null === $row ) {
            return null;
        }

        // Decode the JSON-encoded 'data' column
        $decoded = json_decode( wp_unslash( $row['data'] ), true );
        if ( JSON_ERROR_NONE !== json_last_error() || ! is_array( $decoded ) ) {
            $decoded = [];
        }
        $row['data'] = $decoded;

        // Cast numeric values to int
        $row['id']      = (int) $row['id'];
        $row['post_id'] = (int) $row['post_id'];

        return $row;
    }

    /**
     * Get post IDs for cache entries with status 'cache-ready' older than a given threshold.
     *
     * @param int $older_than_seconds Number of seconds; only entries older than this will be returned.
     * @return int[]                  Array of post IDs matching the criteria.
     */
    public static function get_cache_ready_post_ids( int $older_than_seconds ): array {
        global $wpdb;      

        $post_ids = $wpdb->get_col( $wpdb->prepare("SELECT post_id FROM ". esc_sql( $wpdb->prefix . 'webspeed_cache' ) . " WHERE cache_status = %s AND created_at <= DATE_SUB(NOW(), INTERVAL %d SECOND)", 'cache-ready', $older_than_seconds ) );

        // Cast to integers and return
        return array_map( 'intval', $post_ids );
    }

    /**
     * Determine if the Varnish server is up and running.
     *
     * CloudWays:
     * At server root level Varnish being disabled.
     * HTTP_X_VARNISH - does not exist or is NULL
     * HTTP_X_APPLICATION - contains varnishpass
     *
     * At Application level ( WP install ) - Varnish ON
     * At server level is ON
     * HTTP_X_VARNISH - has random numerical value
     * HTTP_X_APPLICATION - contains value different from varnishpass, usually application name.
     *
     * At Application level ( WP install ) - Varnish OFF
     * At server level is ON
     * HTTP_X_VARNISH - has random numerical value
     * HTTP_X_APPLICATION - contains value varnishpass
     *
     * @since 1.1.3
     */
    public static function is_varnish_layer_started() {
        $data = $_SERVER;

        if ( ! isset( $data['HTTP_X_VARNISH'] ) ) {
            return false;
        }

        if ( isset( $data['HTTP_X_VARNISH'] ) && isset( $data['HTTP_X_APPLICATION'] ) ) {

            if ( 'varnishpass' === trim( $data['HTTP_X_APPLICATION'] ) ) {
                return false;
            } elseif ( 'bypass' === trim( $data['HTTP_X_APPLICATION'] ) ) {
                return false;
            } elseif ( is_null( $data['HTTP_X_APPLICATION'] ) ) {
                return false;
            }
        }

        if ( ! isset( $data['HTTP_X_APPLICATION'] ) ) {
            return false;
        }

        return true;
    }

    /**
     * Checks if the varnish is active on website."
     * x-cache header is checked to verify varnish presence.
     *
     * @return bool
     */
    public static function check_custom_varnish() {

        $unique_string = time();

        $url_ping = trim( home_url() . '?jptgb_check_cache_available=' . $unique_string );

        if( BaseController::get_setting( 'debug_activate' ) ){
            add_filter( 'https_ssl_verify', '__return_false' );
        }

        $headers = wp_get_http_headers( $url_ping );

        if ( empty( $headers ) ) {
            return false;
        }

        $headers = array_change_key_case( $headers->getAll(), CASE_LOWER );

        if ( isset( $headers['x-cache'] ) ) {
            return true;
        }

        return false;
    }

    /**
     * Will return true/false if the cache headers exist.
     *
     * @return bool
     */
    public static function is_varnish_cache_started() {
        if ( isset( $_SERVER['HTTP_X_VARNISH'] ) && is_numeric( $_SERVER['HTTP_X_VARNISH'] ) ) {
            return true;
        }

        /**
         * Return false early if varnish is disabled by the user.
         */
        if ( isset( $data['HTTP_X_APPLICATION'] )
        && ( 'varnishpass' === trim( $data['HTTP_X_APPLICATION'] ) || 'bypass' === trim( $data['HTTP_X_APPLICATION'] ) )
        ) {
            return false;
        }

        $check_local_server = self::is_varnish_layer_started();
        if ( true === $check_local_server ) {
            return true;
        }

        $custom_varnish_active = get_transient( 'jptgb_custom_varnish_server_active' );

        if ( false === $custom_varnish_active ) {
            $custom_varnish_active = (int) self::check_custom_varnish();
            set_transient( 'jptgb_custom_varnish_server_active', $custom_varnish_active, 1 * HOUR_IN_SECONDS );
        }

        return (bool) $custom_varnish_active;
    }

    /**
     * Build an HTML attribute string from an associative array.
     *
     * @param array $attrs Key/value pairs of attributes (e.g. [ 'id' => 'myid', 'class' => 'foo bar' ]).
     * @return string      A string like ` id="myid" class="foo bar"` ready to append to an element.
     */
    public static function build_html_attributes( array $attrs ): string {
        $html = '';

        foreach ( $attrs as $key => $value ) {
            // Sanitize the attribute name (e.g. allow letters, numbers, underscores, dashes)
            $attr_name = sanitize_key( (string) $key );
            // Cast to string and skip empty values
            $attr_value = (string) $value;
            if ( '' === $attr_value ) {
                continue;
            }

            /** Escape both name and value for safe output */
            $html .= sprintf(
                ' %s="%s"',
                esc_attr( $attr_name ),
                esc_attr( $attr_value )
            );
        }

        return $html;
    }

    /**
     * Extract all @font-face rulesets from a CSS bundle,
     * grouping them by font-family name and indexing by weight-style.
     *
     * @param string $css_content The full CSS content to scan.
     * @return array<string, array{rulesets: array<string,string>}>
     *   [
     *     'Barlow' => [
     *       'rulesets' => [
     *         '400-normal' => '@font-face{…}',
     *         '400-italic' => '@font-face{…}',
     *       ],
     *     ],
     *     // …
     *   ]
     */
    public static function extract_font_faces( string $css_content ): array {
        $font_faces = [];

        // 1) Pull each @font-face { … } block
        if ( ! preg_match_all( '/@font-face\s*{.*?}/is', $css_content, $all_faces ) ) {
            return [];
        }

        foreach ( $all_faces[0] as $ruleset ) {
            // 2) Get the family
            if ( ! preg_match( '/font-family\s*:\s*([^;]+)\s*;/i', $ruleset, $m_fam ) ) {
                continue;
            }
            $family = trim( $m_fam[1], " \t\n\r\0\x0B\"'" );

            // 3) Get the weight(s), default to "normal"
            $raw_weights = 'normal';
            if ( preg_match( '/font-weight\s*:\s*([^;]+)\s*;/i', $ruleset, $m_wt ) ) {
                $raw_weights = trim( $m_wt[1] );
            }
            $weights = preg_split( '/\s+/', $raw_weights, -1, PREG_SPLIT_NO_EMPTY );

            // 4) Get the style, default to "normal"
            $style = 'normal';
            if ( preg_match( '/font-style\s*:\s*([^;]+)\s*;/i', $ruleset, $m_st ) ) {
                $style = trim( $m_st[1] );
            }

            // 5) Init container for this family
            if ( ! isset( $font_faces[ $family ] ) ) {
                $font_faces[ $family ] = [
                    'rulesets' => [],
                ];
            }

            // 6) Store under each weight-style key, e.g. "400-normal"
            foreach ( $weights as $weight ) {
                $key = $weight . '-' . $style;
                $font_faces[ $family ]['rulesets'][ $key ] = $ruleset;
            }
        }

        return $font_faces;
    }

    /**
     * Insert one @font-face block into a critical‐only CSS string,
     * preserving any leading @charset declaration.
     *
     * @param string $critical_css    The CSS generated by Penthouse (critical-only).
     * @param string $font_face_ruleset  A single @font-face{…} block, as a string.
     * @return string                 The combined CSS.
     */
    public static function insert_font_face_into_critical_css( string $critical_css, string $font_face_ruleset ): string {
        // Normalize line endings
        $critical_css = str_replace( ["\r\n", "\r"], "\n", $critical_css );
        
        // Look for a leading @charset; capture it and the rest separately
        if ( preg_match( '/^\s*(@charset\s+["\'][^"\']+["\'];)(.*)$/is', $critical_css, $m ) ) {
            $charset   = $m[1] . "\n";
            $remainder = ltrim( $m[2] );
        } else {
            // No charset found
            $charset   = '';
            $remainder = ltrim( $critical_css );
        }

        // Ensure our ruleset ends with a newline
        $font_face_ruleset = rtrim( $font_face_ruleset ) . "\n";

        // Build the final CSS: [charset] + [our font-face] + [the rest]
        return $charset
            . $font_face_ruleset
            . "\n"
            . $remainder;
    }


    /**
     * Extract all font-data annotations from an HTML string,
     * without using DOMDocument—just regex & JSON.
     *
     * Expects attributes like:
     *   data-jptgbfonts="{&quot;fontFamily&quot;:&quot;Barlow, sans-serif&quot;,
     *                     &quot;fontWeight&quot;:&quot;700&quot;,
     *                     &quot;fontStyle&quot;:&quot;normal&quot;}"
     *
     * @param string $html
     * @return array<string, array{styles: string[]}>
     *   [
     *     'Barlow' => [
     *       'styles' => ['700-normal','400-italic',…],
     *     ],
     *     'Inter' => [
     *       'styles' => ['300-normal',…],
     *     ],
     *     …
     *   ]
     */
    public static function extract_fonts_data_from_html( string $html ): array {
        $fonts = [];

        // 1) Find every data-jptgbfonts="…"
        if ( preg_match_all(
            '/\bdata-jptgbfonts="([^"]+)"/i',
            $html,
            $matches,
            PREG_SET_ORDER
        ) ) {
            foreach ( $matches as $m ) {
                $raw = $m[1];

                // 2) HTML-decode & JSON-decode
                $json = html_entity_decode( $raw, ENT_QUOTES | ENT_HTML5 );
                $data = json_decode( $json, true );
                if ( ! is_array( $data ) ) {
                    continue;
                }

                // 3) Pull out family, weight, style
                $family = trim( (string) ( $data['fontFamily'] ?? '' ) );
                // strip off fallbacks after comma
                if ( false !== strpos( $family, ',' ) ) {
                    $family = trim( explode( ',', $family, 2 )[0], "\"' \t\n\r\0\x0B" );
                }
                if ( $family === '' ) {
                    continue;
                }

                $weight = (string) ( $data['fontWeight'] ?? '400' );
                $style  = strtolower( (string) ( $data['fontStyle']  ?? 'normal' ) );
                $key    = $weight . '-' . $style;

                // 4) Group, dedupe
                if ( ! isset( $fonts[ $family ] ) ) {
                    $fonts[ $family ] = [ 'styles' => [] ];
                }
                if ( ! in_array( $key, $fonts[ $family ]['styles'], true ) ) {
                    $fonts[ $family ]['styles'][] = $key;
                }
            }
        }

        return $fonts;
    }

    /**
     * Delete all transients that start with 'jptgb_image_url_404_'.
     *
     * @return void
     */
    public static function delete_jptgb_image_url_404_transients(): void {
        global $wpdb;

        /** Define transient name prefix */
        $transient_prefix = 'jptgb_image_url_404_';

        /** Prepare the SQL LIKE pattern */
        $like_pattern = '_transient_' . $wpdb->esc_like( $transient_prefix ) . '%';

        /** Query all matching transient option names */
        $options = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE %s",
                $like_pattern
            )
        );

        /** Loop through each option and delete both value and timeout */
        foreach ( $options as $option_name ) {
            /** Extract the transient key */
            $transient_key = substr( $option_name, strlen( '_transient_' ) );

            /** Delete both the transient and its timeout */
            delete_transient( $transient_key );
        }
    }

    /**
     * Remove all occurrences of a specific HTML attribute (and its value) from an HTML string.
     *
     * @param string $html      HTML content to process.
     * @param string $attribute The attribute name to strip (e.g. 'data-cat').
     * @return string           Filtered HTML without the specified attribute.
     */
    public static function remove_html_attribute( string $html, string $attribute ): string {
        /** Escape the attribute name for regex safety */
        $attr = preg_quote( $attribute, '/' );

        /**
         * Build a regex that matches:
         *  - one or more whitespace chars (\s+)
         *  - the exact attribute name ($attr)
         *  - optional whitespace, “=”, and an attribute value
         *    in double-quotes, single-quotes, or unquoted
         *
         * By requiring the “=” we avoid stripping substrings like “data-caton”
         * when targeting “data-cat”.
         */
        $pattern = '/\s+' . $attr . '\s*=\s*(?:"[^"]*"|\'[^\']*\'|[^\s>]+)/i';

        /** Remove all matched attributes */
        return (string) preg_replace( $pattern, '', $html );
    }

    /**
     * Retrieve the tag name of the first HTML element found in a string.
     *
     * @param string $html_string HTML snippet to inspect.
     * @return string|null Lowercase tag name (e.g. 'div'), or null if no element found.
     */
    public static function get_tag_name_from_string( string $html_string ): ?string {
        // Look for an opening or closing tag and capture the tag name
        if ( preg_match( '/<\s*\/?\s*([A-Za-z0-9-]+)/', $html_string, $matches ) ) {
            return strtolower( $matches[1] );
        }

        return null;
    }

    

    /**
     * Decode a JSON string into an associative array safely.
     *
     * @param string $json JSON string.
     *
     * @return array Decoded array or empty array on error.
     */
    private static function decode_json_array( string $json ): array {
        if ( '' === trim( $json ) ) {
            return array();
        }

        $decoded = json_decode( $json, true );
        if ( JSON_ERROR_NONE !== json_last_error() || ! is_array( $decoded ) ) {
            return array();
        }

        return $decoded;
    }

    public static function esc_safe_html( $content ) {
        return wp_kses(
            $content,
            [
                'strong' => [],
                'b'      => [],
            ]
        );
    }
}