<?php

class JptgbAdvancedCache {
    /**
     * Define calss properties.
     */
    protected static $current_url   = false;
    protected static $request_uri;
    protected static $http_host;
    protected static $server_name;
    protected static $http_user_agent;

    public static function init() {
        /**
         * Return if our cosntant is not defined.
         */
        if( !defined( 'JPTGB_CACHE_DIR' ) ){
            return;
        }

        /**
         * Bail out of this cache layer on non‑front‑end requests.
         *
         * @return void
         */
        if ( self::jptgb_should_bypass_cache() ) {
            return; // skip cache logic entirely
        }

        self::set_current_url();
        self::serve_cache_file();
    }

    /**
     * Determine as early as possible if we should bypass the static‑cache
     * (i.e. skip serving or generating a cached file) based on the request.
     *
     * Checks for:
     *  - CLI mode (PHP_SAPI)
     *  - wp‑cron.php, xmlrpc.php, admin‑ajax.php, wp‑login.php access
     *  - Any /wp-admin or /wp-json/ URL
     *  - Logged‑in users via the raw HTTP_COOKIE header
     *
     * @return bool True to bypass cache; false to serve/cache.
     */
    public static function jptgb_should_bypass_cache(): bool {
        // 1) CLI (fast constant, no function call)
        if ( PHP_SAPI === 'cli' ) {
            return true;
        }

        // Grab once, avoid repeated $_SERVER lookups
        $script = self::get_server_variable( 'SCRIPT_NAME' );

        /**
         * Define global varaibles
         */
        self::$request_uri      = self::get_server_variable('REQUEST_URI');
        self::$http_host        = self::get_server_variable('HTTP_HOST');
        self::$server_name      = self::get_server_variable('SERVER_NAME');
        self::$http_user_agent  = self::get_server_variable('HTTP_USER_AGENT');

        // 1) Remove query string from URI
        $path = strstr( self::$request_uri, '?', true ) ?: self::$request_uri;

        // 2) Static‐asset extensions: bail on .css, .js, images, fonts, .map
        //    strrchr finds the last “.” and returns from there to end of string
        $ext = strtolower( strrchr( $path, '.' ) );
        if (
            $ext &&
            in_array( $ext, [
                '.css', '.js',
                '.jpg', '.jpeg', '.png', '.gif', '.webp', '.svg', '.ico',
                '.woff', '.woff2', '.ttf', '.eot',
                '.map',
            ], true )
        ) {
            return true;
        }

        // 2) Fast path script checks: wp‑cron, xmlrpc, ajax, login
        if (
            strpos( $script, 'wp-cron.php'   ) !== false ||
            strpos( $script, 'xmlrpc.php'    ) !== false ||
            strpos( $script, 'admin-ajax.php') !== false ||
            strpos( $script, 'wp-login.php'  ) !== false
        ) {
            return true;
        }

        // 3) URL‐based checks: admin pages or REST API
        if (
            strpos( $path, '/wp-admin' ) !== false ||
            strpos( $path, '/wp-json/' ) !== false
        ) {
            return true;
        }

        // 4) Logged‑in user? single strpos on raw cookie header
        if ( self::is_user_logged_in_via_cookie()   ) {
            return true;
        }

        return false;
    }

    /**
     * Safely retrieve and sanitize a value from the $_SERVER superglobal,
     * allowing for query-string characters.
     *
     * @param string $variable The $_SERVER key to fetch (e.g. 'SCRIPT_NAME', 'REQUEST_URI').
     * @return string          The unslashed, sanitized value, or an empty string if not set.
     */
    public static function get_server_variable( string $variable ): string {
        // If not set, bail early.
        $value = $_SERVER[ $variable ] ?? null;

        if ( null === $value ) {
            return '';
        }

        // 1) Uns lash any magic‐quote slashes.
        $value = stripslashes( $value );

        // 2) Sanitize: allow letters, numbers, dots, dashes, slashes,
        //    and query-string chars (?, =, &, %).
        $sanitized = preg_replace( '/[^\p{L}\p{N}\-\.\/\?\=\&\%]/u', '', $value );

        return $sanitized;
    }

    /** 
     * Check if a user is logged in by scanning for login cookies.
     */
    private static function is_user_logged_in_via_cookie() {
        /**
         * Define cookies to skip.
         */
        $cookie_keys = ['comment_author_', 'wordpress_logged_in_', 'wp-postpass_'];

        foreach ($_COOKIE as $key => $value) {
            foreach ($cookie_keys as $cookie_key) {
                if (strpos($key, $cookie_key) === 0) {
                    /**
                     * Found a relevant WordPress cookie, return true.
                     */
                    return true;
                }
            }
        }

        /**
         * No relevant cookies found.
         */
        return false;
    }

    /**
     * Get the current request path without query string, and—if any
     * parameters exist—append their sanitized names as the last segment.
     *
     * Example:
     *   /update-test/?param1=a&param2=b  →  /update-test/param1param2
     *   (if that gets >200 chars, it uses a 12‑char MD5 hash instead)
     *
     * @return string Sanitized request path (no trailing slash).
     */
    public static function jptgb_get_request_path(): string {
        // 1) Grab & sanitize the raw URI
        /** @var string $raw_uri */
        $raw_uri = filter_var( self::$request_uri, FILTER_SANITIZE_URL );

        // 2) Break into components
        $parts = parse_url( $raw_uri );
        $path  = $parts['path'] ?? '/';

        // 3) Normalize base path (no trailing slash)
        $base = rtrim( $path, '/' );

        // 4) If there’s a query, parse out name→value pairs
        if ( ! empty( $parts['query'] ) ) {
            parse_str( $parts['query'], $pairs );

            if ( ! empty( $pairs ) ) {
                $fragments = [];

                // 5) Sanitize each name and value, then concatenate
                foreach ( $pairs as $name => $value ) {
                    /** Sanitize parameter name (A–Z, a–z, 0–9, _ and - only) */
                    $clean_name  = preg_replace( '/[^A-Za-z0-9_-]/', '', (string) $name );
                    /** Sanitize parameter value similarly */
                    $clean_value = preg_replace( '/[^A-Za-z0-9_-]/', '', (string) $value );

                    if ( '' !== $clean_name ) {
                        $fragments[] = $clean_name . $clean_value;
                    }
                }

                if ( ! empty( $fragments ) ) {
                    $combined = implode( '', $fragments );

                    // 6) If that segment gets too long, fall back to a hash
                    $max_len = 200;
                    if ( mb_strlen( $combined ) > $max_len ) {
                        // Hash of the raw name=value pairs for consistency
                        $flat_pairs = [];
                        foreach ( $pairs as $n => $v ) {
                            $flat_pairs[] = "{$n}={$v}";
                        }
                        $combined = substr( md5( implode( ',', $flat_pairs ) ), 0, 12 );
                    }

                    // 7 Prepend the param with a prefix
                    $combined = 'jptgb-has-param-' . $combined;

                    // 8) Append slash‑terminated param segment
                    return $base . '/' . $combined . '/';
                }
            }
        }

        // No params → return base path only
        return $base;
    }
    
    /**
     * Build your cache file path.
     *
     * @param string $device  E.g. 'mobile' or 'desktop'.
     * @return string Full path to index.html in your cache tree.
     */
    public static function jptgb_get_cached_file_path( string $device ): string {
        $request_path = self::jptgb_get_request_path();  // e.g. "/foo/bar/param1param2"
        $dir          = rtrim( JPTGB_CACHE_DIR . $device . $request_path, '/' ) . '/';
        return $dir . 'index.html';
    }

    public static function serve_cache_file() {
        /**
         * Define if reques is from a mobile.
         */
        $device = self::is_mobile() ? '/mobile' : '/desktop';

        /**
         * Define cache file path.
         */
        $cached_file_path = self::jptgb_get_cached_file_path( $device );

        /**
         * Serve the cache file if it exists.
         */
        if( !file_exists( $cached_file_path ) ) {
            return;            
        }

        /**
         * Get the current time.
         */
        $current_time = gmdate( 'D, d M Y H:i:s', time() );

        /**
         * Content type (WP usually handles this automatically, but it's harmless to reset here).
         */
        header( 'Content-Type: text/html; charset=UTF-8' );
        
        /**
         * HTTP/1.1 no-cache directive.
         */
        header( 'Cache-Control: no-cache, must-revalidate, max-age=0' );
        
        /**
         * HTTP/1.0 fallback for older clients.
         */
        header( 'Pragma: no-cache' );
        
        /**
         * Expire immediately.
         */
        header( 'Expires: ' . $current_time . ' GMT' );
        
        /**
         * Custom plugin/cache header. The second argument (true) means “replace any existing header of the same name.”
         */
        header( 'X-WEBSPEED-CACHE: HIT ('. $current_time .')', true );

        /**
         * Output the file content.
         */
        readfile( $cached_file_path );

        /**
         * Stop the script execution.
         */
        exit;
    }
    
    private static function set_current_url() {
        /**
         * Check the scheme (HTTP/HTTPS) - Support for load balancers and reverse proxies.
         */
        $scheme = 'http';
        if( self::get_server_variable('HTTP_X_FORWARDED_PROTO') ) {
            $scheme = strtolower( self::get_server_variable('HTTP_X_FORWARDED_PROTO') ) === 'https' ? 'https' : 'http';

        } elseif ( 'on' === self::get_server_variable('HTTPS') ) {
            $scheme = 'https';

        }
    
        /**
         * Use HTTP_HOST if available, fallback to SERVER_NAME.
         */
        $host = self::$http_host ? self::$http_host : self::$server_name;
        $host = htmlspecialchars($host, ENT_QUOTES, 'UTF-8');
    
        /**
         * Port number logic - only add to URL if non-standard ports are used.
         */
        $port = '';
        $server_port = self::get_server_variable('SERVER_PORT');
        if( $server_port ) {
            $port = ('https' === $scheme && 443 !== $server_port) || ('http' === $scheme && 80 !== $server_port) ? ':' . $server_port : '';
        }
    
        /**
         * Construct the URL.
         */
        $request_uri = htmlspecialchars( self::$request_uri, ENT_QUOTES, 'UTF-8' );
        $current_url = $scheme . '://' . $host . $port . $request_uri;

        /**
         * Check if the string ends with a slash.
         */
        if( substr( $current_url, -1 ) !== '/' ) {
            $current_url .= '/';
        }
    
        /**
         * Set the current url.
         */
        self::$current_url = $current_url;
    }

    public static function is_mobile() {
        $mobile_user_agent = self::get_server_variable('HTTP_SEC_CH_UA_MOBILE');

        if($mobile_user_agent) {
            /**
             * This is the `Sec-CH-UA-Mobile` user agent client hint HTTP request header.
             */
            $is_mobile = ('?1' === $mobile_user_agent);

        } elseif( empty( self::$http_user_agent ) ) {
            $is_mobile = false;

        } elseif( strpos( self::$http_user_agent, 'Mobile' ) !== false
            || strpos( self::$http_user_agent, 'Android' )      !== false
            || strpos( self::$http_user_agent, 'Silk/' )        !== false
            || strpos( self::$http_user_agent, 'Kindle' )       !== false
            || strpos( self::$http_user_agent, 'BlackBerry' )   !== false
            || strpos( self::$http_user_agent, 'Opera Mini' )   !== false
            || strpos( self::$http_user_agent, 'Opera Mobi' )   !== false ) {
                $is_mobile = true;

        } else {
            $is_mobile = false;
        }
    
        return $is_mobile;
    }    
}

JptgbAdvancedCache::init();
