<?php

namespace Jptgb;

use Jptgb\Includes\Jptgb;
use Jptgb\Resources\Utils;
use Jptgb\Controllers\BaseController;
use Jptgb\Controllers\CacheController;
use Jptgb\Controllers\MainController;

/**
 * Currently plugin version.
 * Start at version 1.0.0 and use SemVer - https://semver.org
 * Rename this for your plugin and update it as you release new versions.
 */
define( 'JPTGB_VERSION', '1.8.52' );

/**
 * Create (or update) the custom cache table.
 *
 * Ensures `post_id` is UNIQUE so that upserts against that column
 * will replace existing entries instead of creating duplicates.
 *
 * @return void
 */
function webspeed_create_cache_table(): void {
    global $wpdb;

    $table_name      = $wpdb->prefix . 'webspeed_cache';
    $charset_collate = $wpdb->get_charset_collate();

    /** SQL to create table with a UNIQUE key on post_id */
    $sql = "
    CREATE TABLE {$table_name} (
        id            BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        post_id       BIGINT(20) UNSIGNED NOT NULL,
        created_at    DATETIME           NOT NULL DEFAULT CURRENT_TIMESTAMP,
        cache_status  VARCHAR(100)      NOT NULL,
        data          LONGTEXT          NOT NULL,
        PRIMARY KEY  (id),
        UNIQUE KEY   uq_post_id (post_id)
    ) {$charset_collate};
    ";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );
}

/**
 * Create (or update) the custom cache table.
 *
 * Ensures `post_id` is UNIQUE so that upserts against that column
 * will replace existing entries instead of creating duplicates.
 *
 * @return void
 */
function webspeed_create_consumer_table(): void {
    global $wpdb;

    $table_name      = $wpdb->prefix . 'webspeed_consumer';
    $charset_collate = $wpdb->get_charset_collate();

    /** SQL to create table with a UNIQUE key on post_id */
    $sql = "
    CREATE TABLE {$table_name} (
        id            BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        post_id       BIGINT(20) UNSIGNED NOT NULL,
        created_at    DATETIME           NOT NULL DEFAULT CURRENT_TIMESTAMP,
        status        VARCHAR(100)      NOT NULL,
        data          LONGTEXT          NOT NULL,
        PRIMARY KEY  (id),
        UNIQUE KEY   uq_post_id (post_id)
    ) {$charset_collate};
    ";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta( $sql );
}

/**
 * Function to run on plugin activation.
 */
function jptgb_plugin_activate_callback() {
    /**
     * Create our custom tables.
     */
    webspeed_create_cache_table();

    

    /**
     * Schedule the cron jobs on plugin activation.
     */
    if ( !wp_next_scheduled( 'jptgb_process_consumer_item' ) ) {
        wp_schedule_event( time(), 'every_minute', 'jptgb_process_consumer_item' );
    }

    if ( !wp_next_scheduled( 'jptgb_process_every_hour' ) ) {
        wp_schedule_event( time(), 'hourly', 'jptgb_process_every_hour' );
    }

    /**
     * Maybe autostart the cache.
     */
    if( BaseController::get_setting( 'autostart_cache' ) ){
        BaseController::update_setting( 'autostart_cache', false );
        BaseController::update_setting( 'cache_activate', true );
        CacheController::activate_deactivate_the_cache( true );
    }
}

/**
 * Clear the cron job on plugin deactivation.
 */
function jptgb_plugin_deactivate_callback() {
    /**
     * Clear the consumers hooks.
     */
    wp_clear_scheduled_hook( 'jptgb_process_consumer_item' );
    wp_clear_scheduled_hook( 'jptgb_process_every_hour' );

    /**
     * Remove the site url and domain.
     */
    BaseController::update_setting( 'site_url', '' );
    BaseController::update_setting( 'site_domain', '' );

    /**
     * @deprecated 
     */
    delete_option( 'jptgb_registered_site_url' );
    delete_option( 'jptgb_registered_site_domain' );

    /**
     * Define the cache autostart.
     */
    if( BaseController::get_setting( 'cache_activate' ) ){
        BaseController::update_setting( 'cache_activate', false );
        BaseController::update_setting( 'autostart_cache', true );
    }

    /**
     * Deactivate the cache.
     */
    CacheController::activate_deactivate_the_cache( false );

    /**
     * Flush hosting's cache.
     */
    MainController::flush_hostings_cache();
}

/**
 * Register the activation hook.
 */
register_activation_hook(
    __DIR__ . '/webspeed.php',
    __NAMESPACE__ . '\\jptgb_plugin_activate_callback'
);
register_deactivation_hook(
    __DIR__ . '/webspeed.php',
    __NAMESPACE__ . '\\jptgb_plugin_deactivate_callback'
);

/**
 * Add a "Settings" link to the plugin row in the Plugins section.
 *
 * @param array $links Existing plugin action links.
 * @return array Modified plugin action links.
 */
function jptgb_add_settings_link( $links ) {
    $settings_link = '<a href="' . esc_url( admin_url( 'admin.php?page=jptgb-settings' ) ) . '">' . __( 'Settings', 'webspeed' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
}
add_filter(
    'plugin_action_links_' . plugin_basename( __DIR__ . '/webspeed.php' ),
    __NAMESPACE__ . '\\jptgb_add_settings_link'
);

/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * then kicking off the plugin from this point in the file does
 * not affect the page life cycle.
 *
 * @since    1.0.0
 */
function run_just_press_the_green_button() {
    /**
     * Define constants.
     */
    define( 'JPTGB_PATH', plugin_dir_path( __FILE__ ) ); 
    define( 'JPTGB_URL', plugins_url( '/', __FILE__ ) );

    require_once __DIR__ . '/vendor/autoload.php';

    /**
     * The core plugin class that is used to define internationalization,
     * admin-specific hooks, and public-facing site hooks.
     */
	new Jptgb();
}

run_just_press_the_green_button();
