/**
 * Handles the Admin App.
 *
 * @package Jptgb
 * */

class JptgbAdmin {
	constructor() {
        if (document.readyState != "loading") {
            this.init();
        }
        else {
            document.addEventListener("DOMContentLoaded", () => {
                this.init();
            });
        }
	}

    init() {
        this.statusTasks = {};
        this.isCheckingStatus = false;

        this.adminPage();
        this.toolsColumn();
        this.checkStatusRecursevelyV3();
    }

    checkStatusRecursevelyV3() {
        /**
         * Get all the tool columns in the current page.
         */
        const toolColumns = document.querySelectorAll('.jptgb-tool_column');
        
        /**
         * If there is not .jptgb-tool_column then we do not need to check for the status.
         */
        if( toolColumns.length <= 0 ) {
            return;
        }

        toolColumns.forEach((toolSet) => {
            const activeOption = toolSet.querySelector('.jptgb-tool_column_option.jptgb_active');

            /**
             * Prepare the tool set option information.
             */
            const messageElement    = activeOption.querySelector('.jptgb-tooltip');
            const message           = messageElement ? messageElement.innerHTML : '';
            let percentage          = toolSet.querySelector('.jptgb-working_status_bar_fill').style.width;
            percentage              = percentage.replace('%', '');

            this.statusTasks[toolSet.dataset.id] = {
                message: message,
                percentage : percentage,
                status : toolSet.dataset.cachestatus,
            }
        });

        /**
         * Data to be sent to the server.
         */
        const data = {
            action: 'get_tasks_statuses',
            nonce: jpgbData.toolsColumnGetWorkingOnStatusNonce,
            tasks: JSON.stringify(this.statusTasks),
        };
    
        /** 
         * URL encode the data.
         */
        const urlEncodedData = new URLSearchParams(Object.entries(data));
            
        setInterval(() => {
            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        }, 5000);
    }

    toolsColumn() {
        /**
         * Get all tools set of the current page
         */
        this.toolsSets = document.querySelectorAll('.jptgb-tool_column');
        
        if(!this.toolsSets.length){
            return;
        }
        
        this.toolsSets.forEach((toolsSet) => {
            /**
             * Register create cache button click.
             */
            const createCacheBtn    = toolsSet.querySelector('.jptgb-speed_up_btn');
            const rebuildCacheBtns  = toolsSet.querySelectorAll('.jptgb-ready_rebuild');
            const deleteCacheBtns   = toolsSet.querySelectorAll('.jptgb-ready_delete');
            const toolsSetId        = toolsSet.dataset.id;
            const type              = toolsSet.dataset.type;
            
            /**
             * Create the cache generate event listener.
            */
           createCacheBtn.addEventListener('click', () => {
                /**
                 * Display admin message.
                 */
                this.displayOption('.jptgb-tool_column_loading_icon', toolsSet);

                /**
                 * Add the task to the tests statuses.
                 */
                this.statusTasks[toolsSetId] = { 
                    message: 'Waiting For the Task to be Accepted in the Queue', 
                    percentage : '5', 
                    status : 'working-on', 
                    persist : 2,
                };

                /**
                 * Data to be sent to the server.
                 */
                const data = {
                    action:     'tools_column_generate_cache',
                    nonce:      jpgbData.toolsColumnGenerateCacheNonce,
                    object_id:    toolsSetId,
                    object_type:  type,
                    tasks: JSON.stringify(this.statusTasks),
                };

                /** 
                 * URL encode the data.
                 */
                const urlEncodedData = new URLSearchParams(Object.entries(data));

                /**
                 * Send the class names to the server.
                 */
                this.sendRequest(urlEncodedData, {'toolsSetId': toolsSetId});
            });

            /**
             * Register the rebuild buttons event listerners.
             */
            rebuildCacheBtns.forEach((btn) => {
                btn.addEventListener('click', () => {
                    /**
                     * Display admin message.
                     */
                    this.displayOption('.jptgb-tool_column_loading_icon', toolsSet);

                    /**
                     * Add the task to the tests statuses.
                     */
                    this.statusTasks[toolsSetId] = { 
                        message: 'Waiting For the Task to be Accepted in Queue', 
                        percentage : '5', 
                        status : 'working-on', 
                        persist : 2,
                    };

                    /**
                     * Data to be sent to the server.
                     */
                    const data = {
                        action:     'tools_column_generate_cache',
                        nonce:      jpgbData.toolsColumnGenerateCacheNonce,
                        object_id:    toolsSetId,
                        object_type:  type,
                        tasks: JSON.stringify(this.statusTasks),
                    };
    
                    /** 
                     * URL encode the data.
                     */
                    const urlEncodedData = new URLSearchParams(Object.entries(data));
    
                    /**
                     * Send the class names to the server.
                     */
                    this.sendRequest(urlEncodedData, {'toolsSetId': toolsSetId});
                });
            });

            /**
             * Register the dlete buttons event listerners.
             */
            deleteCacheBtns.forEach((btn) => {
                btn.addEventListener('click', () => {
                    /**
                     * Data to be sent to the server.
                     */
                        const data = {
                        action:     'tools_column_delete_cache',
                        nonce:      jpgbData.toolsColumnGenerateCacheNonce,
                        object_id:    toolsSetId,
                        object_type:  type,
                    };
    
                    /** 
                     * URL encode the data.
                     */
                    const urlEncodedData = new URLSearchParams(Object.entries(data));
    
                    /**
                     * Display admin message.
                     */
                    this.displayOption('.jptgb-tool_column_loading_icon', toolsSet);
    
                    /**
                     * Send the class names to the server.
                     */
                    this.sendRequest(urlEncodedData, {'toolsSetId': toolsSetId});
                });
            });

            /**
             * Check if we are working on the cache of some pages, if so, then
             * lets check their status recursevely.
             */
            if(toolsSet.querySelector('.jptgb-working').classList.contains('jptgb_active')){
                const data = {
                    percentage: toolsSet.querySelector('.jptgb-working_info_percentage').dataset.value,
                    message: toolsSet.querySelector('.jptgb-working_info_status .jptgb-tooltip').dataset.value,
                }
            }
        });
    }

    setNoCacheStatus(data, args){
        /**
         * Get the tool set.
         */
        const toolsSet = document.querySelector(`.jptgb-tool_column[data-id="${args.toolsSetId}"]`);

        /**
         * Stop the recursevely check if is running.
         */
        toolsSet.setAttribute('data-stopcheck', 1);

        this.displayOption('.jptgb-speed_up_btn', toolsSet);
    }

    setBlockedStatus(data, args){
        /**
         * Get the tool set.
         */
        const toolsSet = document.querySelector(`.jptgb-tool_column[data-id="${args.toolsSetId}"]`);

        /**
         * Update the message.
         */
        toolsSet.querySelector('.jptgb-blocked_info .jptgb-tooltip').innerHTML = data?.message ?? 'Block';

        /**
         * Stop the recursevely check if is running.
         */
        toolsSet.setAttribute('data-stopcheck', 1);

        /**
         * Display the blocked view.
         */
        this.displayOption('.jptgb-blocked', toolsSet);
    }

    displayOption(target, toolsSet) {
        toolsSet.querySelectorAll('.jptgb-tool_column_option').forEach((option) => {
            option.classList.remove('jptgb_active');
        });

        toolsSet.querySelector(target).classList.add('jptgb_active');
    }

    adminPage() {       
        /**
         * Get the wrap element.
         */
        this.app = document.querySelector('.jptgb');

        /**
         * Return if we are not in our admin page.
         */
        if(!this.app){
            return;
        }

        this.registerAppEvents();
        this.tabIndexs();
        this.requestNewApiBtn();
        this.testApiEndpointBtn();
        this.deleteUserBtn();
        this.flushAllCache();
        this.flushCriticalCss();
        this.flushAllImages();
        this.syncWithApi();
        this.deletePluginDdata();
        this.displayConditions();
        this.cleanUpMesageParams();
        this.generateAllPagesCache();
    }

    generateAllPagesCache() {
        const btn = document.querySelector('.jptgb-dashboard_main_progress_btn');

        if(!btn){
            return;
        }

        btn.addEventListener('click', () => {
            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'generate_all_pages_cache',
                execute: btn.dataset.execute,
                nonce: jpgbData.generateAllPagesCacheNonce,
            };


            btn.classList.add('jptgb-status__loading');

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    cleanUpMesageParams() {
        const currentUrl = new URL(window.location.href);

        // Check if the message and status parameters exist
        if (currentUrl.searchParams.has('admin_message') || currentUrl.searchParams.has('admin_status')) {
            // Remove the parameters
            currentUrl.searchParams.delete('admin_message');
            currentUrl.searchParams.delete('admin_status');
    
            // Update the URL without reloading the page
            window.history.replaceState({}, document.title, currentUrl.toString());
        }
    }

    registerAppEvents() {
        this.app.addEventListener('click', (event) => {
            this.checkDisplayConditions();
        });
    }

    displayConditions(){
        this.checkDisplayConditions();
    }

    checkDisplayConditions() {
        /**
         * Get all elements that has conditions.
         */
        const displayConditionsElements = document.querySelectorAll('.jptgb-control_group[data-displayconditions]');

        /**
         * Register the conditions for each element.
         */
        displayConditionsElements.forEach((element) => {
            /**
             * Get all conditions.
             */
            const displayConditions = JSON.parse(element.dataset.displayconditions);

            displayConditions.forEach((displayCondition) => {
                /**
                 * Get consition element.
                 */
                const conditionELememnt = this.app.querySelector(`#${displayCondition.id}`);

                /**
                 * Return if cant find the condition elememnt.
                 */
                if(!conditionELememnt){
                    return;
                }

                /**
                 * Get condition value.
                 */
                let conditionValue = conditionELememnt.type === 'checkbox' ? conditionELememnt.checked : conditionELememnt.value;

                /**
                 * Check if condition is met.
                 */
                if(conditionValue == displayCondition.value){
                    element.classList.add('jptgb__active');

                } else {
                    element.classList.remove('jptgb__active');

                }
            });
        });
    }

    syncWithApi() {
        const syncBtn = this.app.querySelector('#jptgb_setting_sync_with_api');

        if(!syncBtn){
            return;
        }

        syncBtn.addEventListener('click', () => {
            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'sync_with_api_ajax',
                nonce: jpgbData.syncApiNonce,
            };

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Display admin message.
             */
            // this.displayAdminNotice('Sync with the Api...');
            syncBtn.textContent = 'Sync with the Api...';

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    displayAdminNotice(message, type = 'info') {
        const noticeClass = `notice notice-${type}`;
        
        /**
         * Create the notice element.
         */
        var notice = document.createElement('div');
        notice.className = noticeClass;
        notice.innerHTML = '<p>' + message + '</p>';

        if( type.includes('is-dismissible') ) {
            notice.innerHTML += `<button type="button" class="notice-dismiss" onclick="this.parentNode.remove();">
                                    <span class="screen-reader-text">Dismiss this notice.</span>
                                </button>`;

        } else {
            /**
             * Optionally, make the notice dismissible.
             */
            setTimeout(() => {
                notice.remove();
            }, 5000);
        }
    
        /**
         * Find a place in the DOM to insert the notice.
         */
        var refElement = document.querySelector('h1');
        if (refElement) {
            refElement.parentNode.insertBefore(notice, refElement.nextSibling);
        }
    }   

    flushCriticalCss() {
        const btns = this.app.querySelectorAll('#jptgb_setting_flush_all_crtical_css, #jptgb_setting_force_flush_all_crtical_css');

        btns.forEach((btn) => {
            btn.addEventListener('click', () => {
                let data = {};

                if( 'jptgb_setting_flush_all_crtical_css' === btn.id ) {
                    data = {
                        action: 'flush_critical_css',
                        nonce: jpgbData.flushCcNonce,
                    };
                } else {
                    data = {
                        action: 'force_flush_critical_css',
                        nonce: jpgbData.forceFlushCcNonce,
                    };
                }

                /**
                 * Display admin message.
                 */
                // this.displayAdminNotice('Flushing critical css...');
                btn.textContent = 'Flushing Critical Css Now...';


                /**
                 * URL encode the data.
                 */
                const urlEncodedData = new URLSearchParams(Object.entries(data));

                /**
                 * Send the class names to the server.
                 */
                this.sendRequest(urlEncodedData);
            });
        });
    }

    deletePluginDdata() {
        const btn = this.app.querySelector('#jptgb_setting_delete_plugins_data');

        if(!btn){
            return;
        }

        btn.addEventListener('click', () => {
            /**
             * Display admin message.
             */
            btn.textContent = 'Deleting Plugin\'s Data Now...';

            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'delete_plugins_data',
                nonce: jpgbData.deletePluginsDataNonce,
            };


            /**
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    flushAllCache() {
        const btn = this.app.querySelector('#jptgb_setting_flush_all_cache');

        if( !btn ){
            return;
        }

        btn.addEventListener('click', () => {
            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'flush_all_cache',
                nonce: jpgbData.flushAllCacheNonce,
            };

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Display admin message.
             */
            btn.textContent = 'Flushing Cache Now...';

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    flushAllImages() {
        const btn = this.app.querySelector('#jptgb_setting_flush_all_images');

        if( !btn ){
            return;
        }

        btn.addEventListener('click', () => {
            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'flush_all_images',
                nonce: jpgbData.flushAllImagesNonce,
            };

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Display admin message.
             */
            // this.displayAdminNotice('Flushing all images...');
            btn.textContent = 'Deleting All Compressed Images...';

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    deleteUserBtn() {
        const btn = this.app.querySelector('#jptgb_setting_delete_user');

        if( !btn ){
            return;
        }

        btn.addEventListener('click', () => {
            /**
             * Get the email.
             */
            const   inputElement = this.app.querySelector('#jptgb_setting_user_email'),
                    value = inputElement.value;

            /**
             * Check if email is valid.
             */
            if(!this.isValidEmail(inputElement, value)){
                return;
            }

            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'delete_user',
                nonce: jpgbData.deleteUserNonce,
                email: value,
            };

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    requestNewApiBtn() {
        const btn = this.app.querySelector('#jptgb_setting_request_new_api');

        if( !btn ){
            return;
        }

        btn.addEventListener('click', () => {
            /**
             * Get the email.
             */
            const   inputElement = this.app.querySelector('#jptgb_setting_user_email'),
                    value = inputElement.value;

            /**
             * Check if email is valid.
             */
            if(!this.isValidEmail(inputElement, value)){
                return;
            }

            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'request_new_api',
                nonce: jpgbData.requestNewApiNonce,
                email: value,
            };

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Display admin message.
             */
            this.displayAdminNotice('Requesting new API key...');

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    testApiEndpointBtn() {
        const btn = this.app.querySelector('#jptgb_setting_test_api_endpoint');

        if( !btn ){
            return;
        }

        btn.addEventListener('click', () => {
            /** 
             * Data to be sent to the server.
             */
            const data = {
                action: 'test_api_endpoint',
                nonce: jpgbData.testApiEndpointNonce,
            };

            /** 
             * URL encode the data.
             */
            const urlEncodedData = new URLSearchParams(Object.entries(data));

            /**
             * Display admin message.
             */
            this.displayAdminNotice('Testing Api Endpoint...');

            /**
             * Send the class names to the server.
             */
            this.sendRequest(urlEncodedData);
        });
    }

    displayInformationMessage(message, element, type = 'jptgb__success') {
        const notElement = element.closest('.jptgb-input').querySelector('.jptgb-setting_information');

        /**
         * Add the meesage.
         */
        notElement.innerHTML = message;

        /**
         * Add the notification type.
         */
        notElement.classList.add(type);
    }

    handleValidationMessage(isValid, invalidMessage, element) {
        /**
         * Get the validation element.
         */
        const validationElement = element.closest('.jptgb-input').querySelector('.jptgb-setting_validation');

        /**
         * Return if we can't find the validation element.
         */
        if(!validationElement){
            return;
        }

        /**
         * Hide any previous validations message.
         */
        validationElement.classList.remove('jptgb__active');

        /**
         * If is valid then just return.
         */
        if(isValid){
            return true;
        }

        /**
         * Assign the meesage.
         */
        validationElement.innerHTML = invalidMessage;

        /**
         * Show the error.
         */
        validationElement.classList.add('jptgb__active');

        return false;
    }

    isValidEmail(element, value) {
        const input = document.createElement('input');
        input.type = 'email';
        input.value = value;

        /**
         * Determine if email is valid or not.
         */
        const isValid = value && input.checkValidity();

        /**
         * Hanlde the validation message.
         */
        this.handleValidationMessage(isValid, 'Invalid email', element);

        return isValid;
    }

    /**
     * Actions to perform after we 
     * have successfully deleted an user.
     */
    afterDeleteUser() {
        const   userEmail   = this.app.querySelector('#jptgb_setting_user_email'),
                apiKey      = this.app.querySelector('#jptgb_setting_api_key');

        userEmail.value = '';
        apiKey.value    = '';
    }

    afterNewApiRequested() {
        /**
         * Notify the user about the email sent.
         */
        this.displayAdminNotice('Warning: Old API key will be still in use until the new one is used!', 'warning is-dismissible');
    }

    refreshPageWithMessage(message, status = 'success') {
        // Encode the message and status for the URL
        const encodedMessage = encodeURIComponent(message);
        const encodedStatus = encodeURIComponent(status);
    
        // Add the message and status as query parameters to the current URL
        const currentUrl = new URL(window.location.href);
        currentUrl.searchParams.set('admin_message', encodedMessage);
        currentUrl.searchParams.set('admin_status', encodedStatus);
    
        // Reload the page with the new query parameters
        window.location.href = currentUrl.toString();
    }

    updateTasks(tasks){

        for (const key in tasks) {
            if (Object.hasOwn(tasks, key)) { 

                if( Object.hasOwn(this.statusTasks[key], 'persist') && this.statusTasks[key]['persist'] > 0 ){
                    this.statusTasks[key]['persist'] = this.statusTasks[key]['persist'] - 1;
                } else {
                    this.statusTasks[key] = tasks[key];
                }
            }
        }

        /**
         * Iterate over each property.
         */
        for (const key in this.statusTasks) {
            /** 
             * Ensures only own properties are included.
             */
            if (Object.hasOwn(this.statusTasks, key)) {
                const data = this.statusTasks[key];

                const toolsSet      = document.querySelector(`.jptgb-tool_column[data-id="${key}"]`);
                const percentage    = data['percentage'];
                const message       = data['message'];
                const status        = data['status'];

                if(status === 'error'){
                    toolsSet.querySelector('.jptgb-blocked_info .jptgb-tooltip').textContent = message;
                    this.displayOption('.jptgb-blocked', toolsSet);

                } else if( status === 'working-on' ) {
                    /**
                     * Update the working on status.
                     */
                    toolsSet.querySelector('.jptgb-working_info_status .jptgb-tooltip').textContent = message;
                    toolsSet.querySelector('.jptgb-working_info_percentage').textContent = `${percentage}%`;
                    toolsSet.querySelector('.jptgb-working_status_bar_fill').style.width = `${percentage}%`;

                    this.displayOption('.jptgb-working', toolsSet);

                } else if( status === 'no-cache' ) {
                    /**
                     * Show the Speed Up button
                     */
                    this.displayOption('.jptgb-speed_up_btn', toolsSet);

                } else if( status === 'cache-ready' ) {
                    /**
                     * Show the Cache Ready button.
                     */
                    this.displayOption('.jptgb-ready', toolsSet);

                }
            }
        }
    }

    updateAllProgressBar(data){
        const totalTasksElement         = this.app.querySelector('.jptgb-dashboard_main_progress_total');
        const progressCountElement      = this.app.querySelector('.jptgb-dashboard_main_progress_count');
        totalTasksElement.innerHTML     = data.totalTasks;
        progressCountElement.innerHTML  = data.readyTasks;

        /**
         * Update percentage.
         */
        const progressBarStatusElement      = this.app.querySelector('.jptgb-dashboard_main_progress_bar_status');
        progressBarStatusElement.innerHTML  = data.percentage + '%';
        progressBarStatusElement.setAttribute('style', `--progress: ${data.percentage}%`);

        if (typeof data.consumerStatus !== 'undefined' && data.consumerStatus === 'paused') {
            this.app.querySelector('.jptgb-dashboard_main_progress').classList.remove('in_progress');
            this.app.querySelector('.jptgb-dashboard_main_progress_btn').dataset.execute = 'activate_consumer';
        }
    }

    /**
     * Function to send class names to the server.
     */
    sendRequest(urlEncodedData, args = {}, singleRequest = true) {
        console.log('Ajax request', urlEncodedData);

        /**
        * Server endpoint URL. 
        */
        const url = new URL(jpgbData.ajaxUrl);

        /**
         * If there is no url the abort.
         */
        if(!url){
            console.error('We can\'t find the wordpress admin-ajax.php endpoint');
            return false;
        }
        
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: urlEncodedData,
        })
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP error! Status: ${response.status}`);
            }
            return response.json();
        })
        .then(response => {
            let message = '';
            let status  = '';

            if( response.hasOwnProperty('status') && response.hasOwnProperty('message') && response['message'] != '' ){
                message = response['message'];
                status  = response['status'];
            }

            const data = response['data'] ?? false;

            if(data){
                if( data.hasOwnProperty('callback') && data['callback'] !== null && data['callback'] !== '' ){    
                    switch (data['callback']) {
                        case 'afterDeleteUser': this.afterDeleteUser();
                            break;
                        case 'afterNewApiRequested': this.afterNewApiRequested();
                            break;
                        case 'checkStatusRecursevely': {
                                this.updateTasks(data.tasks);
                                this.updateAllProgressBar(data);
                            }
                            break;
                        case 'setNoCacheStatus': this.setNoCacheStatus(data, args);
                            break;
                        case 'setBlockedStatus': this.setBlockedStatus(data, args);
                            break;
                        case 'afterSyncWithApi': this.refreshPageWithMessage(message, status);;
                            break;
                        case 'refreshPageWithMessage': this.refreshPageWithMessage(message, status);
                            break;
                        case 'afterGenerateAllCacheBtnPressed': this.afterGenerateAllCacheBtnPressed(status, data);
                            break;
                    
                        default:
                            break;
                    }
                }
            }
        })
        .catch(error => {
            /**
             * Check if the fetch was aborted to avoid logging an error in that case
             */
            if (error.name === 'AbortError') {
                console.log('Fetch aborted');
            } else {
            }
        });
    }

    tabIndexs() {
        const tablinkElements = document.querySelectorAll('.jptgb-tablink');
        const tabcontentElements = document.querySelectorAll('.jptgb-tabcontent');

        tablinkElements.forEach((tablink) => {            
            tablink.addEventListener('click', () => {
                const tabContentIndex = tablink.dataset.tabcontent;

                tablinkElements.forEach((tabLink) => {
                    tabLink.classList.remove('jptgb__active');
                });

                tabcontentElements.forEach((tabContent) => {
                    tabContent.classList.remove('jptgb__active');
                });

                tablink.classList.add('jptgb__active');
                document.querySelector(`#jptgb-tabcontent_${tabContentIndex}`).classList.add('jptgb__active');
                document.querySelector(`#jptgb-current_tab`).value = tabContentIndex;
            });
        });
    }

    afterGenerateAllCacheBtnPressed(status, data) {
        const btn = document.querySelector('.jptgb-dashboard_main_progress_btn');
        btn.classList.remove('jptgb-status__loading');

        if(status !== 'success'){
            return;
        }

        if(data.action === 'show_in_progress'){
            document.querySelector('.jptgb-dashboard_main_progress').classList.add('in_progress');
            btn.dataset.execute = 'pause_consumer';

        } else if(data.action === 'hide_in_progress') {
            document.querySelector('.jptgb-dashboard_main_progress').classList.remove('in_progress');
            btn.dataset.execute = 'activate_consumer';

        }
    }
}

new JptgbAdmin();
