/**
 * Helper class to intercept key page lifecycle event callbacks.
 *
 * This class overrides both document.addEventListener and window.addEventListener for intercepted events
 * (e.g., DOMContentLoaded, readystatechange, load, pageshow), storing their callbacks
 * along with a record of whether they were registered on the document or window.
 * This allows later triggering of these events when desired.
 */
class jptbHelper {
    /**
     * Creates a new instance of jptbHelper.
     *
     * Saves the original addEventListener methods for document and window,
     * then overrides them to capture callbacks for specified events.
     */
    constructor() {
        /**
         * Define if we are in debug environment
         */
        this.isDebug = window.location.hash === `#jptgbdebug`;

        /**
         * An object mapping event names (e.g., "DOMContentLoaded") to an array of callback objects.
         * Each callback object has the form:
         *    { callback: Function, target: "document" | "window" }
         */
        this.customCallbacks = {};

        /**
         * The list of events for which we want to intercept callback registration.
         * Adjust this list as needed.
         */
        this.interceptedEvents = ['DOMContentLoaded', 'readystatechange', 'load', 'pageshow'];

        /**
         * Preserve the original document.addEventListener method.
         */
        this.originalAddEventListenerDocument = document.addEventListener;

        /**
         * Override document.addEventListener to capture callbacks for intercepted events.
         */
        document.addEventListener = (event, callback, ...options) => {
            if (this.interceptedEvents.includes(event)) {
                this.debugLog(`Storing document ${event} callback:`, callback);
                
                if (!this.customCallbacks[event]) {
                    this.customCallbacks[event] = [];
                }

                /**
                 * Store the callback along with a marker indicating registration on document.
                 */
                this.customCallbacks[event].push({ callback, target: "document" });
            } else {
                this.originalAddEventListenerDocument.call(document, event, callback, ...options);
            }
        };

        /**
         * Preserve the original window.addEventListener method.
         */
        this.originalAddEventListenerWindow = window.addEventListener;

        /**
         * Override window.addEventListener to capture callbacks for intercepted events.
         */
        window.addEventListener = (event, callback, ...options) => {
            if (this.interceptedEvents.includes(event)) {
                this.debugLog(`Storing window ${event} callback:`, callback);

                if (!this.customCallbacks[event]) {
                    this.customCallbacks[event] = [];
                }
                /**
                 * Store the callback along with a marker indicating registration on window.
                 */
                this.customCallbacks[event].push({ callback, target: "window" });
            } else {
                this.originalAddEventListenerWindow.call(window, event, callback, ...options);
            }
        };

        /**
         * Override document.readyState getter so that any code checking it sees "loading".
         */
        Object.defineProperty(document, "readyState", {
            get: function() { return "loading"; },
            configurable: true
        });
    }

    debugLog(message, type = 'log'){
        if(!this.isDebug){
            return;
        }

        switch (type) {
            case 'logo':
                console.log(message);
                break;
            case 'error':
                console.error(message);
                break;
        
            default:
                break;
        }
    }
}

/**
 * Create a global instance so it is accessible to later scripts.
 */
window.jptbHelper = new jptbHelper();
