class jptgbInCache {
    static isJptgbFuiEventDispatched            = false;
    static isJptgbDomReadyEventDispatched       = false;
    static isAlreadyWatchingForScriptsToLoad    = false;

    static debugLog(message, type = 'log'){
        if(!this.isDebug){
            return;
        }

        switch (type) {
            case 'log':
                console.log(message);
                break;
            case 'error':
                console.error(message);
                break;
            case 'warn':
                console.warn(message);
                break;
        
            default:
                break;
        }
    }

    static init() {
        /**
         * Define if we are in debug environment
         */
        this.isDebug = window.location.hash === `#jptgbdebug`;

        this.registerCustomEvents();
        this.dispatchCustomEvents();
        this.laterDomLoad();
        this.scriptsLoaderV3();
        this.imagesLoader();
    }

    static scriptsLoaderV3() {
        if( window.JptgbScriptLoaderV3 ){
            return;
        }

        class JptgbScriptLoaderV3 {
            constructor(scripts) {
                jptgbInCache.debugLog('Scripts Loader V3');
                jptgbInCache.debugLog(scripts);

                this.styleScripts       = [];
                this.jsScripts          = [];
                this.iframeScripts      = [];
                this.allScriptsLoaded   = false;
                this.lastId             = 0;
                this.pendingScripts     = new Set();


                /**
                 * Separete the style scripts from the 
                 * js scripts.
                 */
                scripts.forEach((script) => {
                    /**
                     * Continue to next loop if we can't find the 
                     * 'type' property.
                     */
                    if(!script.hasOwnProperty('type')){
                        console.error(`Jptgb: We can't find the type property in the script ${script}`);
                        return;
                    }

                    /**
                     * Add the script to styleScripts if it is 
                     * type of style.
                     */
                    if(script.type === 'style'){
                        this.styleScripts.push(script);
                    }

                    /**
                     * Add the script to styleScripts if it is 
                     * type of style.
                     */
                    if(script.type === 'iframe'){
                        this.iframeScripts.push(script);
                    }

                    /**
                     * Add the script to jsScripts if it is 
                     * type of script.
                     */
                    if(script.type === 'script'){
                        this.jsScripts.push(script);
                    }
                });

                /**
                 * Load the style scripts after 'First User Interaction' (FUI) event.
                 *
                 * We can load the styles as soon as FUI, sicne the styles do not depended
                 * on elements to be present on the DOM. The sooner we load the pending styles
                 * the better.
                 */
                if (jptgbInCache.isJptgbFuiEventDispatched) {
                    this.loadStyles();
                } else {
                    document.addEventListener('jptgbFuiEvent', () => this.loadStyles());
                }

                /**
                 * Load the iframes after 'First User Interaction' (FUI) event.
                 *
                 * We can load the iframes as soon as FUI, sicne the iframes do not depended
                 * on elements to be present on the DOM. The sooner we load the pending iframes
                 * the better.
                 */
                if (jptgbInCache.isJptgbFuiEventDispatched) {
                    this.loadIframes();
                } else {
                    document.addEventListener('jptgbFuiEvent', () => this.loadIframes());
                }

                /**
                 * Load the js scripts after 'Custom Document Ready' (CDR) event.
                 * 
                 * We are loading the scripts on CDR because we need to wait 
                 * until the document is ready to load the scripts,
                 * since the scripts might need some elements from the DOM.
                 */
                if (document.isJptgbCdrEventDispatched) {
                    this.loadScripts();
                } else {
                    document.addEventListener('jptgbCdrEvent', () => {
                        this.loadScripts();
                    });
                }
            }

            loadIframes() {
                jptgbInCache.debugLog('Ready to load the iframes');

                /**
                 * Process each script sequentially.
                 */
                for (const iframeObj of this.iframeScripts) {
                    // Extract the handle using bracket notation.
                    const handle = iframeObj.attributes["data-jptgbscriptid"];
                    delete iframeObj.attributes["data-jptgbscriptid"];

                    // Locate the blocked iframe script.
                    const blockedIframe = document.querySelector(`iframe[data-jptgbid="${handle}"]`);
                    if (!blockedIframe) {
                        jptgbInCache.debugLog(`Blocked iframe not found for handle: ${handle}`, 'warn');
                        return;
                    }

                    for (const key in iframeObj.attributes) {
                        blockedIframe.setAttribute(key, iframeObj.attributes[key]);
                    }

                    jptgbInCache.debugLog(`iframe loadeed successfully: ${handle}`);
                }
            }

            loadStyles() {
                /**
                 * Loop through each style.
                 */
                this.styleScripts.forEach((styleScript) => {
                    /**
                     * If style do not have attributes or do not 
                     * have the href attribute or the href attribute is empty, 
                     * then it is an inline style.
                     */
                    if(!styleScript.hasOwnProperty('attributes') || !styleScript.attributes.hasOwnProperty('href') || styleScript.attributes.href === ''){
                        this.loadInlineStyle(styleScript);
                        
                    /**
                     * Otherwise the style is external.
                     */
                    } else {
                        this.loadExternalStyle(styleScript);
                    }
                });

                /**
                 * Lets leave an space for the new iserted css to settle,
                 * before removing the .jptgb-initial_load and .jptgb_btf_content classes.
                 */
                setTimeout(() => {
                    /**
                     * Now that all the styles are loaded lets remove the jptgb-initial_load class 
                     * from the body. 
                     */
                    document.body.classList.remove('jptgb-initial_load');
    
                    /**
                     * Now that all the styles are loaded, lets remove the 'jptgb_btf_content' and the 'jptgb_atfc'
                     * from those elements which have it.
                     */
                    document.querySelectorAll('.jptgb_btf_content, .jptgb_atfc').forEach((btfElement) => {
                        btfElement.classList.remove('jptgb_btf_content');
                        btfElement.classList.remove('jptgb_atfc');
                    });
                }, 100);
            }

            loadInlineStyle(styleScript) {
                /**
                 * Create the style tag element.
                 */
                const style = document.createElement('style');

                /**
                 * Add the attributes if they exist.
                 */
                if(styleScript.hasOwnProperty('attributes')){
                    for (var key in styleScript.attributes) {
                        if (styleScript.attributes.hasOwnProperty(key)) {
                            style.setAttribute(key, styleScript.attributes[key]);
                        }
                    }
                }
                
                /**
                 * Add the css content if it exist.
                 */
                if(styleScript.hasOwnProperty('content')){
                    style.innerHTML = styleScript.content;
                }

                /**
                 * Insert the style in the DOM.
                 */
                if(styleScript.inFooter){
                    document.body.appendChild(style);
                } else {
                    document.head.appendChild(style);
                }
                
                /** 
                 * Log out the success.
                 */
                jptgbInCache.debugLog('Inline Style Loaded on FUI...');
            }

            loadExternalStyle(styleScript) {
                /**
                 * Get the style url.
                 */
                const href = styleScript.attributes.href;

                /**
                 *  Load on FUI event.
                 */
                var linkTag = document.createElement('link');

                /**
                 * Add all the attributes.
                 */
                for (var key in styleScript.attributes) {
                    if (styleScript.attributes.hasOwnProperty(key)) {
                        linkTag.setAttribute(key, styleScript.attributes[key]);
                    }
                }

                /**
                 * Insert the style in the DOM.
                 */
                if(styleScript.inFooter){
                    document.body.appendChild(linkTag);
                } else {
                    document.head.appendChild(linkTag);
                }

                /**
                 * Log out the success.
                 */
                jptgbInCache.debugLog(`Loading ${href} on FUI event...`);
            }

            /**
             * Loads all scripts in this.jsScripts sequentially.
             *
             * For each script object, if a 'src' attribute exists, then it's treated as an external script
             * and is loaded by inserting a new script element and awaiting its load event.
             * Otherwise, it is treated as an inline script, inserted synchronously.
             *
             * After all scripts are processed, a custom DOMContentLoaded event is manually triggered.
             *
             * @returns {Promise<void>}
             */
            async loadScripts() {
                jptgbInCache.debugLog('Cdr Event Triggered, Lets load the scripts!');
                jptgbInCache.debugLog(this.jsScripts);

                /**
                 * Process each script sequentially.
                 */
                for (const scriptObj of this.jsScripts) {
                    if (scriptObj.attributes.src) {
                        await this.loadExternalScript(scriptObj);
                    } else {
                        await this.loadInlineScript(scriptObj);
                    }
                }

                this.customCallbacks = window.jptbHelper.customCallbacks;
                this.interceptedEvents = window.jptbHelper.interceptedEvents;

                this.triggerCustomEvents();
            }

            triggerCustomEvent(eventName) {
                if (this.customCallbacks[eventName] && this.customCallbacks[eventName].length > 0) {
                    jptgbInCache.debugLog(`Calling custom ${eventName} callbacks`);

                    /**
                     * Create a new event object with the given name.
                     */ 
                    const eventObj = new Event(eventName);
                    this.customCallbacks[eventName].forEach(cbObj => {
                        try {
                            /**
                             * Call the callback using the stored target as context.
                             */
                            if (cbObj.target === 'window') {
                                cbObj.callback.call(window, eventObj);
                            } else {
                                cbObj.callback.call(document, eventObj);
                            }
                        } catch (error) {
                            console.error(`Error in ${eventName} callback:`, error);
                        }
                    });
                    /**
                     * Clear the callbacks so that they are not executed again.
                     */
                    this.customCallbacks[eventName] = [];
                } else {
                    jptgbInCache.debugLog(`No callbacks stored for ${eventName}`);
                }
            };

            triggerCustomEvents() {
                /**
                 * Phase 1: Set readyState to "interactive" and trigger early events.
                 */
                Object.defineProperty(document, "readyState", {
                    get: function() { return "interactive"; },
                    configurable: true
                });
                jptgbInCache.debugLog("Document readyState set to interactive");
                this.triggerCustomEvent('DOMContentLoaded');
                this.triggerCustomEvent('readystatechange');

                /**
                 * Phase 2: After a short delay, set readyState to "complete" and trigger later events.
                 */
                setTimeout(() => {
                    Object.defineProperty(document, "readyState", {
                        get: function() { return "complete"; },
                        configurable: true
                    });
                    jptgbInCache.debugLog("Document readyState set to complete");
                    this.triggerCustomEvent('load');
                    this.triggerCustomEvent('pageshow');

                    /**
                     * Dispatch a custom event indicating all custom events have been triggered.
                     */
                    const customEvent = new Event('jptgbCustomEventsDispatched');
                    document.dispatchEvent(customEvent);
                    document.jptgbCustomEventsDispatched = true;
                }, 10); // Adjust the delay as needed.
            }

            /**
             * Loads a blocked external script by reinserting it into the DOM so that it loads normally.
             *
             * This method locates the corresponding blocked script and template in the DOM, creates a new
             * executable script element using the provided attributes, and inserts it. It waits for the load
             * (or error) event on the new script before resolving.
             *
             * @param {Object} scriptObj - The script object with properties and attributes.
             * @returns {Promise<void>}
             */
            loadExternalScript(scriptObj) {
                jptgbInCache.debugLog('Handling external script');
                jptgbInCache.debugLog(scriptObj);

                return new Promise((resolve, reject) => {
                    /**
                     * Extract the unique script handle.
                     */
                    const jptgbScriptid = scriptObj.attributes["data-jptgbscriptid"];
                    /**
                     * Remove the handle property from the attributes.
                     */
                    delete scriptObj.attributes["data-jptgbscriptid"];

                    /**
                     * Use either the object's handle or the extracted script id.
                     */
                    const id = scriptObj.handle ?? jptgbScriptid;

                    /**
                     * Define script tag.
                     */
                    const tag = scriptObj.tag_name || 'script';

                    /**
                     * Locate the blocked external script by matching data attributes.
                     */
                    const blockedScript = document.querySelector(`${tag}[data-jptgbid="${jptgbScriptid}"][data-jptgbsource="external"]`);
                    if (!blockedScript) {
                        console.warn(`Blocked external script not found for jptgbScriptid: ${jptgbScriptid}`);
                        return resolve();
                    }

                    /**
                     * Locate the insertion template element.
                     */
                    const templateEl = document.querySelector(`template.jptgb-script[data-jptgbscript="${jptgbScriptid}"]`);
                    if (!templateEl) {
                        console.warn(`Template not found for external script with jptgbScriptid: ${jptgbScriptid}`);
                        return resolve();
                    }

                    /**
                     * Create a new executable script element.
                     */
                    const newScript = document.createElement(tag);

                    /**
                     * Copy each attribute from scriptObj.attributes (such as src, async, defer, etc.).
                     */
                    for (const key in scriptObj.attributes) {
                        newScript.setAttribute(key, scriptObj.attributes[key]);
                    }

                    /**
                     * If the new script doesn't have a src but the blocked script does, use the blocked script's src.
                     */
                    if (!newScript.src && blockedScript.src) {
                        newScript.src = blockedScript.src;
                    }

                    /**
                     * Attach event listeners to resolve the promise once the script is loaded or failed.
                     */
                    newScript.addEventListener('load', () => {
                        jptgbInCache.debugLog(`External script with id ${id} loaded.`);

                        resolve();
                    });
                    newScript.addEventListener('error', (err) => {
                        console.error(`Error loading external script with id ${id}:`, err);

                        /**
                         * Resolve (or reject) here, depending on whether you want to stop on error.
                         */
                        resolve();
                    });

                    /**
                     * Insert the new script into the DOM, triggering its load.
                     */
                    templateEl.parentNode.insertBefore(newScript, templateEl);

                    /**
                     * Remove the now-unneeded template and blocked script.
                     */
                    templateEl.parentNode.removeChild(templateEl);
                    blockedScript.parentNode.removeChild(blockedScript);
                });
            }

            /**
             * Loads a blocked inline script by reinserting its code into a new executable script element.
             *
             * The method locates the blocked inline script and corresponding template element in the DOM,
             * creates a new script element with the proper attributes and code, inserts it (triggering execution),
             * and then cleans up. Since inline scripts execute synchronously, this function returns an immediately
             * resolved promise.
             *
             * @param {Object} scriptObj - The script object containing the inline script properties.
             * @returns {Promise<void>}
             */
            loadInlineScript(scriptObj) {
                jptgbInCache.debugLog('Handling inline script');
                jptgbInCache.debugLog(scriptObj);

                // Extract the handle using bracket notation.
                const handle = scriptObj.attributes["data-jptgbscriptid"];
                delete scriptObj.attributes["data-jptgbscriptid"];

                // Get the script attributes.
                const attributes = scriptObj.attributes;

                let tag = 'script';
                if( attributes.rel && ( attributes.rel === 'preload' || attributes.rel === 'modulepreload' ) ) {
                    tag = 'link';
                }

                // If attributes.rel is undefined, the `if` is false and `tag` stays 'script'

                // Locate the blocked inline script.
                const blockedScript = document.querySelector(`${tag}[data-jptgbid="${handle}"]`);
                if (!blockedScript) {
                    console.warn(`Blocked inline script not found for handle: ${handle}`);
                    return Promise.resolve();
                }

                // Locate the template element for insertion.
                const templateEl = document.querySelector(`template.jptgb-script[data-jptgbscript="${handle}"]`);
                if (!templateEl) {
                    console.warn(`Template not found for inline script with handle: ${handle}`);
                    return Promise.resolve();
                }

                // Create a new executable script element.
                const newScript = document.createElement(tag);
                for (const key in scriptObj.attributes) {
                    newScript.setAttribute(key, scriptObj.attributes[key]);
                }

                // Copy the inline code into the new script element.
                newScript.text = blockedScript.textContent;

                // Insert the new script into the DOM to trigger its execution.
                templateEl.parentNode.insertBefore(newScript, templateEl);

                // Remove the template element and the original blocked script.
                templateEl.parentNode.removeChild(templateEl);
                blockedScript.parentNode.removeChild(blockedScript);

                return Promise.resolve();
            }
        }

        window.JptgbScriptLoaderV3 = JptgbScriptLoaderV3;
    }

    static imagesLoader() {
        if (document.isJptgbCdrEventDispatched) {
            this.imagesHandler();
        } else {
            document.addEventListener('jptgbCdrEvent', () => this.imagesHandler());
        }
    }

    static imagesHandler() {
        /**
         * Get all the images.
         */
        const elements = document.querySelectorAll('img[data-jptgb-original], [data-jptgb-bgimage]');

        elements.forEach(element => {
            /**
             * Check if it is image tag or element with background image.
             */
            if(element.hasAttribute('data-jptgb-original')){
                /**
                 * Retrieve the value of 'data-jptgb-srcset'
                 */
                const srcOriginalValue = element.getAttribute('data-jptgb-original');
    
                /**
                 * Set the 'src' attribute using the retrieved value.
                 */
                element.setAttribute('src', srcOriginalValue);
    
                /**
                 * Remove the data-jptgb-original attribute.
                 */
                element.removeAttribute('data-jptgb-original');
    
                /**
                 * Check if the image has the data-jptgb-srcset attribute.
                 */
                if(element.hasAttribute('data-jptgb-srcset')){
                    /**
                     * Retrieve the value of 'data-jptgb-srcset'
                     */
                    const srcsetValue = element.getAttribute('data-jptgb-srcset');
        
                    /**
                     * Set the 'srcset' attribute using the retrieved value.
                     */
                    element.setAttribute('srcset', srcsetValue);
        
                    /**
                     * Remove the original 'data-jptgb-srcset' attribute.
                     */
                    element.removeAttribute('data-jptgb-srcset');
                }

            } else if(element.hasAttribute('data-jptgb-bgimage')) { 

                /**
                 * Check if the element had an inlined bacground image.
                 */
                if(element.hasAttribute('data-original-inlinedbgimage')){
                    const srcsetValue = element.getAttribute('data-original-inlinedbgimage');
                    element.style.backgroundImage = srcsetValue;

                } else {
                    /**
                     * If the element is an iframe, then lets NOT remove the bg image.
                     */
                    if( element.tagName !== 'IFRAME' ){
                        element.style.removeProperty('background-image');
                    }
                }

                /**
                 * Check if the element has a before/after bg image selector and remove it.
                 */
                const classes = Array.from(element.classList);

                // Define the prefixes we want to remove
                const prefixes = [ 'jptgb_bg_before_', 'jptgb_bg_after_' ];

                // For each class, if it starts with one of our prefixes, remove it
                classes.forEach((cls) => {
                    if (prefixes.some((pre) => cls.startsWith(pre))) {
                        element.classList.remove(cls);
                    }
                });
            }
        });
    }
    
    static laterDomLoad(){
        jptgbInCache.debugLog('Later Dom Load');
        jptgbInCache.debugLog(document.jptgbData);

        if(document.jptgbData.isIdl){
            this.immediateDomLoad();

        } else if (document.jptgbData.isLdl){
            this.lazyDomLoad();

        }
    }

    /** 
     * Dispatches a custom event and removes event listeners.
     * This method is statically defined and should be called on user interaction.
     */
    static dispatchFuiEvent(eventName = '') {
        if (!this.isJptgbFuiEventDispatched) {
            jptgbInCache.debugLog(`Dispatching FUI. Fired event: ${eventName}`);
            
            const jptgbFuiEvent = new CustomEvent('jptgbFuiEvent');
            document.dispatchEvent(jptgbFuiEvent);

            this.isJptgbFuiEventDispatched = true;
            this.removeFuiEventsListeners();
        } else {
            jptgbInCache.debugLog('FUI event has already been dispatched.');
        }
    }

    /** 
     * Removes the event listeners from the document.
     * This is to ensure the custom event is dispatched only once.
     */
    static removeFuiEventsListeners() {
        document.removeEventListener('mousemove',   this.fuiEventMouseMoveHandler);
        document.removeEventListener('scroll',      this.fuiEventScrollHandler);
        document.removeEventListener('click',       this.fuiEventClickHandler);
        document.removeEventListener('touchstart',  this.fuiEventTouchstartHandler);
    }

    /** 
     * Registers event listeners for mousemove, scroll, and click events.
     * The listeners are set to trigger the dispatch of the custom event.
     */
    static registerFuiEvent() {
        this.fuiEventMouseMoveHandler   = () => this.dispatchFuiEvent('mousemove');
        this.fuiEventScrollHandler      = () => this.dispatchFuiEvent('scroll');
        this.fuiEventClickHandler       = () => this.dispatchFuiEvent('click');
        this.fuiEventTouchstartHandler  = () => this.dispatchFuiEvent('touchstart');
        
        jptgbInCache.debugLog('Adding the FUI event listeners');
        document.addEventListener('mousemove',  this.fuiEventMouseMoveHandler, { passive: true });
        document.addEventListener('scroll',     this.fuiEventScrollHandler, { passive: true });
        document.addEventListener('click',      this.fuiEventClickHandler, { passive: true });
        document.addEventListener('touchstart', this.fuiEventTouchstartHandler, { passive: true });
    }
    
    static registerCustomEvents() {
        this.registerFuiEvent();
        this.registerClickEvent();
    }

    static registerClickEvent() {
        let documentClickEventFired = false;

        const clickHandler = (event) => {
            jptgbInCache.debugLog('Click event registered: ', event.target);

            /**
             * Return if the event already fired.
             */
            if (documentClickEventFired || document.jptgbCustomEventsDispatched) {
                document.removeEventListener('click', clickHandler);
                return;
            }

            /**
             * Flag the current click event as triggered.
             */
            documentClickEventFired = true;

            /**
             * Get the loading animation visible.
             */
            const loadingAnimation = document.querySelector('.jptgb_loading_animation');

            /**
             * Show up the loading animation.
             */
            if (loadingAnimation) {
                /**
                 * Show the animation.
                 */
                loadingAnimation.style.display = 'block';

                /**
                 * If something goes wrong, then remove the animation yes or yes after 1.5 second.
                 */
                setTimeout(() => {
                    loadingAnimation.remove();
                }, 1500);
            }

            /**
             * Dispatch the click event on that element that was clicked before 
             * the scripts were ready.
             */
            const customEventHandler = () => {
                setTimeout(() => {
                    /**
                     * Get the clicked element.
                     */
                    let clickedElement = event.target;

                    /**
                     * Traverse up the DOM tree until an element with a click method is found.
                     */
                    while (clickedElement && typeof clickedElement.click !== 'function') {
                        clickedElement = clickedElement.parentElement;
                    }

                    /**
                     * If a valid element with a click method is found, click it.
                     */
                    if (clickedElement) {
                        jptgbInCache.debugLog(`About to click ${clickedElement}`);
                        clickedElement.click();

                    } else {
                        jptgbInCache.debugLog('No clickable element found.');

                    }

                    /**
                     * Remove the loading animation.
                     */
                    if (loadingAnimation) {
                        loadingAnimation.remove();
                    }

                    /**
                     * Remove the event listeners after they have triggered.
                     */
                    document.removeEventListener('jptgbCustomEventsDispatched', customEventHandler);
                    document.removeEventListener('click', clickHandler);
                }, 100);
            };

            document.addEventListener('jptgbCustomEventsDispatched', customEventHandler);
        };

        document.addEventListener('click', clickHandler);
    }

    static dispatchCdrEvent() {
        /**
         * Return if the event alredy dispatched.
         */
        if (document.isJptgbCdrEventDispatched) {
            return;
        }

        /**
         * Notify the console.
         */
        jptgbInCache.debugLog('Dispatching Jptgb CDR Event.');

        /**
         * Dispatch the FUI envent.
         */
        const jptgbCdrEvent = new CustomEvent('jptgbCdrEvent');
        document.dispatchEvent(jptgbCdrEvent);

        /**
         * Register the event dispathc on the document.
         */
        document.isJptgbCdrEventDispatched = true;
    }

    static dispatchCustomEvents() {
        /**
         * If is we are loading IDL or LDL content, then lets 
         * dispatch the CDR event after all the content is 
         * inserted in the DOM.
         */
        if (document.jptgbData.isIdl || document.jptgbData.isLdl) {
            jptgbInCache.debugLog('Is Idl or Ldl');
            if (this.isJptgbDomReadyEventDispatched) {
                jptgbInCache.debugLog('jptgbDomReadyEvent was Already Dispatched');
                this.dispatchCdrEvent();
            } else {
                document.addEventListener('jptgbDomReadyEvent', this.dispatchCdrEvent);
            }

        /**
         * By default dispatch the Custom Document Ready (CDR) event right after the FUI event.
         */
        } else {
            if (this.isJptgbFuiEventDispatched) {
                jptgbInCache.debugLog('Dispatching the CDR Event');
                this.dispatchCdrEvent();
            } else {
                document.addEventListener('jptgbFuiEvent', this.dispatchCdrEvent);
            }
        }
    }

    static immediateDomLoad() {
        if (this.isJptgbFuiEventDispatched) {
            this.loadIdlContent();
        } else {
            document.addEventListener('jptgbFuiEvent', this.loadIdlContent);
        }
    }

    static lazyDomLoad() {
        if (this.isJptgbFuiEventDispatched) {
            jptgbInCache.debugLog('Is FUI Event Dispatched from here');
            this.loadLdlContent();
        } else {
            document.addEventListener('jptgbFuiEvent', this.loadLdlContent);
        }
    }

    static insertLdlInDom( div ) {
        /**
         * Get the innerHTML, remove the comment markers.
         */
        let content = jptgbInCache.removeComments( div.innerHTML );

        /**
         * Mark the div as empty.
         */
        div.setAttribute('data-jptgbemptyplaceholder', '1');

        /**
         * Insert the content in the dom.
         */
        div.parentNode.innerHTML += content;
    }

    static loadLdlContent() {
        /**
        * Server endpoint URL. 
        */
        const url = new URL(document.jptgbData.ajaxUrl);

        /** 
         * Data to be sent to the server.
         */
        const data = {
            action          : 'get_ldl_data',
            requested_url   : document.jptgbData.requestedUrl,
            is_mobile       : document.jptgbData.isMobile
        };

        /** 
         * URL encoded the data.
         */
        const urlEncodedData = new URLSearchParams(Object.entries(data));

        /**
         * Show the loading icon on all place holders while requesting 
         * the lazy load content.
         */
        document.querySelectorAll('.jptgb_ldl_placeholder').forEach((placeholder) => {
            placeholder.innerHTML = `<img style="display: block; margin-left: auto; margin-right: auto; width: 60px; height: 60px; object-fit: contain;" src="${document.jptgbData.loadingIcon}">`;
        });

        /**
         * Call the server.
         */
        fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: urlEncodedData
        })
        .then(response  => response.json())
        .then(responseData => {
            jptgbInCache.debugLog(responseData);

            /**
             * Return if there is no response data.
             */
            if(!responseData.data){
                return;
            }

            /**
             * Iterate over each key and value from 
             * the LDL data.
             */
            for (const key in responseData.data) {
                if (!responseData.data.hasOwnProperty(key)) {
                    continue;
                }

                /**
                 * Get the placeholder element.
                 */
                const placeholder = document.getElementById(key);

                /**
                 * Continue to next iteration if we can't find the placeholder in the DOM.
                 */
                if(!placeholder){
                    jptgbInCache.debugLog('We can\'t find the placeholder: ', key);
                    continue;
                }

                /**
                 * Get the html content.
                 */
                const htmlContent = responseData.data[key];

                /**
                 * Replace the placeholder with the html content.
                 */
                placeholder.outerHTML = htmlContent;
            }

            /**
             * Dispatch the Dom Ready Event.
             */
            jptgbInCache.dispatchDomReadyEvent();

            /**
             * If a placeholder failed to load its content, then ensure 
             * it doesn't keeps showing the loading icon.
             */
            document.querySelectorAll('.jptgb_ldl_placeholder').forEach((placeholder) => {
                placeholder.remove();
            });
        })
        .catch(error => console.error('Error:', error));
    }

    static loadIdlContent() {
        jptgbInCache.debugLog('Loading IDL Content');

        /** 
         * Select all placeholders.
         */
        document.querySelectorAll('.jptgb_ldl_placeholder').forEach((placeholder) => {
            /**
             * Get the template element.
             */
            const templateElement = document.getElementById(placeholder.dataset.templateid);

            /**
             * Skip to the next iteration if we can't find the template element.
             */
            if(!templateElement){
                /**
                 * Remove the placeholder.
                 */
                placeholder.remove();

                /**
                 * Log the error.
                 */
                jptgbInCache.debugLog(`Jptgb: We can't find the template for the placeholder ${placeholder.dataset.templateid}`);

                /**
                 * Return to skip to the next iteration.
                 */
                return;
            }

            /**
             * Get the template content.
             */
            const templateContent = templateElement.content.cloneNode(true);

            /**
             * Insert the cloned content into the DOM, replacing the placeholder.
             */
            placeholder.replaceWith(templateContent);

            /**
             * Delete the template content to keep the DOM clean.
             */
            templateElement.remove();
        });

        /**
         * Dispatch the Dom Ready Event.
         */
        jptgbInCache.dispatchDomReadyEvent();
    }

    static dispatchDomReadyEvent() {
        /**
         * Return if the event already got triggered.
         */
        if (this.isJptgbDomReadyEventDispatched) {
            return;
        } 

        /**
         * Notify the console.
         */
        jptgbInCache.debugLog('Dispatching Jptgb Dom Ready Event.');

        /**
         * Dispatch the FUI envent.
         */
        const jptgbDomReadyEvent = new CustomEvent('jptgbDomReadyEvent');
        document.dispatchEvent(jptgbDomReadyEvent);

        /**
         * Register the event dispatch on the document.
         */
        this.isJptgbDomReadyEventDispatched = true;
    }

    static removeComments(content) {
        // Define markers
        const startMarker = 'JPTGBLDLREMOVESTART';
        const endMarker = 'JPTGBLDLREMOVEEND';
        const charsBeforeStartMarker = 4;
        const charsAfterEndMarker = 3;

        // Remove all occurrences of start and end markers
        content = this.removeAllOccurrences(content, startMarker, charsBeforeStartMarker, 0);
        content = this.removeAllOccurrences(content, endMarker, 0, charsAfterEndMarker);

        return content;
    }

    static removeAllOccurrences(content, marker, charsBefore, charsAfter) {
        let index = content.indexOf(marker);
        while (index !== -1) {
            // Calculate start index for removal, ensuring it doesn't go below 0
            let start = Math.max(index - charsBefore, 0);
            // Calculate end index for removal, ensuring it doesn't exceed content length
            let end = index + marker.length + charsAfter;
            // Remove the marker and its adjacent characters
            content = content.slice(0, start) + content.slice(end);
            // Find the next occurrence of the marker
            index = content.indexOf(marker, start);
        }
        return content;
    }
}

jptgbInCache.init();
