// self-hosted/api-controllers-interface.js

const path                          = require('path');
const https                         = require('https');
const axios                         = require('axios');
const fs                            = require('fs').promises;
const generateCachePage             = require('./includes/generateCachePage.js');
const generateCriticalCssContent    = require('./includes/generateCriticalCss.js');
const imageToWebp                   = require('./includes/imageToWebp.js');
const optimizeSvg                   = require('./includes/optimzeSvg.js');

async function executeInterface(){
    /**
     * Get the arguments from the command line.
     */
    const args = process.argv.slice(2); // Skip the first two elements

    /**
     * Check if any arguments were passed.
     */
    if (args.length <= 1 ) {
        console.error('Error: We expect two parameter, at least function name and an argument.');
        return;
    }

    /**
     * Get the function name.
     */
    const functionName = args[0];

    /**
     * Get the arguments for the function.
     */
    const functionArgs = args[1];

    /**
     * Parese the received arguments into a json object.
     */
    const pasrsedArgs = JSON.parse(functionArgs);

    switch (functionName) {
        case 'generateCache':{
            /**
             * Get the argument's data.
             */
            const data      = pasrsedArgs['data'] ?? {};
            const device    = pasrsedArgs['device'] ?? {};
    
            /**
             * Call the function.
             */
            const htmlContent = await generateCachePage(data, device);
    
            /**
             * Display the content
             */
            if(htmlContent){
                console.log(htmlContent);
            } else {
                console.error(`There was an error while genertating the Cache page`);
                return;
            }
        }
        case 'generateCriticalCss':{
            /**
             * Get the argument's data.
             */
            const requestedUrl  = pasrsedArgs['requestedUrl'] ?? '';
            const targetHtmlUrl = pasrsedArgs['targetHtmlUrl'] ?? '';
            const bundleCssUrl  = pasrsedArgs['bundleCssUrl'] ?? '';
            const chromiumPath  = pasrsedArgs['chromiumPath'] ?? '';
            const device        = pasrsedArgs['device'] ?? '';

            /**
             * If any of the parameters is missing then return an error.
             */
            if(!requestedUrl || !targetHtmlUrl || !bundleCssUrl || !device){
                console.error(`Some parameters are missing requestedUrl: ${requestedUrl} | targetHtmlUrl: ${targetHtmlUrl} | bundleCssUrl: ${bundleCssUrl} | device: ${device}`);
            }
    
            /**
             * Call the function.
             */
            const criticalCssContent = await generateCriticalCssContent(requestedUrl, targetHtmlUrl, bundleCssUrl, device, chromiumPath);
    
            /**
             * Display the content
             */
            if(criticalCssContent){
                console.log(criticalCssContent);
            } else {
                console.error(`There was an error while genertating critical css`);
                return;
            }
        }
        break;
        case 'optimizeImage':{
            /**
             * Try to write the image in the local server.
             */
            try {
                /**
                 * Get the argument's data.
                 */
                const format            = pasrsedArgs['format'] ?? '';
                const urlForApi         = pasrsedArgs['urlForApi'] ?? '';
                const destinationPath   = pasrsedArgs['destinationPath'] ?? '';
                const compressionRate   = pasrsedArgs['compressionRate'] ?? '';

                /**
                 * If any of the parameters is missing then return an error.
                 */
                if(!urlForApi || !destinationPath || !compressionRate || !format ){
                    console.error(`Some parameters are missing urlForApi: ${urlForApi} | destinationPath: ${destinationPath} | compressionRate: ${compressionRate} | format: ${format}`);
                }

                /**
                 * Disable SSL certificate validation for this specific request (Not recommended for production).
                 */
                const agent = new https.Agent({
                    rejectUnauthorized: false // Disables SSL validation only for this agent
                });

                /**
                 * Define axios options.
                 */
                const options = {
                    httpsAgent: agent,
                    maxRedirects: 0,
                    responseType: 'arraybuffer' 
                };

                /**
                 * Fetch the image content using axios.
                 */
                const response      = await axios.get(urlForApi, options);
                let imageBuffer     = Buffer.from(response.data, 'binary');

                /**
                 * Compress the image according to its format.
                 */
                if( format === 'svg' ){
                    imageBuffer = await optimizeSvg(imageBuffer);
                    
                } else {
                    imageBuffer = await imageToWebp(imageBuffer, compressionRate);
                }

                /**
                 * Log an error if there is has been an error compressing/converting the image.
                 */
                if(!imageBuffer){
                    console.error(`There is has been an issue by converting the image: ${urlForApi}`);
                    return;
                }

                /**
                 * Ensure the directory exists.
                 */
                await fs.mkdir(path.dirname(destinationPath), { recursive: true });
    
                /**
                 * Write the file.
                 */
                await fs.writeFile(destinationPath, imageBuffer);
                
            } catch (error) {
                console.error(`There has been an error while trying to save the image in the local server: ${error}`);
                return;
            }

            console.log('ok');
        }
        break;
    
        default: {
            console.error(`Error: We can't find the function ${functionName}.`);
        }
        break;
    }
}

executeInterface();
